/**
 * JASMINe Deploy ME [Managed Element]
 * Copyright (C) 2008 Bull S.A.S.
 * Copyright (C) 2008 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Configurator.java 2730 2008-11-07 14:32:05Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.deployme.configApply;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;

import org.apache.log4j.Logger;
import org.ow2.jasmine.deployme.generated.Configuration;
import org.ow2.jasmine.deployme.generated.Topology.Domains.Domain;
import org.ow2.jasmine.deployme.utils.ServerDomain;

/**
 * Configure a jonas server
 * 
 * @author Remy Bresson
 */
public class Configurator {

    /**
     * The logger
     */
    protected static Logger logger = Logger.getLogger(org.ow2.jasmine.deployme.configApply.Configurator.class);

    /**
     * Set the configuration of this cluster daemon, only used for tests
     * 
     * @param domainName Cluster daemon domain name
     * @param config The config of the cluster daemon
     * @param jonasRoot JONAS_ROOT directory to use
     */
    public static void setClusterAConfiguration(final String domainName, final Domain.ClusterDaemons.ClusterDaemon cluster,
        final String jonasRoot) {
        logger.info("Starting configuration set up for cluster " + cluster.getClusterDaemonName());

        cluster.setJonasRoot(jonasRoot);
        ConfigClusterApply apply = new ConfigClusterApply(jonasRoot);
        apply.setClusterConf(domainName, cluster);

        apply.execute();
        createScripts(cluster.getJavaHome(), null, cluster.getJonasRoot(), cluster.getJonasBasePrefix(), domainName, cluster
            .getClusterDaemonName(), true);
        logger.info("Cluster " + cluster.getClusterDaemonName() + " configured successfully");
    }

    /**
     * Set the configuration of this cluster daemon
     * 
     * @param domainName Cluster daemon domain name
     * @param config The config of the cluster daemon
     */
    public static void setClusterAConfiguration(final String domainName, final Domain.ClusterDaemons.ClusterDaemon cluster) {
        if (cluster.getJonasRoot() == null) {
            throw new IllegalArgumentException("The topology file doesn't define any JONAS_ROOT");
        }
        setClusterAConfiguration(domainName, cluster, cluster.getJonasRoot());
    }

    /**
     * Set the configuration of this server, only used for tests
     * 
     * @param config The config
     * @param sd The server and his domain
     * @param jonasRoot path to the jonas root to use
     */
    public static void setAConfiguration(final Configuration config, final ServerDomain sd, final String jonasRoot) {
        logger.info("Starting configuration set up for server " + sd);
        if (config.getGlobalJonas() != null) {
            config.getGlobalJonas().setJonasRoot(jonasRoot);
        }
        ConfigJonasApply apply = new ConfigJonasApply(jonasRoot);
        apply.setGlobalJonas(sd.getDomain().getName(), config.getGlobalJonas());
        apply.setProtocols(config.getProtocols());
        apply.setActiveServices(config.getActiveServices());
        apply.setSecurityManager(config.getSecurityManager());
        apply.setEjbClusteringConfiguration(config.getEjbClustering());
        apply.setJmsConfiguration(config.getJms());
        apply.setJdbcConfiguration(config.getJdbc());

        apply.execute();
        createScripts(config.getGlobalJonas().getJdk(), config.getGlobalJonas().getJavaOpts(), config.getGlobalJonas()
            .getJonasRoot(), config.getGlobalJonas().getJonasBase(), sd.getDomain().getName(), config.getGlobalJonas()
            .getJonasName(), false);
        logger.info("Server " + sd + " configured successfully");
    }

    /**
     * Set the configuration of this server
     * 
     * @param config The config
     * @param sd The server and his domain
     */
    public static void setAConfiguration(final Configuration config, final ServerDomain sd) {
        if (config.getGlobalJonas() == null || config.getGlobalJonas().getJonasRoot() == null) {
            throw new IllegalArgumentException("The topology file doesn't define any JONAS_ROOT");
        }
        setAConfiguration(config, sd, config.getGlobalJonas().getJonasRoot());
    }

    /**
     * Create the setenv, start and stop scripts
     * 
     * @param javaHome JAVA_HOME
     * @param javaOpts JAVA_OPTS
     * @param jonasRoot JONAS_ROOT
     * @param jonasBase JONAS_BASE
     * @param jonasDomain JOnAS server domain
     * @param jonasName JOnAS server name
     * @param clusterDaemon true if instance is a cluster daemon, false
     *        otherwise
     */
    private static void createScripts(final String javaHome, String javaOpts, final String jonasRoot, final String jonasBase,
        String jonasDomain, String jonasName, final boolean clusterDaemon) {
        if (javaOpts == null) {
            javaOpts = "";
        }
        if (jonasName == null) {
            jonasName = "jonas";
        }
        if (jonasDomain == null) {
            jonasDomain = "jonas";
        }
        String jonasExec;
        if (clusterDaemon) {
            jonasExec = "jclusterd";
        } else {
            jonasExec = "jonas";
        }

        try {
            File f;
            FileWriter fw;

            f = new File(jonasBase, "setenv.bat");
            fw = new FileWriter(f, false);
            if (javaHome != null) {
                fw.write("SET JAVA_HOME=" + javaHome + "\r\n");
            }
            fw.write("SET JAVA_OPTS=" + javaOpts + "\r\n\r\n");
            fw.write("SET JONAS_ROOT=" + jonasRoot + "\r\n");
            fw.write("SET JONAS_BASE=" + jonasBase + "\r\n");
            fw.write("SET PATH=" + jonasRoot + "\\bin;" + jonasRoot + "\\bin\\nt;%PATH%\r\n");
            fw.close();

            f = new File(jonasBase, "setenv");
            fw = new FileWriter(f, false);
            fw.write("#!/bin/sh\n");
            if (javaHome != null) {
                fw.write("JAVA_HOME=" + javaHome + "\n");
                fw.write("export JAVA_HOME\n");
            }
            fw.write("JAVA_OPTS=" + javaOpts + "\n");
            fw.write("export JAVA_OPTS\n\n");
            fw.write("JONAS_ROOT=" + jonasRoot + "\n");
            fw.write("export JONAS_ROOT\n");
            fw.write("JONAS_BASE=" + jonasBase + "\n");
            fw.write("export JONAS_BASE\n");
            fw.write("PATH=" + jonasRoot + "/bin:" + jonasRoot + "/bin/unix:$PATH\n");
            fw.write("export PATH\n");
            fw.close();
            try {
                Runtime.getRuntime().exec("chmod a+x " + f.getAbsolutePath());
            } catch (IOException ignored) {
                // This is Windows
            }

            f = new File(jonasBase, "start.bat");
            fw = new FileWriter(f, false);
            fw.write("CALL \"%~dp0\\setenv.bat\"\r\n");
            fw.write("CALL " + jonasExec + " start -n " + jonasName + " -Ddomain.name=" + jonasDomain + "\r\n");
            fw.close();

            f = new File(jonasBase, "start");
            fw = new FileWriter(f, false);
            fw.write("#!/bin/sh\n");
            fw.write("savewd=`pwd`\n");
            fw.write("cd `dirname $0`\n");
            fw.write("zdir=`pwd`\n");
            fw.write("cd $savewd\n");
            fw.write(". ./setenv\n");
            fw.write(jonasExec + " start -n " + jonasName + " -Ddomain.name=" + jonasDomain + "\n");
            fw.close();
            try {
                Runtime.getRuntime().exec("chmod a+x " + f.getAbsolutePath());
            } catch (IOException ignored) {
                // This is Windows
            }

            f = new File(jonasBase, "stop.bat");
            fw = new FileWriter(f, false);
            fw.write("CALL \"%~dp0\\setenv.bat\"\r\n");
            fw.write("CALL " + jonasExec + " stop -n " + jonasName + " -Ddomain.name=" + jonasDomain + "\r\n");
            fw.close();

            f = new File(jonasBase, "stop");
            fw = new FileWriter(f, false);
            fw.write("#!/bin/sh\n");
            fw.write("savewd=`pwd`\n");
            fw.write("cd `dirname $0`\n");
            fw.write("zdir=`pwd`\n");
            fw.write("cd $savewd\n");
            fw.write(". ./setenv\n");
            fw.write(jonasExec + " stop -n " + jonasName + " -Ddomain.name=" + jonasDomain + "\n");
            fw.close();
            try {
                Runtime.getRuntime().exec("chmod a+x " + f.getAbsolutePath());
            } catch (IOException ignored) {
                // This is Windows
            }

            if (!clusterDaemon) {
                f = new File(jonasBase, "halt.bat");
                fw = new FileWriter(f, false);
                fw.write("CALL \"%~dp0\\setenv.bat\"\r\n");
                fw.write("CALL jonas halt -n " + jonasName + " -Ddomain.name=" + jonasDomain + "\r\n");
                fw.close();

                f = new File(jonasBase, "halt");
                fw = new FileWriter(f, false);
                fw.write("#!/bin/sh\n");
                fw.write("savewd=`pwd`\n");
                fw.write("cd `dirname $0`\n");
                fw.write("zdir=`pwd`\n");
                fw.write("cd $savewd\n");
                fw.write(". ./setenv\n");
                fw.write("jonas halt -n " + jonasName + " -Ddomain.name=" + jonasDomain + "\n");
                fw.close();
                try {
                    Runtime.getRuntime().exec("chmod a+x " + f.getAbsolutePath());
                } catch (IOException ignored) {
                    // This is Windows
                }
            }
        } catch (Exception e) {
            throw new IllegalStateException("Cannot write the scripts", e);
        }
    }
}
