/**
 * JASMINe Deploy ME [Managed Element]
 * Copyright (C) 2008 Bull S.A.S.
 * Copyright (C) 2008 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CommandLineParser.java 2789 2008-11-25 14:35:20Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.deployme;

import java.io.File;
import java.net.URL;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.ow2.jasmine.deployme.utils.CommandLine;

/**
 * Parse the command line and call DeployMe main class
 * 
 * @author Remy Bresson
 */
public class CommandLineParser {

    /**
     * Main
     * 
     * @param args command line
     */
    public static void main(final String[] args) {
        try {
            if (args == null || args.length < 1) {
                throw new IllegalArgumentException("You have launched JASMINe Deploy ME without any arguments.\n\n"
                    + "To use JASMINe Deploy ME, you must first create a topology file that describes\n"
                    + "your server infrastructure. Please refer to the JASMINe Deploy ME XSD file for\n"
                    + "details or use JASMINe Mapping utility to create a topology file based on a\n"
                    + "JASMINe Design file.\n\n" + "Once you have specified the topologyFile option, you can deploy:\n\n"
                    + "    - All servers on a given machine. For that, use the machineName option.\n"
                    + "    - A cluster daemon. For that, use the domainName and clusterName options.\n"
                    + "    - A server. For that, use the domainName and serverName options.\n\n"
                    + "You can also set the -verbose option if you think JASMINe Deploy ME is too silent.");
            }

            CommandLine command = parseCommandLine(args);

            if (command.getVerbose()) {
                Logger.getRootLogger().setLevel(Level.ALL);
            }

            URL urlXML;
            File topologyFile = new File(command.getTopologyFile());

            if (topologyFile.isFile()) {
                urlXML = topologyFile.toURI().toURL();
            } else {
                urlXML = CommandLineParser.class.getClassLoader().getResource(command.getTopologyFile());
            }

            if (urlXML == null) {
                throw new IllegalArgumentException("Cannot find the specified topology file: '" + command.getTopologyFile()
                    + "'");
            }

            new DeployMe(urlXML, command.getMachineName(), command.getDomainName(), command.getServerName(), command
                .getClusterName());
        } catch (Exception e) {
            System.out.println(e.getMessage());
            if (e.getMessage() == null) {
                e.printStackTrace();
            }
            System.exit(-1);
        }
    }

    /**
     * Parse the command line
     * 
     * @param args command line
     * @return A object who represent parameters to use with deployMe
     */
    protected static CommandLine parseCommandLine(final String[] args) {
        CommandLine ret = new CommandLine();

        for (String tmp : args) {
            String argEntry = checkArgEntry(tmp);

            if (CommandLine.commandLineVerbose.equals(argEntry)) {
                ret.addValue(argEntry, null);
            } else {
                String[] argsSeparated = argEntry.split("=");
                if (argsSeparated.length != 2) {
                    throw new IllegalArgumentException("The argument '" + argEntry
                        + "' doesn't have a valid syntax, an argument must contain one and only one '='");
                }

                ret.addValue(argsSeparated[0], argsSeparated[1]);
            }
        }

        ret.checkCommandLine();

        return ret;
    }

    /**
     * Check if the arg begin with '-' and remove '-'
     * 
     * @param argEntry the argument to check
     * @return argEntry without '-'
     * @throws IllegalArgumentException when argEntry doesn't begin with '-'
     */
    protected static String checkArgEntry(final String argEntry) {
        if (argEntry.charAt(0) != '-') {
            throw new IllegalArgumentException("The argument '" + argEntry
                + "' doesn't have a valid syntax, an argument must start with '-'");
        }

        return argEntry.substring(1);
    }

}
