/**
 * JASMINe Deploy ME [Managed Element]
 * Copyright (C) 2008 Bull S.A.S.
 * Copyright (C) 2008 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CommandLineParser.java 4120 2009-06-23 09:55:57Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.deployme;

import java.io.File;
import java.net.URL;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.ow2.jasmine.deployme.utils.CommandLine;

/**
 * Parse the command line and call DeployMe main class
 * 
 * @author Remy Bresson
 */
public class CommandLineParser {

    /**
     * Main
     * 
     * @param args command line
     */
    public static void main(final String[] args) {
        try {
            if (args == null || args.length < 1) {
                throw new IllegalArgumentException(help());
            }

            CommandLine command = parseCommandLine(args);

            if (command.getVerbose()) {
                Logger.getRootLogger().setLevel(Level.ALL);
            }

            URL urlXML;
            File topologyFile = new File(command.getTopologyFile());

            if (topologyFile.isFile()) {
                urlXML = topologyFile.toURI().toURL();
            } else {
                urlXML = CommandLineParser.class.getClassLoader().getResource(command.getTopologyFile());
            }

            if (urlXML == null) {
                throw new IllegalArgumentException("Cannot find the specified topology file: '" + command.getTopologyFile()
                    + "'");
            }

            new DeployMe(urlXML, command.getMachineName(), command.getDomainName(), command.getServerName(), command
                .getClusterName());
        } catch (Exception e) {
            System.out.println(e.getMessage());
            if (e.getMessage() == null) {
                e.printStackTrace();
            }
            System.exit(-1);
        }
    }

    /**
     * Parse the command line
     * 
     * @param args command line
     * @return A object who represent parameters to use with deployMe
     */
    protected static CommandLine parseCommandLine(final String[] args) {
        CommandLine ret = new CommandLine();

        for (String tmp : args) {
            String argEntry = checkArgEntry(tmp);

            if (CommandLine.commandLineVerbose.equals(argEntry)) {
                ret.addValue(argEntry, null);
            } else {
                String[] argsSeparated = argEntry.split("=");
                if (argsSeparated.length != 2) {
                    throw new IllegalArgumentException("The argument '" + argEntry
                        + "' doesn't have a valid syntax, an argument must contain one and only one '='");
                }

                ret.addValue(argsSeparated[0], argsSeparated[1]);
            }
        }

        ret.checkCommandLine();

        return ret;
    }

    /**
     * Check if the arg begin with '-' and remove '-'
     * 
     * @param argEntry the argument to check
     * @return argEntry without '-'
     * @throws IllegalArgumentException when argEntry doesn't begin with '-'
     */
    protected static String checkArgEntry(final String argEntry) {
        if (argEntry.charAt(0) != '-') {
            throw new IllegalArgumentException("The argument '" + argEntry
                + "' doesn't have a valid syntax, an argument must start with '-'");
        }

        return argEntry.substring(1);
    }

    private static String help() {

        StringBuffer buffer = new StringBuffer();

        buffer.append("\n");
        buffer
            .append("Usage : java -jar deployme.jar -topologyFile=topology_xml (-machineName=host | -domainName=domain [-serverName=server | -clusterName=cluster]) [-verbose]");
        buffer.append("\n\n");
        buffer.append(" -topologyFile=topology_xml : ");
        buffer.append("\n\t");
        buffer.append("This file describes your server infrastructure. You have two ways to create it : ");
        buffer.append("\n\t");
        buffer.append("1. Export a topology file from JASMINe Design. This is the recommended method.");
        buffer.append("\n\t");
        buffer
            .append("2. Manually create your topology file. Please refer to the JASMINe Deploy ME XSD file and/or the JASMINe Deploy ME samples distribution for details.");
        buffer.append("\n\n");
        buffer.append("-machineName=host : name of a machine in the topology.xml file");
        buffer.append("\n\t");
        buffer.append("Deploy all servers on a given machine");
        buffer.append("\n\n");
        buffer.append("-domainName=domain : A domain name of the topology.xml file.");
        buffer.append("\n");
        buffer.append("-serverName=server : A server name of the topology.xml file.");
        buffer.append("\n\t");
        buffer.append("Deploy the server of the the specified domain.");
        buffer.append("\n");
        buffer.append("-clusterName=cluster : A cluster daemon name of the topology.xml file.");
        buffer.append("\n\t");
        buffer.append("Deploy the cluster daemon of the specified domain.");

        return buffer.toString();
    }

}
