/**
 * JASMINe Deploy ME [Managed Element]
 * Copyright (C) 2008 Bull S.A.S.
 * Copyright (C) 2008 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: XmlLoader.java 5189 2009-09-08 16:13:22Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.deployme.loadXml;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.net.URLConnection;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;

import org.ow2.jasmine.deployme.generated.Topology;
import org.xml.sax.SAXException;

/**
 * Load the xml topology
 * 
 * @author Remy Bresson
 */
public class XmlLoader {

    /**
     * The loaded topology
     */
    Topology topology;

    /**
     * Constructor. Loas the topology
     * 
     * @param topologyUrl path to the xml topology file
     * @param schemaUrl path to the xml topology schema
     * @param xmlns XML namespace, will be added if non found in the XML
     * @param schemaLocation XML schema location, will be added if non found in
     *        the XML
     * @throws SAXException
     * @throws JAXBException
     * @throws IOException
     */
    public XmlLoader(final URL topologyUrl, final URL schemaUrl, final String xmlns, final String schemaLocation)
        throws SAXException, JAXBException, IOException {
        this.loadTopology(topologyUrl, schemaUrl, xmlns, schemaLocation);
    }

    /**
     * Load the topology
     * 
     * @param urlXML topologyUrl path to the xml topology file
     * @param schemaUrl schemaUrl path to the xml topology schema
     * @param xmlns XML namespace, will be added if non found in the XML
     * @param schemaLocation XML schema location, will be added if non found in
     *        the XML
     * @throws SAXException
     * @throws JAXBException
     * @throws IOException
     */
    private void loadTopology(final URL urlXML, final URL schemaUrl, final String xmlns, final String schemaLocation)
        throws SAXException, JAXBException, IOException {
        JAXBContext jc;

        jc = JAXBContext.newInstance(Topology.class.getPackage().getName());

        Unmarshaller unMarshaller;

        unMarshaller = jc.createUnmarshaller();

        SchemaFactory schemaFactory = SchemaFactory.newInstance("http://www.w3.org/2001/XMLSchema");

        Schema schema = null;
        schema = schemaFactory.newSchema(schemaUrl);

        unMarshaller.setSchema(schema);

        String xml = readURL(urlXML);
        final int topologyStart = xml.indexOf("<topology");
        if (topologyStart != -1) {
            final int topologyEnd = xml.indexOf(">", topologyStart);

            if (topologyEnd != -1) {
                final int xmlnsIndex = xml.indexOf("xmlns", topologyStart);

                if (xmlnsIndex == -1) {
                    xml = xml.substring(0, topologyStart) + "<topology xmlns=\"" + xmlns
                        + "\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"" + xmlns + " "
                        + schemaLocation + "\"" + xml.substring(topologyStart + 9);
                }
            }
        }
        InputStream xmlInputStream = new ByteArrayInputStream(xml.getBytes());

        this.topology = (Topology) unMarshaller.unmarshal(xmlInputStream);
    }

    private static String readURL(final URL url) throws IOException {
        URLConnection connection = url.openConnection();
        BufferedReader in = new BufferedReader(new InputStreamReader(connection.getInputStream()));

        try {
            StringBuilder sb = new StringBuilder();
            String inputLine;
            while ((inputLine = in.readLine()) != null) {
                if (sb.length() > 0) {
                    sb.append('\n');
                }
                sb.append(inputLine);
            }

            return sb.toString();
        } finally {
            in.close();
        }
    }

    /**
     * Return the loaded topology
     * 
     * @return the topology
     */
    public Topology getTopology() {
        return this.topology;
    }

}
