/**
 * JASMINe Deploy ME [Managed Element]
 * Copyright (C) 2008 Bull S.A.S.
 * Copyright (C) 2008 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DeployMe.java 5206 2009-09-09 13:50:19Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.deployme;

import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import javax.xml.bind.JAXBException;

import org.ow2.jasmine.deployme.configApply.Configurator;
import org.ow2.jasmine.deployme.generated.Configuration;
import org.ow2.jasmine.deployme.generated.Topology;
import org.ow2.jasmine.deployme.generated.Topology.Domains.Domain;
import org.ow2.jasmine.deployme.generated.Topology.Domains.Domain.Servers.Server;
import org.ow2.jasmine.deployme.loadXml.XmlLoader;
import org.ow2.jasmine.deployme.utils.DeployMEPropertiesManager;
import org.ow2.jasmine.deployme.utils.ServerDomain;
import org.xml.sax.SAXException;

/**
 * The main class of deployMe project.
 * 
 * @author Remy Bresson
 */
public class DeployMe {

    /**
     * The loaded from xml topology
     */
    protected Topology topology;

    /**
     * List of the servers witch deployMe will work
     */
    protected List<ServerDomain> serversDomains;

    /**
     * Default constructor, do nothing <br />
     * Only for test
     */
    protected DeployMe() {

    }

    /**
     * JASMINe-CJC constructor : <br />
     * - Load the topology <br />
     * - Apply the config of the servers(machine or domain/server/cluster)
     * 
     * @param topologyUrl topology xml file to use url
     * @param machine The machine to use. if set domain or server or cluster
     *        name must not be set
     * @param domain The domain where found the server to use. If set, server or
     *        cluster has to be set, and machine must no be set
     * @param server The server to use. If set, domain has to be set and cluster
     *        or machine must not be set
     * @param cluster The cluster daemon to use. If set, domain has to be set
     *        and server or machine must not be set
     * @throws JAXBException JAXB exception.
     * @throws SAXException SAX exception.
     * @throws IOException If topologyUrl cannot be retrieved.
     */
    public DeployMe(final URL topologyUrl, final String machine, final String domain, final String server, final String cluster)
        throws SAXException, JAXBException, IOException {
        this.loadTopology(topologyUrl);

        boolean clusterMode = cluster != null;
        Domain.ClusterDaemons.ClusterDaemon clusterDaemon = null;

        if (clusterMode) {
            clusterDaemon = this.getClusterDaemonConf(domain, cluster);
            if (clusterDaemon == null) {
                throw new IllegalArgumentException("Cannot find the specified cluster daemon '" + cluster
                    + "' on specified domain '" + domain + "'");
            }
        }

        this.serversDomains = this.buildServerDomainList(machine, domain, server, cluster);

        if (clusterMode) {
            /* cluster mode */
            /* Merge configuration with domain */
            clusterDaemon = this.mergeClusterDaemonAndDomainConf(clusterDaemon, this.serversDomains.get(0).getDomain());

            /* Configure cluster */
            Configurator.setClusterAConfiguration(this.serversDomains.get(0).getDomain().getName(), clusterDaemon);
        }

        int i = 1;
        for (ServerDomain sd : this.serversDomains) {
            Configuration confToApply = new ConfiguratorBuilder(sd).getConfig();

            if (clusterMode) {
                /* Create the requested JONAS_BASE */
                confToApply.getGlobalOptions().setJonasBase(clusterDaemon.getJonasBasePrefix() + i++);
            }

            Configurator.setAConfiguration(confToApply, sd);
        }
    }

    /**
     * Load the topology from the url urlTopology using the schema from
     * schemaUrl. Store the topology on DeployMe.topology variable.
     * 
     * @param urlTopology Url of the topology on xml format
     * @throws JAXBException JAXB exception.
     * @throws SAXException SAX exception.
     * @throws IOException If urlTopology cannot be retrieved.
     */
    protected void loadTopology(final URL urlTopology) throws SAXException, JAXBException, IOException {
        XmlLoader xmlLoader = new XmlLoader(urlTopology, DeployMEPropertiesManager.getXsdTopologyURL(),
            DeployMEPropertiesManager.getTopologyXMLNS(), DeployMEPropertiesManager.getTopologySchemaLocation());
        this.topology = xmlLoader.getTopology();
    }

    protected Domain.ClusterDaemons.ClusterDaemon getClusterDaemonConf(final String domain, final String clusterName) {
        Domain.ClusterDaemons.ClusterDaemon workingCluster = null;

        for (Domain d : this.topology.getDomains().getDomain()) {
            if (d.getName().equals(domain)) {
                for (Domain.ClusterDaemons.ClusterDaemon cd : d.getClusterDaemons().getClusterDaemon()) {
                    if (cd.getName().equals(clusterName)) {
                        workingCluster = cd;
                    }
                }
            }
        }

        return workingCluster;
    }

    /**
     * Build the server list that will be used.
     * 
     * @param machine host name
     * @param domain domain name
     * @param server server name
     * @param clusterName cluster daemon name
     * @return A list with the server to configure
     */
    protected List<ServerDomain> buildServerDomainList(final String machine, final String domain, final String server,
        final String clusterName) {
        List<ServerDomain> ret;

        if (machine != null) {
            ret = this.buildServerListWithHostname(machine);
            if (ret == null || ret.size() < 1) {
                throw new IllegalArgumentException("The machine named '" + machine + "' does not appear in the topology file!");
            }
        } else if (clusterName == null) {
            ret = this.buildServerListWithServer(domain, server);
            if (ret == null || ret.size() < 1) {
                throw new IllegalArgumentException("The server named '" + domain + "." + server
                    + "' does not appear in the topology file!");
            }
        } else {
            ret = this.buildServerListWithCluster(domain, clusterName);
            if (ret == null || ret.size() < 1) {
                throw new IllegalArgumentException("The cluster daemon named '" + domain + "." + clusterName
                    + "' does not appear in the topology file!");
            }
        }

        return ret;
    }

    /**
     * Build the servers list (in this case, only 1 server in the list) defined
     * by domain.server
     * 
     * @param domain the domain
     * @param server the server
     * @return A list with the server to configure
     */
    private List<ServerDomain> buildServerListWithCluster(final String domain, final String cluster) {
        List<ServerDomain> ret = new ArrayList<ServerDomain>();

        Domain.ClusterDaemons.ClusterDaemon workingCluster = null;
        Domain workingDomain = null;
        List<Server> servers = new ArrayList<Server>();

        for (Domain d : this.topology.getDomains().getDomain()) {
            if (d.getName().equals(domain)) {
                workingDomain = d;
                for (Domain.ClusterDaemons.ClusterDaemon cd : d.getClusterDaemons().getClusterDaemon()) {
                    if (cd.getName().equals(cluster)) {
                        workingCluster = cd;
                        for (String serverName : cd.getServers().getServerName()) {
                            boolean found = false;
                            for (Server s : workingDomain.getServers().getServer()) {
                                if (s.getName().equals(serverName)) {
                                    if (servers.contains(s)) {
                                        throw new IllegalArgumentException("Server '" + serverName
                                            + "' appears twice in the cluster daemon '" + cluster + "' server list!");
                                    }
                                    servers.add(s);
                                    found = true;
                                    break;
                                }
                            }
                            if (!found) {
                                throw new IllegalArgumentException("Server '" + serverName + "' in the cluster daemon '"
                                    + cluster + "' server list does not appear in the topology!");
                            }
                        }
                    }
                }
            }
            if (workingCluster != null) {
                break;
            }
        }

        for (Server s : servers) {
            ret.add(new ServerDomain(s, workingDomain));
        }

        return ret;
    }

    /**
     * Build the servers list (in this case, only 1 server in the list) defined
     * by domain.server
     * 
     * @param domain the domain
     * @param server the server
     * @return A list with the server to configure
     */
    private List<ServerDomain> buildServerListWithServer(final String domain, final String server) {
        List<ServerDomain> ret = new ArrayList<ServerDomain>();

        for (Domain d : this.topology.getDomains().getDomain()) {
            if (d.getName().equals(domain)) {
                for (Server s : d.getServers().getServer()) {
                    if (s.getName().equals(server)) {
                        ret.add(new ServerDomain(s, d));
                        break;
                    }
                }
                if (!ret.isEmpty()) {
                    break;
                }
            }
        }

        return ret;
    }

    /**
     * Build the servers list hosted by the machine machine
     * 
     * @param machine The machine where are located the servers to configure
     * @return A list with the servers to configure
     */
    private List<ServerDomain> buildServerListWithHostname(final String machine) {
        List<ServerDomain> ret = new ArrayList<ServerDomain>();

        for (Domain d : this.topology.getDomains().getDomain()) {
            for (Server s : d.getServers().getServer()) {
                if (s.getMachine().equals(machine)) {
                    ret.add(new ServerDomain(s, d));
                }
            }
        }

        return ret;
    }

    /**
     * Merge the configuration of the domain with the configuration of the
     * cluster daemon
     * 
     * @param cd cluster daemon configuration
     * @param domain the domain
     * @return the merged configuration
     */
    protected Domain.ClusterDaemons.ClusterDaemon mergeClusterDaemonAndDomainConf(final Domain.ClusterDaemons.ClusterDaemon cd,
        final Domain domain) {
        if (cd.getJonasRoot() == null) {
            cd.setJonasRoot(domain.getConfiguration().getGlobalOptions().getJonasRoot());
        }
        if (cd.getJavaHome() == null) {
            cd.setJavaHome(domain.getConfiguration().getGlobalOptions().getJdk());
        }

        return cd;
    }

}
