package org.objectweb.fractal.rmi;

import java.io.File;
import java.lang.reflect.Method;

import org.objectweb.asm.ClassVisitor;
import org.objectweb.asm.ClassWriter;
import org.objectweb.asm.Label;
import org.objectweb.asm.MethodVisitor;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;
import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.objectweb.fractal.api.control.ClassLoaderController;
import org.objectweb.fractal.deployment.local.api.GenericInstallingFactory;
import org.objectweb.fractal.rmi.stub.Skeleton;
import org.objectweb.fractal.rmi.stub.Stub;
import org.objectweb.util.monolog.api.BasicLevel;
import org.objectweb.util.monolog.api.Logger;

public class ClassGeneratorImpl implements Opcodes, ClassGenerator,
		ClassLoaderController {

	/**
	 * This flag is set to true if generated stub classes must be written to
	 * files.
	 */
	static private boolean writeStubs = false;

	/**
	 * This flag is set to true if generated skeleton classes must be written to
	 * files.
	 */
	static private boolean writeSkels = false;

	/**
	 * The logger used to log messages. May be <tt>null</tt>.
	 */

	protected Logger logger;

	/**
	 * A ClassLoaderController of this component
	 */
	private ClassLoaderController clc;

	/**
	 * A class loader used by this component to load classes
	 */
	protected ClassLoader cl = Thread.currentThread().getContextClassLoader();

	private EmbeddedLoader el = new EmbeddedLoader(cl);

	public Class loadClass(String className) throws ClassNotFoundException {
		return cl.loadClass(className);		
	}

	public Class generateClass(String className, ClassLoader cl) {
		setFcClassLoader(cl);
		return generateClass(className);
	}

	public Class generateClass(String className) {
		try {
			return el.loadClass(className);
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
		}
		return null;
	}

	// --------------------------------------------------------------------------
	// Class loading
	// --------------------------------------------------------------------------

	class EmbeddedLoader extends ClassLoader {

		/**
		 * Constructs a {@link EmbeddedLoader} with the given parent class
		 * loader.
		 * 
		 * @param parent
		 *            the parent class loader.
		 */

		public EmbeddedLoader(final ClassLoader parent) {
			super(parent);
		}

		/**
		 * Finds the class whose name is given. If the given name is of the form
		 * <i>prefix</i><tt>_Stub</tt> then this method calls the {@link
		 * #generateStubClass generateStubClass} method with the <i>prefix</i>
		 * class as parameter. If the given name is of the form <i>prefix</i><tt>_Skel</tt>
		 * then this method calls the
		 * {@link #generateSkeletonClass generateSkeletonClass} method with the
		 * <i>prefix</i> class as parameter. In all other cases this method
		 * throws an exception.
		 * 
		 * @param name
		 *            the name of the class to be found.
		 * @return the specified class.
		 * @throws ClassNotFoundException
		 *             if the class cannot be found.
		 */
		protected Class findClass(final String name)
				throws ClassNotFoundException {
			if (isStubClassName(name)) {
				Class itf = loadClass(resolveStubClassName(name));
				ClassWriter cw = new ClassWriter(true);
				generateStubClass(cw, itf);
				byte[] b = cw.toByteArray();
				if (logger != null && logger.isLoggable(BasicLevel.INFO)) {
					logger
							.log(BasicLevel.INFO, "Stub class generated: "
									+ name);
				}
				if (writeStubs) {
					try {
						String dirName = "./"
								+ name.substring(0, name.lastIndexOf('.'))
										.replace('.', '/');
						// create the directory
						new File(dirName).mkdirs();

						java.io.OutputStream os;
						os = new java.io.FileOutputStream("./"
								+ name.replace('.', '/') + ".class");
						os.write(b);
						os.close();
					} catch (Exception e) {
						// TODO: modified
						e.printStackTrace();
					}
				}
				return defineClass(name, b, 0, b.length);
			} else if (isSkelClassName(name)) {
				Class itf = loadClass(resolveSkelClassName(name));
				ClassWriter cw = new ClassWriter(true);
				generateSkeletonClass(cw, itf);
				byte[] b = cw.toByteArray();
				if (logger != null && logger.isLoggable(BasicLevel.INFO)) {
					logger.log(BasicLevel.INFO, "Skeleton class generated: "
							+ name);
				}
				if (writeSkels) {
					try {
						String dirName = "./"
								+ name.substring(0, name.lastIndexOf('.'))
										.replace('.', '/');
						// create the directory
						new File(dirName).mkdirs();

						java.io.OutputStream os;
						os = new java.io.FileOutputStream("./"
								+ name.replace('.', '/') + ".class");
						os.write(b);
						os.close();
					} catch (Exception e) {
						// TODO: modified
						e.printStackTrace();
					}
				}
				return defineClass(name, b, 0, b.length);
			} else {
				throw new ClassNotFoundException(name);
			}
		}
	}

	// --------------------------------------------------------------------------
	// Utility methods: class generation
	// --------------------------------------------------------------------------

	/**
	 * Generates a stub class for the given Java interface. This method
	 * generates a sub class of the {@link Stub} class that implements the given
	 * interface, by using the given class visitor.
	 * 
	 * @param cv
	 *            the class visitor to be used to generate the stub class.
	 * @param itf
	 *            the Java interface that the stub class must implement.
	 */

	protected void generateStubClass(final ClassVisitor cv, final Class itf) {
		int access = ACC_PUBLIC;
		String name = getStubClassName(itf.getName()).replace('.', '/');
		String superClass = "org/objectweb/fractal/rmi/stub/Stub";
		String[] itfs = new String[] { Type.getInternalName(itf) };
		cv.visit(V1_1, access, name, null, superClass, itfs);
		generateConstructor(cv, superClass);
		Method[] meths = itf.getMethods();
		sort(meths);
		for (int i = 0; i < meths.length; ++i) {
			Method meth = meths[i];
			generateStubMethod(cv, name, meth, i);
		}
	}

	/**
	 * Generates a skeleton class for the given Java interface. This method
	 * generates a sub class of the {@link Skeleton} class whose {@link
	 * Skeleton#target target} is an object implementing the given interface, by
	 * using the given class visitor.
	 * 
	 * @param cv
	 *            the class visitor to be used to generate the stub class.
	 * @param itf
	 *            the Java interface implemented by the skeleton's target.
	 */

	protected void generateSkeletonClass(final ClassVisitor cv, final Class itf) {
		int access = ACC_PUBLIC;
		String name = getSkelClassName(itf.getName()).replace('.', '/');
		String superClass = "org/objectweb/fractal/rmi/stub/Skeleton";
		cv.visit(V1_1, access, name, null, superClass, null);
		generateConstructor(cv, superClass);
		generateSkeletonMethod(cv, name, itf);
		cv.visitEnd();
	}

	/**
	 * Generates an empty constructor for the given class.
	 * 
	 * @param cv
	 *            the class visitor to be used to generate the constructor.
	 * @param superClass
	 *            the internal name of the super class of the generated class.
	 *            This name is used to generate a call to the super constructor.
	 */

	protected void generateConstructor(final ClassVisitor cv,
			final String superClass) {
		MethodVisitor mv = cv.visitMethod(ACC_PUBLIC, "<init>", "()V", null,
				null);
		// generates the bytecode corresponding to 'super(...);'
		mv.visitVarInsn(ALOAD, 0);
		mv.visitMethodInsn(INVOKESPECIAL, superClass, "<init>", "()V");
		mv.visitInsn(RETURN);
		mv.visitMaxs(1, 1);
		mv.visitEnd();
	}

	/**
	 * Generates a stub method. A stub method is of the following form:
	 * <p>
	 * 
	 * <pre>
	 *           public &lt;i&gt;T&lt;/i&gt; m (&lt;i&gt;T0 arg0&lt;/i&gt;, ...) throws &lt;i&gt;E0&lt;/i&gt;, ... {
	 *             try {
	 *               Marshaller marshaller = request();
	 *               ReplyInterface reply = prepareInvocation(marshaller);
	 *               marshaller.writeInt(&lt;i&gt;methodIndex&lt;/i&gt;);
	 *               marshaller.write&lt;i&gt;XXX&lt;/i&gt;(&lt;i&gt;arg0&lt;/i&gt;);
	 *               ...
	 *               invoke(marshaller);
	 *               UnMarshaller unmarshaller = reply.listen();
	 *               &lt;i&gt;T&lt;/i&gt; result = (&lt;i&gt;T&lt;/i&gt;)unmarshaller.read&lt;i&gt;XXX&lt;/i&gt;();
	 *               unmarshaller.close();
	 *               return result;
	 *             } catch (Exception e) {
	 *               e = handleException(e);
	 *               if (e instanceof &lt;i&gt;E0&lt;/i&gt;) throw (&lt;i&gt;E0&lt;/i&gt;)e;
	 *               ...
	 *               if (e instanceof RuntimeException) throw (RuntimeException)e;
	 *               throw new RemoteException(&quot;server side exception&quot;, e);
	 *             }
	 *           }
	 * </pre>
	 * 
	 * @param cv
	 *            the class visitor to be used to generate the stub method.
	 * @param name
	 *            the internal name of the generated stub class.
	 * @param m
	 *            the signature of the method that must be generated.
	 * @param index
	 *            the index of this method. This index will be marshalled by the
	 *            generated stub method in each invocation message, to specify
	 *            the method that must be called on the server side. This index
	 *            is the index of the method in the {@link #sort sorted} array
	 *            of all the methods implemented by the stub.
	 */

	protected void generateStubMethod(final ClassVisitor cv, final String name,
			final Method m, final int index) {
		// generate the method header
		String mName = m.getName();
		String mDesc = Type.getMethodDescriptor(m);
		Class[] params = m.getParameterTypes();
		int locals = 1;
		for (int i = 0; i < params.length; ++i) {
			locals += getSize(params[i]);
		}
		Class result = m.getReturnType();
		Class[] exceptions = m.getExceptionTypes();
		String[] excepts = new String[exceptions.length];
		for (int i = 0; i < exceptions.length; ++i) {
			excepts[i] = Type.getInternalName(exceptions[i]);
		}
		MethodVisitor mv = cv.visitMethod(ACC_PUBLIC, mName, mDesc, null,
				excepts);

		Label beginL = new Label();
		Label endL = new Label();

		mv.visitLabel(beginL);
		// generate "Marshaller marshaller = request();"
		mv.visitVarInsn(ALOAD, 0);
		mv.visitMethodInsn(INVOKEVIRTUAL, name, "request",
				"()Lorg/objectweb/jonathan/apis/presentation/Marshaller;");
		mv.visitVarInsn(ASTORE, locals);

		// generate "ReplyInterface reply = prepareInvocation(marshaller);"
		mv.visitVarInsn(ALOAD, 0);
		mv.visitVarInsn(ALOAD, locals);
		mv
				.visitMethodInsn(
						INVOKEVIRTUAL,
						name,
						"prepareInvocation",
						"(Lorg/objectweb/jonathan/apis/presentation/Marshaller;)Lorg/objectweb/jonathan/apis/protocols/ReplyInterface;");
		mv.visitVarInsn(ASTORE, locals + 1);

		// generate "marshaller.writeInt(<index>);"
		mv.visitVarInsn(ALOAD, locals);
		mv.visitIntInsn(SIPUSH, index);
		mv.visitMethodInsn(INVOKEINTERFACE,
				"org/objectweb/jonathan/apis/presentation/Marshaller",
				"writeInt", "(I)V");

		// parameter marshalling
		int offset = 1;
		for (int i = 0; i < params.length; ++i) {
			mv.visitVarInsn(ALOAD, locals);
			if (params[i].isPrimitive()) {
				mv.visitVarInsn(ILOAD + getOpcodeOffset(params[i]), offset);
				mv.visitMethodInsn(INVOKEINTERFACE,
						"org/objectweb/jonathan/apis/presentation/Marshaller",
						"write" + getMarshallerMethod(params[i]), "("
								+ Type.getDescriptor(params[i]) + ")V");
			} else {
				if (isClassParameter(m, i)) {
					mv.visitVarInsn(ALOAD, 0);
					mv.visitVarInsn(ILOAD + getOpcodeOffset(params[i]), offset);
					mv.visitMethodInsn(INVOKEVIRTUAL,
							"org/objectweb/fractal/rmi/stub/Stub",
							"replaceClassName",
							"(Ljava/lang/Object;)Ljava/lang/Object;");
				} else {
					mv.visitVarInsn(ILOAD + getOpcodeOffset(params[i]), offset);
				}
				mv.visitMethodInsn(INVOKEINTERFACE,
						"org/objectweb/jonathan/apis/presentation/Marshaller",
						"writeValue", "(Ljava/lang/Object;)V");
			}
			offset += getSize(params[i]);
		}

		// generate "invoke(marshaller);"
		mv.visitVarInsn(ALOAD, 0);
		mv.visitVarInsn(ALOAD, locals);
		mv.visitMethodInsn(INVOKEVIRTUAL, name, "invoke",
				"(Lorg/objectweb/jonathan/apis/presentation/Marshaller;)V");

		// generate "UnMarshaller unmarshaller = reply.listen();"
		mv.visitVarInsn(ALOAD, locals + 1);
		mv.visitMethodInsn(INVOKEINTERFACE,
				"org/objectweb/jonathan/apis/protocols/ReplyInterface",
				"listen",
				"()Lorg/objectweb/jonathan/apis/presentation/UnMarshaller;");
		mv.visitVarInsn(ASTORE, locals + 2);

		// return value unmarshalling
		if (result != Void.TYPE) {
			if (result.isPrimitive()) {
				mv.visitVarInsn(ALOAD, locals + 2);
				mv
						.visitMethodInsn(
								INVOKEINTERFACE,
								"org/objectweb/jonathan/apis/presentation/UnMarshaller",
								"read" + getMarshallerMethod(result), "()"
										+ Type.getDescriptor(result));
			} else {
				if (isClassParameter(m, -1)) {
					mv.visitVarInsn(ALOAD, 0);
					mv.visitVarInsn(ALOAD, locals + 2);
					mv
							.visitMethodInsn(
									INVOKEINTERFACE,
									"org/objectweb/jonathan/apis/presentation/UnMarshaller",
									"readValue", "()Ljava/lang/Object;");
					mv.visitMethodInsn(INVOKEVIRTUAL,
							"org/objectweb/fractal/rmi/stub/Stub",
							"replaceClassValue",
							"(Ljava/lang/Object;)Ljava/lang/Object;");
				} else {
					mv.visitVarInsn(ALOAD, locals + 2);
					mv
							.visitMethodInsn(
									INVOKEINTERFACE,
									"org/objectweb/jonathan/apis/presentation/UnMarshaller",
									"readValue", "()Ljava/lang/Object;");
				}
				mv.visitTypeInsn(CHECKCAST, Type.getInternalName(result));
			}
			mv.visitVarInsn(ISTORE + getOpcodeOffset(result), locals + 3);
		}

		// generate "unmarshaller.close();"
		mv.visitVarInsn(ALOAD, locals + 2);
		mv.visitMethodInsn(INVOKEINTERFACE,
				"org/objectweb/jonathan/apis/presentation/UnMarshaller",
				"close", "()V");

		// generate "return result";
		if (result != Void.TYPE) {
			mv.visitVarInsn(ILOAD + getOpcodeOffset(result), locals + 3);
			mv.visitInsn(IRETURN + getOpcodeOffset(result));
		} else {
			mv.visitInsn(RETURN);
		}
		mv.visitLabel(endL);

		mv.visitVarInsn(ASTORE, locals);

		// generate "e = handleException(e);"
		mv.visitVarInsn(ALOAD, 0);
		mv.visitVarInsn(ALOAD, locals);
		mv.visitMethodInsn(INVOKEVIRTUAL, name, "handleException",
				"(Ljava/lang/Exception;)Ljava/lang/Exception;");
		mv.visitVarInsn(ASTORE, locals);

		// generate "if (e instanceof <type>) throw (<type>)e;"
		for (int i = 0; i < exceptions.length; ++i) {
			String type = Type.getInternalName(exceptions[i]);
			mv.visitVarInsn(ALOAD, locals);
			mv.visitTypeInsn(INSTANCEOF, type);
			Label next = new Label();
			mv.visitJumpInsn(IFEQ, next);
			mv.visitVarInsn(ALOAD, locals);
			mv.visitTypeInsn(CHECKCAST, type);
			mv.visitInsn(ATHROW);
			mv.visitLabel(next);
		}

		// generate "if (e instanceof RuntimeException) throw
		// (RuntimeException)e;"
		mv.visitVarInsn(ALOAD, locals);
		mv.visitTypeInsn(INSTANCEOF, "java/lang/RuntimeException");
		Label next = new Label();
		mv.visitJumpInsn(IFEQ, next);
		mv.visitVarInsn(ALOAD, locals);
		mv.visitTypeInsn(CHECKCAST, "java/lang/RuntimeException");
		mv.visitInsn(ATHROW);
		mv.visitLabel(next);

		// generate "throw new RemoteException("server side exception", e);"
		mv.visitTypeInsn(NEW, "org/objectweb/fractal/rmi/RemoteException");
		mv.visitInsn(DUP);
		mv.visitLdcInsn("server side exception");
		mv.visitVarInsn(ALOAD, locals);
		mv.visitMethodInsn(INVOKESPECIAL,
				"org/objectweb/fractal/rmi/RemoteException", "<init>",
				"(Ljava/lang/String;Ljava/lang/Exception;)V");
		mv.visitInsn(ATHROW);

		// end method body
		mv.visitTryCatchBlock(beginL, endL, endL, "java/lang/Exception");
		mv.visitMaxs(0, 0);
		mv.visitEnd();
	}

	/**
	 * Generates a skeleton's {@link Skeleton#send send} method. The generated
	 * method is of the following form:
	 * <p>
	 * 
	 * <pre>
	 *           public void send (UnMarshaller unmarshaller, ReplySession session)
	 *             throws JonathanException
	 *           {
	 *             int methodIndex = unmarshaller.readInt();
	 *             try {
	 *               switch (methodIndex) {
	 *                 case 0:
	 *                   &lt;i&gt;T0 arg0&lt;/i&gt; = unmarshaller.read&lt;i&gt;XXX&lt;/i&gt;();
	 *                   ...
	 *                   unmarshaller.close();
	 *                   &lt;i&gt;T&lt;/i&gt; result = ((&lt;i&gt;I&lt;/i&gt;)target).&lt;i&gt;m0&lt;/i&gt;(&lt;i&gt;arg0&lt;/i&gt;, ... );
	 *                   Marshaller marshaller = session.prepareReply();
	 *                   marshaller.write&lt;i&gt;XXX&lt;/i&gt;(result);
	 *                   session.send(marshaller);
	 *                   session.close();
	 *                   return;
	 *                 ...
	 *                 default:
	 *                   handleInterfaceMethods(unmarshaller, session, methodIndex);
	 *               }
	 *             } catch (Exception e) {
	 *               handleException(e, session);
	 *             }
	 *           }
	 * </pre>
	 * 
	 * @param cv
	 *            the class visitor to be used to generate the stub method.
	 * @param name
	 *            the internal name of the generated stub class.
	 * @param itf
	 *            the target object's interface that must be exported by the
	 *            skeleton.
	 */

	protected void generateSkeletonMethod(final ClassVisitor cv,
			final String name, final Class itf) {
		MethodVisitor mv = cv
				.visitMethod(
						ACC_PUBLIC,
						"send",
						"(Lorg/objectweb/jonathan/apis/presentation/UnMarshaller;Lorg/objectweb/jonathan/apis/protocols/ReplySession;)V",
						null,
						new String[] { "org/objectweb/jonathan/apis/kernel/JonathanException" });

		Label beginL = new Label();
		Label endL = new Label();
		Label defaultL = new Label();

		// generate "int methodIndex = unmarshaller.readInt();"
		mv.visitLabel(beginL);
		mv.visitVarInsn(ALOAD, 1);
		mv.visitMethodInsn(INVOKEINTERFACE,
				"org/objectweb/jonathan/apis/presentation/UnMarshaller",
				"readInt", "()I");
		mv.visitVarInsn(ISTORE, 3);

		// generate the "switch (methodIndex)" statement
		Method[] meths = itf.getMethods();
		sort(meths);
		Label[] cases = new Label[meths.length];
		for (int i = 0; i < cases.length; ++i) {
			cases[i] = new Label();
		}
		if (meths.length > 0) {
			mv.visitVarInsn(ILOAD, 3);
			mv.visitTableSwitchInsn(0, meths.length - 1, defaultL, cases);
		}

		// generate the "case" blocks
		for (int i = 0; i < meths.length; ++i) {
			Method m = meths[i];
			String mName = m.getName();
			String mDesc = Type.getMethodDescriptor(m);
			Class[] params = m.getParameterTypes();
			Class result = m.getReturnType();

			mv.visitLabel(cases[i]);

			// generate "<type> obj = (<type>)target;"
			mv.visitVarInsn(ALOAD, 0);
			mv.visitFieldInsn(GETFIELD, name, "target", "Ljava/lang/Object;");
			mv.visitTypeInsn(CHECKCAST, Type.getInternalName(itf));

			// parameter unmarshalling
			for (int j = 0; j < params.length; ++j) {
				if (params[j].isPrimitive()) {
					mv.visitVarInsn(ALOAD, 1);
					mv
							.visitMethodInsn(
									INVOKEINTERFACE,
									"org/objectweb/jonathan/apis/presentation/UnMarshaller",
									"read" + getMarshallerMethod(params[j]),
									"()" + Type.getDescriptor(params[j]));
				} else {
					if (isClassParameter(m, j)) {
						mv.visitVarInsn(ALOAD, 0);
						mv.visitVarInsn(ALOAD, 1);
						mv
								.visitMethodInsn(
										INVOKEINTERFACE,
										"org/objectweb/jonathan/apis/presentation/UnMarshaller",
										"readValue", "()Ljava/lang/Object;");
						mv.visitMethodInsn(INVOKEVIRTUAL,
								"org/objectweb/fractal/rmi/stub/Skeleton",
								"replaceClassValue",
								"(Ljava/lang/Object;)Ljava/lang/Object;");
					} else {
						mv.visitVarInsn(ALOAD, 1);
						mv
								.visitMethodInsn(
										INVOKEINTERFACE,
										"org/objectweb/jonathan/apis/presentation/UnMarshaller",
										"readValue", "()Ljava/lang/Object;");
					}
					mv
							.visitTypeInsn(CHECKCAST, Type
									.getInternalName(params[j]));
				}
			}

			// generate "unmarshaller.close();"
			mv.visitVarInsn(ALOAD, 1);
			mv.visitMethodInsn(INVOKEINTERFACE,
					"org/objectweb/jonathan/apis/presentation/UnMarshaller",
					"close", "()V");

			// generate code to call target method and store result
			mv.visitMethodInsn(INVOKEINTERFACE, Type.getInternalName(itf),
					mName, mDesc);
			if (result != Void.TYPE) {
				mv.visitVarInsn(ISTORE + getOpcodeOffset(result), 5);
			}

			// generate "Marshaller marshaller = session.prepareReply();"
			mv.visitVarInsn(ALOAD, 2);
			mv.visitMethodInsn(INVOKEINTERFACE,
					"org/objectweb/jonathan/apis/protocols/ReplySession",
					"prepareReply",
					"()Lorg/objectweb/jonathan/apis/presentation/Marshaller;");
			mv.visitVarInsn(ASTORE, 4);

			// generate code to marshall result value
			if (result != Void.TYPE) {
				mv.visitVarInsn(ALOAD, 4);
				if (result.isPrimitive()) {
					mv.visitVarInsn(ILOAD + getOpcodeOffset(result), 5);
					mv
							.visitMethodInsn(
									INVOKEINTERFACE,
									"org/objectweb/jonathan/apis/presentation/Marshaller",
									"write" + getMarshallerMethod(result), "("
											+ Type.getDescriptor(result) + ")V");
				} else {
					if (isClassParameter(m, -1)) {
						mv.visitVarInsn(ALOAD, 0);
						mv.visitVarInsn(ILOAD + getOpcodeOffset(result), 5);
						mv.visitMethodInsn(INVOKEVIRTUAL,
								"org/objectweb/fractal/rmi/stub/Skeleton",
								"replaceClassName",
								"(Ljava/lang/Object;)Ljava/lang/Object;");
					} else {
						mv.visitVarInsn(ILOAD + getOpcodeOffset(result), 5);
					}
					mv
							.visitMethodInsn(
									INVOKEINTERFACE,
									"org/objectweb/jonathan/apis/presentation/Marshaller",
									"writeValue", "(Ljava/lang/Object;)V");
				}
			}

			// generate "session.send(marshaller);"
			mv.visitVarInsn(ALOAD, 2);
			mv.visitVarInsn(ALOAD, 4);
			mv.visitMethodInsn(INVOKEINTERFACE,
					"org/objectweb/jonathan/apis/protocols/ReplySession",
					"send",
					"(Lorg/objectweb/jonathan/apis/presentation/Marshaller;)V");

			// generate "session.close();"
			mv.visitVarInsn(ALOAD, 2);
			mv.visitMethodInsn(INVOKEINTERFACE,
					"org/objectweb/jonathan/apis/protocols/ReplySession",
					"close", "()V");

			// generate "return;"
			mv.visitInsn(RETURN);
		}

		// generate default block:
		// "handleInterfaceMethods(unmarshaller, session, methodIndex);"
		// "return;"
		mv.visitLabel(defaultL);
		mv.visitVarInsn(ALOAD, 0);
		mv.visitVarInsn(ALOAD, 1);
		mv.visitVarInsn(ALOAD, 2);
		mv.visitVarInsn(ILOAD, 3);
		mv
				.visitMethodInsn(
						INVOKEVIRTUAL,
						name,
						"handleInterfaceMethods",
						"(Lorg/objectweb/jonathan/apis/presentation/UnMarshaller;Lorg/objectweb/jonathan/apis/protocols/ReplySession;I)V");
		mv.visitInsn(RETURN);

		// generate "catch (Exception e) { handleException(e, session); }"
		mv.visitLabel(endL);
		mv.visitVarInsn(ASTORE, 3);
		mv.visitVarInsn(ALOAD, 0);
		mv.visitVarInsn(ALOAD, 3);
		mv.visitVarInsn(ALOAD, 2);
		mv
				.visitMethodInsn(INVOKEVIRTUAL, name, "handleException",
						"(Ljava/lang/Exception;Lorg/objectweb/jonathan/apis/protocols/ReplySession;)V");
		mv.visitInsn(RETURN);

		// end method code
		mv.visitTryCatchBlock(beginL, endL, endL, "java/lang/Exception");
		mv.visitMaxs(0, 0);
		mv.visitEnd();
	}

	/**
	 * Returns <tt>true</tt> if the specified parameter contains the name of a
	 * class. This method is used to marshall a <tt>Class</tt> object, instead
	 * of a <tt>String</tt>, for parameters that contains class names, in
	 * order to enable these classes to be downloaded from the network if
	 * needed. The default implementation of this method returns <tt>true</tt>
	 * for:
	 * <ul>
	 * <li>the return value of <tt>Factory.getFcContentDesc</tt>.</li>
	 * <li>the second parameter of <tt>GenericFactory.newFcInstance</tt>.</li>
	 * <li>the return value of <tt>InterfaceType.getFcItfSignature</tt>.</li>
	 * <li>the second parameter of <tt>TypeFactory.createFcItfType</tt>.</li>
	 * </ul>
	 * 
	 * @param m
	 *            a method.
	 * @param p
	 *            index of a parameter of this method, or <tt>-1</tt> to
	 *            designate the return value.
	 * @return <tt>true</tt> if the specified parameter contains the name of a
	 *         class.
	 */

	protected boolean isClassParameter(final Method m, final int p) {
		String name = m.getName();
		if (name.equals("getFcContentDesc")) {
			return p == -1;
		} else if (name.equals("newFcInstance")) {
			return p == 2;
		} else if (name.equals("getFcItfSignature")) {
			return p == -1;
		} else if (name.equals("createFcItfType")) {
			return p == 1;
		}
		return false;
	}

	/**
	 * Sorts the given methods. This method is used to assign an unambiguous
	 * index to each method of an interface (the index of a method in the array
	 * returned by {@link Class#getMethods() getMethods} cannot directly be used
	 * for this purpose, since this method returns the methods in any order).
	 * 
	 * @param methods
	 *            the method array to be sorted.
	 */

	protected static void sort(final Method[] methods) {
		String[] descs = new String[methods.length];
		for (int i = 0; i < descs.length; ++i) {
			descs[i] = methods[i].getName()
					+ Type.getMethodDescriptor(methods[i]);
		}
		for (int i = 0; i < descs.length; ++i) {
			for (int j = i + 1; j < descs.length; ++j) {
				if (descs[j].compareTo(descs[i]) < 0) {
					String s = descs[i];
					descs[i] = descs[j];
					descs[j] = s;
					Method m = methods[i];
					methods[i] = methods[j];
					methods[j] = m;
				}
			}
		}
	}

	/**
	 * Returns the suffix of the (un)marshaller method corresponding to the
	 * given primitive type.
	 * 
	 * @param type
	 *            a primitive type.
	 * @return the suffix of the (un)marshaller method corresponding to the
	 *         given primitive type (such as "Byte" for byte, or "Char16" for
	 *         char).
	 */

	private static String getMarshallerMethod(final Class type) {
		if (type == Byte.TYPE) {
			return "Byte";
		} else if (type == Integer.TYPE) {
			return "Int";
		} else if (type == Boolean.TYPE) {
			return "Boolean";
		} else if (type == Double.TYPE) {
			return "Double";
		} else if (type == Float.TYPE) {
			return "Float";
		} else if (type == Long.TYPE) {
			return "Long";
		} else if (type == Character.TYPE) {
			return "Char16";
		} else /* if (result == Short.TYPE) */{
			return "Short";
		}
	}

	/**
	 * Returns the offset which must be added to some opcodes to get an opcode
	 * of the given type. More precisely, returns the offset which must be added
	 * to an opc_iXXXX opcode to get the opc_YXXXX opcode corresponding to the
	 * given type. For example, if the given type is double the result is 3,
	 * which means that opc_dload, opc_dstore, opc_dreturn... opcodes are equal
	 * to opc_iload+3, opc_istore+3, opc_ireturn+3...
	 * 
	 * @param type
	 *            a Java class representing a Java type (primitive or not).
	 * @return the opcode offset of the corresponding to the given type.
	 */

	private static int getOpcodeOffset(final Class type) {
		if (type == Double.TYPE) {
			return 3;
		} else if (type == Float.TYPE) {
			return 2;
		} else if (type == Long.TYPE) {
			return 1;
		} else if (type.isPrimitive()) {
			return 0;
		}
		return 4;
	}

	/**
	 * Returns the size of the given type. This size is 2 for the double and
	 * long types, and 1 for the other types.
	 * 
	 * @param type
	 *            a Java class representing a Java type (primitive or not).
	 * @return the size of the given type.
	 */

	private static int getSize(final Class type) {
		return (type == Double.TYPE || type == Long.TYPE ? 2 : 1);
	}

	/**
	 * Returns <tt>true</tt> if the given name if the name of a stub class.
	 * 
	 * @param className
	 *            a fully qualified class name.
	 * @return <tt>true</tt> if the given name if the name of a stub class.
	 */

	private static boolean isStubClassName(final String className) {
		return className.endsWith("_JavaStub") || className.endsWith("_Stub");
	}

	/**
	 * Returns the class name corresponding to the given stub class name.
	 * 
	 * @param stubClassName
	 *            a fully qualified stub class name.
	 * @return the class name corresponding to the given stub class name.
	 */

	private static String resolveStubClassName(final String stubClassName) {
		if (stubClassName.endsWith("_JavaStub")) {
			return stubClassName.substring(5, stubClassName.length() - 9);
		} else {
			return stubClassName.substring(0, stubClassName.length() - 5);
		}
	}

	/**
	 * Returns <tt>true</tt> if the given name if the name of a skeleton
	 * class.
	 * 
	 * @param className
	 *            a fully qualified class name.
	 * @return <tt>true</tt> if the given name if the name of a skeleton
	 *         class.
	 */

	private static boolean isSkelClassName(final String className) {
		return className.endsWith("_JavaSkel") || className.endsWith("_Skel");
	}

	/**
	 * Returns the class name corresponding to the given skeleton class name.
	 * 
	 * @param skelClassName
	 *            a fully qualified skeleton class name.
	 * @return the class name corresponding to the given skeleton class name.
	 */

	private static String resolveSkelClassName(final String skelClassName) {
		if (skelClassName.endsWith("_JavaSkel")) {
			return skelClassName.substring(5, skelClassName.length() - 9);
		} else {
			return skelClassName.substring(0, skelClassName.length() - 5);
		}
	}

	/**
	 * Returns the name of the skeleton class for the given class.
	 * 
	 * @param className
	 *            a fully qualified class name.
	 * @return the name of the skeleton class for the given class.
	 */

	private static String getSkelClassName(final String className) {
		if (className.startsWith("java.")) {
			return "skel." + className + "_JavaSkel";
		} else {
			return className + "_Skel";
		}
	}

	/**
	 * Returns the name of the stub class for the given class.
	 * 
	 * @param className
	 *            a fully qualified class name.
	 * @return the name of the stub class for the given class.
	 */

	private static String getStubClassName(final String className) {
		if (className.startsWith("java.")) {
			return "stub." + className + "_JavaStub";
		} else {
			return className + "_Stub";
		}
	}

	public ClassLoader getFcClassLoader() {
		return this.cl;
	}

	public void setFcClassLoader(ClassLoader cl) {
		this.cl = cl;
		this.el = new EmbeddedLoader(cl);
	}
}
