/***
 * Fractal RMI: a binder for remote method calls between Fractal components.
 * Copyright (C) 2003 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact: Eric.Bruneton@rd.francetelecom.com
 *
 * Author: Eric Bruneton
 *
 * adapted from Jonathan:
 *   org.objectweb.jeremie.libs.presentation.StdMarshallerFactory
 *     (authors: B. Dumant, K. Milsted)
 */

package org.objectweb.fractal.rmi.io;

import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectStreamClass;
import java.rmi.server.RMIClassLoader;

import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.Interface;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.objectweb.fractal.api.control.ClassLoaderController;
import org.objectweb.fractal.rmi.ClassGenerator;
import org.objectweb.jonathan.apis.binding.Identifier;
import org.objectweb.jonathan.apis.binding.NamingContext;
import org.objectweb.jonathan.apis.kernel.Context;
import org.objectweb.jonathan.apis.kernel.ContextFactory;

/**
 * An {@link ObjectInputStream} that replaces {@link Ref} objects with stubs.
 */

public class MyRmiObjectInputStream extends ObjectInputStream {

	/**
	 * The naming context used to decode the identifiers contained in
	 * {@link Ref} objects.
	 */

	protected NamingContext domain;

	/**
	 * The context factory used to create hints for the {@link Identifier#bind
	 * bind} method.
	 */

	protected ContextFactory contextFactory;

	/**
	 * The code base to be used to load classes.
	 */
	protected String codeBase;

	/**
	 * A loader to be used by this input stream
	 */
	private ClassLoaderController clc = null;

	/**
	 * Constructs a new {@link MyRmiObjectInputStream}.
	 * 
	 * @param is
	 *            the underlying input stream.
	 * @param domain
	 *            the naming context to be used to decode the identifiers
	 *            contained in {@link Ref} objects.
	 * @param contextFactory
	 *            the context factory to be used to create hints for the
	 *            {@link Identifier#bind bind} method.
	 * @throws IOException
	 *             if the super constructor throws an exception.
	 */

	public MyRmiObjectInputStream(final InputStream is,
			final NamingContext domain, final ContextFactory contextFactory,
			final ClassGenerator loader) throws IOException {
		super(is);
		enableResolveObject(true);
		this.domain = domain;
		this.contextFactory = contextFactory;
		this.codeBase = readUTF();

		Component loaderComponent = ((Interface) loader).getFcItfOwner();
		try {
			this.clc = (ClassLoaderController) loaderComponent
					.getFcInterface("class-loader-controller");
		} catch (NoSuchInterfaceException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Replaces {@link Ref} objects with corresponding stubs. This method uses
	 * the {@link #domain domain} to decode the identifier encoded in a {@link
	 * Ref} object, and then uses the {@link Identifier#bind bind} method of the
	 * decoded identifier to get a stub corresponding to this identifier. The
	 * type stored in the {@link Ref} object is passed as an hint to the
	 * {@link Identifier#bind bind} method.
	 * 
	 * @param obj
	 *            an object.
	 * @return a stub if obj is a {@link Ref} object, or <tt>obj</tt>
	 *         otherwise.
	 * @throws IOException
	 *             if a {@link Ref} object cannot be replaced with a stub.
	 */

	protected Object resolveObject(final Object obj) throws IOException {
		if (obj instanceof Ref) {
			try {
				Ref ref = (Ref) obj;
				Identifier id = domain.decode(ref.id, 0, ref.id.length);
				Context hints = contextFactory.newContext();
				hints.addElement("interface_type", String.class, ref.type,
						(char) 0);
				Object newObj = id.bind(new Identifier[] { id }, hints);
				hints.release();
				return newObj;
			} catch (Exception e) {
				throw new IOException("cannot bind to object: " + e);
			}
		} else {
			return obj;
		}
	}

	protected Class resolveClass(ObjectStreamClass desc)
			throws ClassNotFoundException, IOException {
		try {
			Class result = null;
			try {
				result = RMIClassLoader.loadClass((String) codeBase, desc
						.getName(), clc.getFcClassLoader());
			} catch (Exception e) {
				try {
					result = RMIClassLoader.loadClass((String) codeBase, desc
							.getName(), Thread.currentThread()
							.getContextClassLoader());
				} catch (Exception ex) {
					result = getClass().getClassLoader().loadClass(
							desc.getName());
				}
			}
			return result;
		} catch (ClassNotFoundException e) {
			System.err.println("WARNING: " + e.toString());
			throw e;
		}
	}
}
