/***
 * Fractal RMI: a binder for remote method calls between Fractal components.
 * Copyright (C) 2003 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact: Eric.Bruneton@rd.francetelecom.com
 *
 * Author: Bruno Dillenseger, Eric Bruneton
 */

package org.objectweb.fractal.rmi.registry;

import java.net.InetAddress;
import java.util.HashMap;
import java.util.Map;

import org.objectweb.fractal.adl.Factory;
import org.objectweb.fractal.adl.FactoryFactory;
import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.control.ClassLoaderController;
import org.objectweb.fractal.api.control.ContentController;
import org.objectweb.fractal.api.factory.GenericFactory;
import org.objectweb.fractal.api.type.ComponentType;
import org.objectweb.fractal.api.type.InterfaceType;
import org.objectweb.fractal.api.type.TypeFactory;
import org.objectweb.fractal.deployment.local.api.GenericInstallingFactory;
import org.objectweb.fractal.jonathan.JContextFactory;
import org.objectweb.fractal.util.Fractal;
import org.objectweb.jonathan.apis.binding.Identifier;
import org.objectweb.jonathan.apis.binding.NamingContext;
import org.objectweb.jonathan.apis.kernel.Context;
import org.objectweb.jonathan.apis.kernel.JonathanException;

/**
 * Provides static methods to launch and get access to a Fractal registry.
 */

public class Registry {

    /**
     * The default port used to export the naming service interface.
     */

    public final static int DEFAULT_PORT = 1234;

    /**
     * Private constructor (uninstantiable class).
     */

    private Registry() {
    }

    /**
     * Launches a Fractal registry on the local host.
     *
     * @param args
     *            a single argument setting the port number where to pick the
     *            naming service interface.
     * @throws Exception
     *             if something goes wrong.
     */

    public static void main(final String[] args) throws Exception {
        // port number to export the registry interface
        int port = DEFAULT_PORT;
        if (args.length == 1) {
            try {
                port = Integer.parseInt(args[0]);
            } catch (NumberFormatException nfe) {
            }
        }
        createRegistry(port);
    }

    /**
     * Creates {@link NamingService} component and exports its server interface.
     *
     * @param port
     *            the port number to be used to export the naming service
     *            interface.
     * @throws Exception
     *             if something goes wrong.
     */

    public static Component createRegistry(final int port) throws Exception {
        return createRegistry(port, new HashMap());
    }

    /**
     * Creates {@link NamingService} component, using the provided
     * context/hints, and exports its server interface.
     *
     * @param port
     *            the port number to be used to export the naming service
     *            interface.
     * @param hints
     *            context/hints used when getting the factory, creating the ORB
     *            component and getting the bootstrap component
     * @see FactoryFactory#getFactory(java.lang.String, java.util.Map)
     * @see Factory#newComponent(java.lang.String, java.util.Map)
     * @see Fractal#getBootstrapComponent(java.util.Map)
     * @throws Exception
     *             if something goes wrong.
     */

    public static Component createRegistry(final int port, Map hints)
            throws Exception {
        // the RMI class loader does not work if there is no security manager
        System.setSecurityManager(new SecurityManager());

        // classloader hints set-up
        if (hints == null) {
            hints = new HashMap();
            hints.put("classloader", Thread.currentThread()
                    .getContextClassLoader() == null ? Registry.class
                    .getClassLoader() : Thread.currentThread()
                    .getContextClassLoader());
        }

        // constructs the Fractal RMI binder
        Factory f = FactoryFactory.getFactory(FactoryFactory.FRACTAL_BACKEND,
                hints);
        Component comp = (Component) f.newComponent(
                "org.objectweb.fractal.rmi.ORB", hints);

        ClassLoader cl = (ClassLoader) hints.get("classloader");
        if (cl != null) {
            ContentController cc = Fractal.getContentController(comp);
            Component[] subCmps = cc.getFcSubComponents();
            for (int i = 0; i < subCmps.length; i++) {
                if (Fractal.getNameController(subCmps[i]).getFcName().equals(
                        "class-generator")) {
                    ClassLoaderController clc = (ClassLoaderController) subCmps[i]
                            .getFcInterface("class-loader-controller");
                    clc.setFcClassLoader(cl);
                    break;
                }
            }
        }

        Component bootComp = Fractal.getBootstrapComponent();
        GenericInstallingFactory gif = (GenericInstallingFactory) bootComp
                .getFcInterface("generic-installing-factory");
        if (gif != null) {
            Fractal.getBindingController(comp).bindFc("component-factory", gif);
        }

        Fractal.getLifeCycleController(comp).startFc();
        NamingContext binder = (NamingContext) comp.getFcInterface("context");

        // creates the name service component
        Component boot = Fractal.getBootstrapComponent(hints);
        TypeFactory tf = Fractal.getTypeFactory(boot);
        ComponentType nsType = tf
                .createFcType(new InterfaceType[] {
                        tf
                                .createFcItfType(
                                        NamingService.NAMING_SERVICE,
                                        "org.objectweb.fractal.rmi.registry.NamingService",
                                        TypeFactory.SERVER,
                                        TypeFactory.MANDATORY,
                                        TypeFactory.SINGLE),
                        tf
                                .createFcItfType(
                                        "attribute-controller",
                                        "org.objectweb.fractal.rmi.registry.NamingServiceAttributes",
                                        TypeFactory.SERVER,
                                        TypeFactory.MANDATORY,
                                        TypeFactory.SINGLE) });
        GenericFactory cf = Fractal.getGenericFactory(boot);
        Component nsId = cf.newFcInstance(nsType, "primitive",
                "org.objectweb.fractal.rmi.registry.NamingServiceImpl");
        Fractal.getLifeCycleController(nsId).startFc();

        // exports the NamingService interface of the naming service component
        Context ctxt = new JContextFactory().newContext();
        ctxt.addElement("port", Integer.class, new Integer(port), (char) 0);
        ctxt.addElement("key", Integer.class, new Integer(-1), (char) 0);
        binder.export(nsId.getFcInterface(NamingService.NAMING_SERVICE), ctxt);

        System.out.println("Fractal registry is ready.");

        return comp;
    }

    /**
     * Returns a reference to the {@link NamingService} on the given host and
     * port.
     *
     * @param host
     *            the host where the naming service is located.
     * @param port
     *            the port that was used to export the naming service on the
     *            given host.
     * @param cl
     *            a class loader
     * @return a reference to the {@link NamingService} on the given host and
     *         port.
     * @throws Exception
     *             if something goes wrong.
     */

    public static Component createRegistry(final int port, final ClassLoader cl)
            throws Exception {

        Map hints = new HashMap();
        hints.put("classloader", cl);
        return createRegistry(port, hints);
    }

    /**
     * Returns a reference to the {@link NamingService} on the local host, for
     * the default {@link #DEFAULT_PORT port}.
     *
     * @return a reference to the {@link NamingService} on the local host, for
     *         the default {@link #DEFAULT_PORT port}.
     * @throws Exception
     *             if something goes wrong.
     */

    public static NamingService getRegistry() throws Exception {
        String host = InetAddress.getLocalHost().getHostName();
        return getRegistry(host);
    }

    /**
     * Returns a reference to the {@link NamingService} on the given host, for
     * the default {@link #DEFAULT_PORT port}.
     *
     * @param host
     *            the host where the naming service is located.
     * @return a reference to the {@link NamingService} on the given host, for
     *         the default {@link #DEFAULT_PORT port}.
     * @throws Exception
     *             if something goes wrong.
     */

    public static NamingService getRegistry(final String host) throws Exception {
        return getRegistry(host, DEFAULT_PORT);
    }

    /**
     * Returns a reference to the {@link NamingService} on the given host and
     * port.
     *
     * @param host
     *            the host where the naming service is located.
     * @param port
     *            the port that was used to export the naming service on the
     *            given host.
     * @return a reference to the {@link NamingService} on the given host and
     *         port.
     * @throws Exception
     *             if something goes wrong.
     */

    public static NamingService getRegistry(final String host, final int port)
            throws Exception {
//		// constructs the Fractal RMI binder
//		Factory f = FactoryFactory.getFactory(FactoryFactory.FRACTAL_BACKEND);
//		Component comp = (Component) f.newComponent(
//				"org.objectweb.fractal.rmi.ORB", null);
//		Component bootComp = Fractal.getBootstrapComponent();
//		GenericInstallingFactory gif = (GenericInstallingFactory) bootComp
//				.getFcInterface("generic-installing-factory");
//		if (gif != null) {
//			Fractal.getBindingController(comp).bindFc("component-factory", gif);
//		}
//		Fractal.getLifeCycleController(comp).startFc();
//		NamingContext binder = (NamingContext) comp.getFcInterface("context");
//		return getRegistry(host, port, binder);
        Map hints = new HashMap();
        hints.put("classloader", Registry.class.getClassLoader());
        return getRegistry(host, port, hints);
    }

    /**
     * Returns a reference to the {@link NamingService} on the given host and
     * port.
     *
     * @param host
     *            the host where the naming service is located.
     * @param port
     *            the port that was used to export the naming service on the
     *            given host.
     * @param class
     *            loader a class loader used by this ORB
     * @return a reference to the {@link NamingService} on the given host and
     *         port.
     * @throws Exception
     *             if something goes wrong.
     */
    public static NamingService getRegistry(final String host, final int port,
            final ClassLoader cl) throws Exception {

        Map hints = new HashMap();
        hints.put("classloader", cl);
        return getRegistry(host, port, hints);

    }

    /**
     * Returns a reference to the {@link NamingService} on the given host and
     * port.
     *
     * @param host
     *            the host where the naming service is located.
     * @param port
     *            the port that was used to export the naming service on the
     *            given host.
     * @param hints
     *            hints
     * @return a reference to the {@link NamingService} on the given host and
     *         port.
     * @throws Exception
     *             if something goes wrong.
     */

    public static NamingService getRegistry(final String host, final int port,
            final Map hints) throws Exception {

        Component comp = getRegistryComponent(host, port, hints);
        //
        // if(loader != null && cl != null) {
        // Loader ldr = (Loader) loader.getFcInterface("loader");
        // // fix the class loader used by the loader component
        // ldr.setParentLoader(cl);
        // }
        NamingContext binder = (NamingContext) comp.getFcInterface("context");
        return getRegistry(host, port, binder);
    }

    /**
     * Returns a reference to the {@link NamingService} on the given host, for
     * the default {@link #DEFAULT_PORT port}.
     *
     * @param host
     *            the host where the naming service is located.
     * @param binder
     *            the binder to be used to create the binding to the naming
     *            service interface.
     * @return a reference to the {@link NamingService} on the given host, for
     *         the default {@link #DEFAULT_PORT port}.
     * @throws JonathanException
     *             if something goes wrong.
     */

    public static NamingService getRegistry(final String host,
            final NamingContext binder) throws JonathanException {
        return getRegistry(host, DEFAULT_PORT, binder);
    }

    /**
     * Returns a reference to the {@link NamingService} on the given host and
     * port.
     *
     * @param host
     *            the host where the naming service is located.
     * @param port
     *            the port that was used to export the naming service on the
     *            given host.
     * @param binder
     *            the binder to be used to create the binding to the naming
     *            service interface.
     * @return a reference to the {@link NamingService} on the given host and
     *         port.
     * @throws JonathanException
     *             if something goes wrong.
     */

    public static NamingService getRegistry(final String host, final int port,
            final NamingContext binder) throws JonathanException {
        short len = (short) host.length();
        byte[] b = new byte[len + 10];
        // port
        b[0] = (byte) ((port >>> 24) & 0xFF);
        b[1] = (byte) ((port >>> 16) & 0xFF);
        b[2] = (byte) ((port >>> 8) & 0xFF);
        b[3] = (byte) (port & 0xFF);
        // host
        b[4] = (byte) ((len >>> 8) & 0xFF);
        b[5] = (byte) (len & 0xFF);
        for (int i = 0; i < len; ++i) {
            b[i + 6] = (byte) host.charAt(i);
        }
        // key
        int key = -1;
        b[len + 6] = (byte) ((key >>> 24) & 0xFF);
        b[len + 7] = (byte) ((key >>> 16) & 0xFF);
        b[len + 8] = (byte) ((key >>> 8) & 0xFF);
        b[len + 9] = (byte) (key & 0xFF);

        Identifier id = binder.decode(b, 0, b.length);

        Context hints = new JContextFactory().newContext();
        hints.addElement("interface_type", String.class,
                "org.objectweb.fractal.rmi.registry.NamingService", (char) 0);

        return (NamingService) id.bind(new Identifier[] { id }, hints);
    }

    public static Component getRegistryComponent(final String host,
            final int port, final ClassLoader cl) throws Exception {

        Map hints = new HashMap();
        hints.put("component-classloader", cl);
        return getRegistryComponent(host, port, hints);

    }

    public static Component getRegistryComponent(final String host,
            final int port, final Map hints) throws Exception {
        ClassLoader registryCl = (ClassLoader) hints
                .get("registry-classloader");
        ClassLoader componentCl = (ClassLoader) hints
                .get("component-classloader");

        if (registryCl != null) {
            hints.put("classloader", registryCl);
        }
        // constructs the Fractal RMI binder
        Factory f = FactoryFactory.getFactory(FactoryFactory.FRACTAL_BACKEND);
        Component comp = (Component) f.newComponent(
                "org.objectweb.fractal.rmi.ORB", hints);

        ClassLoaderController clc = null;
        ContentController cc = Fractal.getContentController(comp);
        Component[] subCmps = cc.getFcSubComponents();
        for (int i = 0; i < subCmps.length; i++) {
            if (Fractal.getNameController(subCmps[i]).getFcName().equals(
                    "class-generator")) {
                clc = (ClassLoaderController) subCmps[i]
                        .getFcInterface("class-loader-controller");
                break;
            }
        }

        if (clc != null
                && (componentCl != null || (componentCl = (ClassLoader) hints
                        .get("classloader")) != null)) {
            clc.setFcClassLoader(componentCl);
        }

        Component bootComp = Fractal.getBootstrapComponent();
        GenericInstallingFactory gif = (GenericInstallingFactory) bootComp
                .getFcInterface("generic-installing-factory");
        if (gif != null) {
            Fractal.getBindingController(comp).bindFc("component-factory", gif);
        }

        Fractal.getLifeCycleController(comp).startFc();

        return comp;
    }
}
