/***
 * Fractal RMI: a binder for remote method calls between Fractal components.
 * Copyright (C) 2003-2006 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact: fractal@objectweb.org
 *
 * Author: Eric Bruneton
 *
 * Contributors: Bruno Dillenseger, Philippe Merle
 *
 * with some comments copied from Jonathan:
 *   org.objectweb.jonathan.apis.stub_factories.StubFactory (author: B. Dumant)
 *
 * Bruno Dillenseger added main method to generate stubs and skeletons (like a compiler).
 *
 * Philippe Merle added generation of stubs/skeletons for interfaces provided by
 * standard Java runtime librairies (e.g., java.lang.Runnable).
 *
 * $Id: MyRmiStubFactory.java 738 2008-02-14 15:16:59Z loris $
 */

package org.objectweb.fractal.rmi.stub;

import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.Interface;
import org.objectweb.fractal.api.control.BindingController;
import org.objectweb.fractal.deployment.local.api.GenericInstallingFactory;
import org.objectweb.fractal.deployment.local.api.Loader;
import org.objectweb.fractal.rmi.ClassGenerator;
import org.objectweb.fractal.util.Fractal;
import org.objectweb.jonathan.apis.binding.ExportException;
import org.objectweb.jonathan.apis.binding.Identifier;
import org.objectweb.jonathan.apis.kernel.Context;
import org.objectweb.jonathan.apis.kernel.JonathanException;
import org.objectweb.jonathan.apis.presentation.MarshallerFactory;
import org.objectweb.jonathan.apis.protocols.RequestSession;
import org.objectweb.jonathan.apis.protocols.SessionIdentifier;
import org.objectweb.jonathan.apis.stub_factories.StubFactory;
import org.objectweb.util.monolog.api.BasicLevel;
import org.objectweb.util.monolog.api.Logger;
import org.objectweb.util.monolog.api.LoggerFactory;

/**
 * A stub and skeleton factory based on <a
 * href="http://www.objectweb.org/asm">ASM</a>. This factory generates the stub
 * and skeleton classes dynamically, when they are needed. It is therefore not
 * necessary for the user to statically generate these classes with a tool such
 * as <tt>rmic</tt> or <tt>idlc</tt>. <br>
 * The stubs and skeletons created by this factory marshall and unmarshall the
 * method invocations by using the following format: a method invocation message
 * contains a four bytes method index, which uniquely and unambiguously
 * identifies a method among the methods provided by a Java interface, followed
 * by the method's arguments, marshalled in the order of their declaration in
 * the method's signature.
 * <p>
 * In order to be able to manually generate stubs and skeletons before execution
 * and avoid their generation at runtime (typically for performance reasons,
 * although no real testing has been performed), this class can be used as a
 * kind of a stub compiler through its main method, giving as arguments the Java
 * interfaces you want to compile stubs for. Note that this generation is
 * performed at byte code level only, i.e. it takes compiled interfaces as input
 * and generates compiled stubs and skeletons classes. In other words, the
 * interfaces (fully-qualified) names you specify must be compiled and reachable
 * in the classpath, and the result is a corresponding set of stub and skeleton
 * compiled classes (taking current directory as base directory).
 */

public class MyRmiStubFactory implements StubFactory, SkeletonFactory,
		BindingController {

	/**
	 * A loader interface of this component
	 */
	private ClassGenerator classGenerator = null;

	/**
	 * A generic installing factory interface of this component
	 */
	private GenericInstallingFactory gif = null;

	/**
	 * The marshaller factory to be used by the stubs created by this factory.
	 */

	protected MarshallerFactory marshallerFactory;

	/**
	 * The optional logger factory used to get a logger for this component.
	 */

	protected LoggerFactory loggerFactory;

	/**
	 * The logger used to log messages. May be <tt>null</tt>.
	 */

	protected Logger logger;

	/**
	 * Constructs a new {@link MyRmiStubFactory}.
	 */

	public MyRmiStubFactory() {
	}

	// --------------------------------------------------------------------------
	// Implementation of the BindingController interface
	// --------------------------------------------------------------------------

	public String[] listFc() {
		return new String[] { "marshaller-factory", "logger-factory",
				"class-generator", "component-factory" };
	}

	public Object lookupFc(final String clientItfName) {
		if (clientItfName.equals("marshaller-factory")) {
			return marshallerFactory;
		} else if (clientItfName.equals("logger-factory")) {
			return loggerFactory;
		} else if (clientItfName.equals("class-generator")) {
			return classGenerator;
		} else if (clientItfName.equals("component-factory")) {
			return gif;
		}
		return null;
	}

	public void bindFc(final String clientItfName, final Object serverItf) {
		if (clientItfName.equals("marshaller-factory")) {
			marshallerFactory = (MarshallerFactory) serverItf;
		} else if (clientItfName.equals("logger-factory")) {
			loggerFactory = (LoggerFactory) serverItf;
			logger = loggerFactory.getLogger(getClass().getName());
		} else if (clientItfName.equals("class-generator")) {
			classGenerator = (ClassGenerator) serverItf;
		} else if (clientItfName.equals("component-factory")) {
			gif = (GenericInstallingFactory) serverItf;
		}
	}

	public void unbindFc(final String clientItfName) {
		if (clientItfName.equals("marshaller-factory")) {
			marshallerFactory = null;
		} else if (clientItfName.equals("logger-factory")) {
			loggerFactory = null;
			logger = null;
		} else if (clientItfName.equals("class-generator")) {
			classGenerator = null;
		} else if (clientItfName.equals("component-factory")) {
			gif = null;
		}
	}

	// --------------------------------------------------------------------------
	// Implementation of the StubFactory interface
	// --------------------------------------------------------------------------

	/**
	 * Creates a new stub. A stub plays two roles:
	 * <ul>
	 * <li>It is the local representative of a (remote) object, and thus should
	 * bear a set of identifiers for that remote object;</li>
	 * <li>It is part of the binding between the client and the server, and is
	 * thus related to a given protocol. The session identifier provided as an
	 * argument is the manifestation of this relation. It can be used to obtain
	 * a session allowing the stub to send data to the remote object.</li>
	 * </ul>
	 * 
	 * @param sessionId
	 *            a session identifier, to be used to send marshalled data to
	 *            the object represented by the stub;
	 * @param ids
	 *            the set of identifiers of the stub;
	 * @param hints
	 *            other data possibly used to create the stub. This method
	 *            requires the fully qualified name of the Java interface to
	 *            which the stub will gives access. This name must be associated
	 *            to the "interface_type" key.
	 * @return an instance of a sub class of the {@link Stub} class.
	 * @throws JonathanException
	 *             if something goes wrong.
	 */

	public Object newStub(final SessionIdentifier sessionId,
			final Identifier[] ids, final Context hints)
			throws JonathanException {

		String implClassName = (String) hints.getValue("interface_type",
				(char) 0);

		Class implClass = null;
		Class stubClass = null;
		try {
			implClass = classGenerator.loadClass(implClassName);
			stubClass = classGenerator.generateClass(getStubClassName(implClass
					.getName()));
		} catch (ClassNotFoundException e1) {
			implClassName = "org.objectweb.fractal.api.Component";

			try {
				implClass = classGenerator.loadClass(implClassName);
			} catch (ClassNotFoundException e) {
				e1 = e;
				e1.printStackTrace();
			}

			stubClass = classGenerator.generateClass(getStubClassName(implClass
					.getName()));
		}

		try {
			if (ids.length != 1) {
				throw new JonathanException();
			}
			Stub stub = (Stub) stubClass.newInstance();
			stub.id = ids[0];
			stub.sessionId = sessionId;
			stub.marshallerFactory = marshallerFactory;
			if (logger != null && logger.isLoggable(BasicLevel.INFO)) {
				logger.log(BasicLevel.INFO, "Stub created for id " + ids[0]);
			}

			// TODO: remove
			// System.err.println("\n" + implClassName);
			// System.err.println(stub);
			ClassLoader ld = stubClass.getClassLoader();
			for (int i = 0; ld != null; i++) {
				// System.err.println(ld + " " + ld.getClass());
				ld = ld.getParent();
			}

			return stub;
		} catch (Exception e) {
			throw new JonathanException(e);
		}
	}

	// --------------------------------------------------------------------------
	// Implementation of the StubFactory interface
	// --------------------------------------------------------------------------

	public RequestSession newSkeleton(final Object target)
			throws JonathanException {
		try {
			if (!(target instanceof Interface)) {
				throw new Exception("target object must implement Interface");
			}
			Class implClass = target.getClass();
			Class[] itfs = implClass.getInterfaces();
			if (itfs.length == 0) {
				throw new Exception("target object must implement an interface");
			}

			ClassLoader cl = null;
			Class skelClass = null;
			Component owner = ((Interface) target).getFcItfOwner();

			try {
				if (gif != null) {
					cl = gif.getFcLoaderForComponent(owner).getClassLoader();
					if (cl != null) {
						skelClass = classGenerator.generateClass(
								getSkelClassName(itfs[0].getName()), cl);
					}
				}
			} catch (Exception e) {
				// ignored
			}

			// if (skelClass == null) {
			// Component boot = Fractal.getBootstrapComponent();
			// GenericInstallingFactory bootFact = (GenericInstallingFactory)
			// boot
			// .getFcInterface("generic-installing-factory");
			// Loader ldr = bootFact.getFcLoaderForComponent(owner);
			// if (ldr != null) {
			// cl = ldr.getClassLoader();
			// }
			// if (cl != null) {
			// skelClass = classGenerator.generateClass(
			// getSkelClassName(itfs[0].getName()), cl);
			// }
			// }

			if (skelClass == null) {
				skelClass = classGenerator.generateClass(
						getSkelClassName(itfs[0].getName()), this.getClass()
								.getClassLoader());
			}
			Skeleton skel = (Skeleton) skelClass.newInstance();
			skel.target = target;
			if (logger != null && logger.isLoggable(BasicLevel.INFO)) {
				logger.log(BasicLevel.INFO, "Skeleton created for target "
						+ target);
			}
			return skel;
		} catch (Exception e) {
			throw new ExportException(e);
		}
	}

	// TODO: remove
	/**
	 * Returns the name of the skeleton class for the given class.
	 * 
	 * @param className
	 *            a fully qualified class name.
	 * @return the name of the skeleton class for the given class.
	 */

	private static String getSkelClassName(final String className) {
		if (className.startsWith("java.")) {
			return "skel." + className + "_JavaSkel";
		} else {
			return className + "_Skel";
		}
	}

	/**
	 * Returns the name of the stub class for the given class.
	 * 
	 * @param className
	 *            a fully qualified class name.
	 * @return the name of the stub class for the given class.
	 */

	private static String getStubClassName(final String className) {
		if (className.startsWith("java.")) {
			return "stub." + className + "_JavaStub";
		} else {
			return className + "_Stub";
		}
	}
}
