/***
 * Fractal RMI: a binder for remote method calls between Fractal components.
 * Copyright (C) 2003 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact: Eric.Bruneton@rd.francetelecom.com
 *
 * Author: Eric Bruneton
 *
 * with some comments copied from Jonathan:
 *   org.objectweb.jonathan.apis.protocols.RequestSession (author: B. Dumant)
 */

package org.objectweb.fractal.rmi.stub;

import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.Interface;
import org.objectweb.fractal.api.Type;
import org.objectweb.jonathan.apis.kernel.JonathanException;
import org.objectweb.jonathan.apis.presentation.Marshaller;
import org.objectweb.jonathan.apis.presentation.UnMarshaller;
import org.objectweb.jonathan.apis.protocols.ReplySession;
import org.objectweb.jonathan.apis.protocols.RequestSession;

/**
 * Super class of the skeleton classes generated by the {@link RmiStubFactory}.
 * A skeleton gives access to a local interface of a Fractal
 * component. It unmarshalls invocation messages and calls the corresponding
 * methods on the local interface to which it gives access. Like stubs, a
 * skeleton gives access to the "functional" interface provided by a server
 * interface, and also gives access to the methods of the {@link Interface}
 * interface implemented by this server interface.
 */

public abstract class Skeleton implements RequestSession {

  /**
   * The local server interface to which this skeleton gives access.
   */

  protected Object target;

  /**
   * Constructs a new {@link Skeleton}.
   */

  public Skeleton () {
  }

  // --------------------------------------------------------------------------
  // Implementation of the RequestSession interface
  // --------------------------------------------------------------------------

  /**
   * Returns the target object represented by this request session.
   *
   * @return the target object represented by this request session.
   */

  public Object getTarget () {
    return target;
  }

  /**
   * Sends a message (a request) to its recipient. The unmarshaller representing
   * the invocation is sent together with a reply session, to be used by the
   * recipient to send the reply. The reply session may be null if no response
   * is expected. It is the responsibility of the recipient to make sure that
   * the unmarshaller will properly be {@link UnMarshaller#close() closed}.
   *
   * @param unmarshaller the unmarshaller representing the request.
   * @param session the session to send the reply.
   * @throws JonathanException if something goes wrong.
   */

  public abstract void send (UnMarshaller unmarshaller, ReplySession session)
    throws JonathanException;

  // --------------------------------------------------------------------------
  // Utility methods
  // --------------------------------------------------------------------------

  /**
   * Handles invocations messages corresponding to methods of the {@link
   * Interface} interface.
   *
   * @param unmarshaller the unmarshaller representing the request.
   * @param session the session to send the reply.
   * @param methodIndex the index of the method to be called.
   * @throws JonathanException if something goes wrong.
   */

  protected void handleInterfaceMethods (
    final UnMarshaller unmarshaller,
    final ReplySession session,
    final int methodIndex) throws JonathanException
  {
    Interface itf = (Interface)target;
    Marshaller marshaller;
    try {
      switch (methodIndex) {
        case -1:
          unmarshaller.close();
          Component id = itf.getFcItfOwner();
          marshaller = session.prepareReply();
          marshaller.writeValue(id);
          session.send(marshaller);
          session.close();
          return;
        case -2:
          unmarshaller.close();
          String name = itf.getFcItfName();
          marshaller = session.prepareReply();
          marshaller.writeValue(name);
          session.send(marshaller);
          session.close();
          return;
        case -3:
          unmarshaller.close();
          Type type = itf.getFcItfType();
          marshaller = session.prepareReply();
          marshaller.writeValue(type);
          session.send(marshaller);
          session.close();
          return;
        case -4:
          unmarshaller.close();
          boolean internal = itf.isFcInternalItf();
          marshaller = session.prepareReply();
          marshaller.writeBoolean(internal);
          session.send(marshaller);
          session.close();
          return;
        default:
          throw new Exception("No such method");
      }
    } catch (Exception e) {
      handleException(e, session);
    }
  }

  /**
   * Handles an exception that happened in this skeleton. This method sends the
   * given exception to the caller by using the given reply session.
   *
   * @param e the exception that happened.
   * @param session the session to send the reply.
   * @throws JonathanException if the given exception cannot be sent back to the
   *      caller.
   */

  protected void handleException (final Exception e, final ReplySession session)
    throws JonathanException
  {
    try {
      Marshaller marshaller = session.prepareExceptionReply();
      marshaller.writeValue(e);
      session.send(marshaller);
    } catch (Exception f) {
      f.printStackTrace();
      throw new JonathanException(
        "error during marshalling of exception by skeleton");
    }
    session.close();
  }
  
  protected Object replaceClassName (Object o) throws ClassNotFoundException {
    if (o instanceof String) {
      return Class.forName((String)o);
    } else if (o instanceof Object[]) {
      Object[] desc = (Object[])o;
      if (desc.length == 2 && desc[1] instanceof String) {
        return new Object[] { desc[0], Class.forName((String)desc[1]) };
      }
    }
    return o;
  }
  
  protected Object replaceClassValue (Object o) {
    if (o instanceof Class) {
      return ((Class)o).getName();
    } else if (o instanceof Object[]) {
      Object[] desc = (Object[])o;
      if (desc.length == 2 && desc[1] instanceof Class) {
        return new Object[] { desc[0], ((Class)desc[1]).getName() };
      }
    }
    return o;
  }
}
