/***
 * Fractal RMI: a binder for remote method calls between Fractal components.
 * Copyright (C) 2003 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact: Eric.Bruneton@rd.francetelecom.com
 *
 * Author: Eric Bruneton
 *
 * with some code fragments copied from Jonathan:
 *   org.objectweb.jeremie.libs.stub_factories.std.RefImpl
 *     (authors: B. Dumant, K. Milsted)
 */

package org.objectweb.fractal.rmi.stub;

import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.Interface;
import org.objectweb.fractal.api.Type;
import org.objectweb.fractal.rmi.RemoteException;
import org.objectweb.jonathan.apis.binding.Identifier;
import org.objectweb.jonathan.apis.binding.Reference;
import org.objectweb.jonathan.apis.kernel.JonathanException;
import org.objectweb.jonathan.apis.presentation.MarshalException;
import org.objectweb.jonathan.apis.presentation.Marshaller;
import org.objectweb.jonathan.apis.presentation.MarshallerFactory;
import org.objectweb.jonathan.apis.presentation.UnMarshaller;
import org.objectweb.jonathan.apis.protocols.ReplyInterface;
import org.objectweb.jonathan.apis.protocols.ServerException;
import org.objectweb.jonathan.apis.protocols.SessionIdentifier;
import org.objectweb.jonathan.apis.protocols.Session_High;

/**
 * Super class of the stub classes generated by the {@link RmiStubFactory}.
 * A stub gives access to an interface of a Fractal component.
 * More precisely, it gives access to the "functional" interface provided by
 * a server interface, and also gives access to the {@link Interface} interface
 * implemented by this server interface. The
 * "functional" interface is implemented by a sub class of this class, while
 * the {@link Interface} interface is implemented
 * by this class once and for all.
 */

public class Stub implements Reference, Interface {

  /**
   * The identifier of the remote object to which this stub gives access.
   */

  protected Identifier id;

  /**
   * The marshaller factory to be used to create new invocation messages.
   */

  protected MarshallerFactory marshallerFactory;

  /**
   * The identifier of the session that gives access to the remote object.
   */

  protected SessionIdentifier sessionId;

  /**
   * The session that gives access to the remote object. This session is used to
   * send the invocation messages down the protocol stack.
   */

  protected Session_High session;

  /**
   * Constructs a new {@link Stub}.
   */

  public Stub () {
  }

  // --------------------------------------------------------------------------
  // Implementation of the Reference interface
  // --------------------------------------------------------------------------

  /**
   * Returns the set of identifiers associated with this reference.
   *
   * @return the set of identifiers associated with this reference.
   * @see #setIdentifiers
   */

  public Identifier[] getIdentifiers () {
    return new Identifier[] { id };
  }

  /**
   * Sets the set of identifiers associated with this reference.
   *
   * @param ids the set of identifiers to be associated with this reference.
   * @see #getIdentifiers
   */

  public void setIdentifiers (Identifier[] ids) {
    this.id = ids[0];
  }

  // --------------------------------------------------------------------------
  // Implementation of the Interface interface
  // --------------------------------------------------------------------------

  public Component getFcItfOwner () {
    try {
      Marshaller marshaller = request();
      ReplyInterface reply = prepareInvocation(marshaller);
      marshaller.writeInt(-1);
      invoke(marshaller);
      UnMarshaller unmarshaller = reply.listen();
      Component result = (Component)unmarshaller.readValue();
      unmarshaller.close();
      return result;
    } catch (Exception e) {
      e = handleException(e);
      if (e instanceof RuntimeException) {
        throw (RuntimeException)e;
      } else {
        throw new RemoteException("server side exception", e);
      }
    }
  }

  public String getFcItfName () {
    try {
      Marshaller marshaller = request();
      ReplyInterface reply = prepareInvocation(marshaller);
      marshaller.writeInt(-2);
      invoke(marshaller);
      UnMarshaller unmarshaller = reply.listen();
      String result = (String)unmarshaller.readValue();
      unmarshaller.close();
      return result;
    } catch (Exception e) {
      e = handleException(e);
      if (e instanceof RuntimeException) {
        throw (RuntimeException)e;
      } else {
        throw new RemoteException("server side exception", e);
      }
    }
  }

  public Type getFcItfType () {
    try {
      Marshaller marshaller = request();
      ReplyInterface reply = prepareInvocation(marshaller);
      marshaller.writeInt(-3);
      invoke(marshaller);
      UnMarshaller unmarshaller = reply.listen();
      Type result = (Type)unmarshaller.readValue();
      unmarshaller.close();
      return result;
    } catch (Exception e) {
      e = handleException(e);
      if (e instanceof RuntimeException) {
        throw (RuntimeException)e;
      } else {
        throw new RemoteException("server side exception", e);
      }
    }
  }

  public boolean isFcInternalItf () {
    try {
      Marshaller marshaller = request();
      ReplyInterface reply = prepareInvocation(marshaller);
      marshaller.writeInt(-4);
      invoke(marshaller);
      UnMarshaller unmarshaller = reply.listen();
      boolean result = unmarshaller.readBoolean();
      unmarshaller.close();
      return result;
    } catch (Exception e) {
      e = handleException(e);
      if (e instanceof RuntimeException) {
        throw (RuntimeException)e;
      } else {
        throw new RemoteException("server side exception", e);
      }
    }
  }

  // --------------------------------------------------------------------------
  // Utility methods
  // --------------------------------------------------------------------------

  /**
   * Obtains a marshaller for marshalling invocation data.
   *
   * @return a marshaller for marshalling invocation data.
   * @throws MarshalException if something goes wrong.
   */

  protected Marshaller request () throws MarshalException {
    if (marshallerFactory == null) {
      throw new MarshalException("null marshaller factory");
    }

    return marshallerFactory.newMarshaller();
  }

  /**
   * Prepares the marshaller prior to marshalling invocation data. This
   * preparation is protocol-specific and typically consists of adding headers
   * to the marshaller. In line with Java method invocation semantics, a reply
   * is expected after the remote invocation is executed; a reply interface
   * is thus returned in order to listen for the reply.
   *
   * @param marshaller a marshaller to be used for the invocation data.
   * @return a reply interface for listening for the reply.
   * @throws MarshalException if something goes wrong.
   */

  protected ReplyInterface prepareInvocation (final Marshaller marshaller)
    throws MarshalException
  {
    try {
      synchronized(this) {
        if (session == null) {
          session = sessionId.bind(null);
        }
      }
      return session.prepareInvocation(marshaller);
    } catch (Exception e) {
      throw new MarshalException("exception preparing marshaller: " + e);
    }
  }

  /**
   * Sends the marshaller with the invocation data down the protocol stack.
   *
   * @param marshaller a marshaller with the invocation data.
   * @throws MarshalException if something goes wrong.
   */

  protected void invoke (final Marshaller marshaller) throws MarshalException {
    try {
      synchronized(this) {
        if (session == null) {
          throw new MarshalException("null session");
        }
      }
      session.send(marshaller);
    } catch (Exception e) {
      throw new MarshalException("exception at invocation: "  + e);
    }
  }

  /**
   * Handles an exception that happen in this stub.
   *
   * @param e an exception.
   * @return the exception encapsulated in the given exception (if it is a
   *      {@link ServerException}, i.e., an exception that was
   *      thrown by the remote object and sent back in a reply message), or a
   *      new {@link RemoteException} encaspulating the given
   *      exception in the other cases (case of an exception that was thrown
   *      locally by the stub itself).
   */

  protected Exception handleException (final Exception e) {
    if (e instanceof ServerException) {
      try {
        UnMarshaller unmarshaller = ((ServerException)e).unmarshaller;
        Exception f = (Exception)unmarshaller.readValue();
        unmarshaller.close();
        return f;
      } catch (JonathanException je) {
        return new RemoteException(
          "error during exception unmarshalling by stub", je);
      }
    } else {
      return new RemoteException(
        "error during marshalling/unmarshalling by stub", e);
    }
  }

  protected Object replaceClassName (Object o) throws ClassNotFoundException {
    if (o instanceof String) {
      // return Class.forName((String)o);
    	return o;
    } else if (o instanceof Object[]) {
      Object[] desc = (Object[])o;
      if (desc.length == 2 && desc[1] instanceof String) {
        // return new Object[] { desc[0], Class.forName((String)desc[1]) };
    	return new Object[] { desc[0], desc[1] };
      }
    }
    return o;
  }
  
  protected Object replaceClassValue (Object o) {
    if (o instanceof Class) {
      return ((Class)o).getName();
    } else if (o instanceof Object[]) {
      Object[] desc = (Object[])o;
      if (desc.length == 2 && desc[1] instanceof Class) {
        return new Object[] { desc[0], ((Class)desc[1]).getName() };
      }
    }
    return o;
  }

  // --------------------------------------------------------------------------
  // Overriden methods
  // --------------------------------------------------------------------------

  /**
   * Tests if the given object is equal to this stub.
   *
   * @param o the object to be compared to this stub.
   * @return <tt>true</tt> if the given object is a stub whose {@link #id id} is
   *      equal to the {@link #id id} of this stub, <tt>false</tt> otherwise.
   */

  public boolean equals (final Object o) {
    if (o instanceof Stub) {
      return id.equals(((Stub)o).id);
    }
    return false;
  }

  /**
   * Returns the hashcode of this stub.
   *
   * @return the hashcode of the {@link #id id} of this stub.
   */

  public int hashCode () {
    return id.hashCode();
  }

  /**
   * Finalizes this object. This method closes the {@link #session session}
   * associated to this stub if it is not <tt>null</tt>.
   */

  protected void finalize () {
    if (session != null) {
      session.close();
      session = null;
    }
  }
}
