/*==============================================================================
 Fraclet annotation - Copyright (C) 2002-2006 INRIA Futurs / LIFL
 Fractal Component Model (contact: fractal@objectweb.org)

 This library is free software; you can redistribute it and/or modify it under 
 the terms of the GNU Lesser General Public License as published by the Free 
 Software Foundation; either version 2.1 of the License, or any later version.

 This library is distributed in the hope that it will be useful, but WITHOUT ANY 
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public License along 
 with this library; if not, write to the Free Software Foundation, Inc., 
 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

 Initial developer(s): Nicolas Pessemier (nicolas.pessemier@lifl.fr)
 ==============================================================================*/

package org.ow2.jasmine.jade.fractal.fraclet.annotation.generator;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.objectweb.fractal.fraclet.annotation.Attribute;
import org.objectweb.fractal.fraclet.annotation.FractalComponent;
import org.objectweb.fractal.fraclet.annotation.Interface;
import org.objectweb.fractal.fraclet.annotation.Provides;
import org.objectweb.fractal.fraclet.annotation.Requires;
import org.objectweb.fractal.fraclet.annotation.generator.template.AttributeTemplate;
import org.objectweb.fractal.fraclet.annotation.generator.template.util.None;
import org.objectweb.fractal.fraclet.annotation.processor.util.ADLFileFactory;

import org.ow2.jasmine.jade.fractal.fraclet.annotation.GenericAttribute;

import spoon.reflect.declaration.CtClass;
import spoon.reflect.declaration.CtField;
import spoon.reflect.reference.CtTypeReference;

/**
 * An ADL generator for the component definitions.
 * 
 * @author Nicolas Pessemier <Nicolas.Pessemier@lifl.fr>
 * <p>
 * contributor : <a href="mailto:julien.legrand@inrialpes.fr">Julien Legrand</a>
 * 
 */
public class ComponentADLGenerator {

    private PrintWriter pw = null;

    private CtClass<?> processedClass;

    public ComponentADLGenerator(CtClass<?> processedClass,
            FractalComponent fcAnnotation) {
        this.processedClass = processedClass;
        // -----------------------------------------------------
        // create .fractal file
        // -----------------------------------------------------
        File outFile = ADLFileFactory.createADLFile(processedClass.getFactory()
                .getEnvironment().getDefaultFileGenerator()
                .getOutputDirectory(), processedClass);
        try {
            pw = new PrintWriter(outFile);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }
        // -----------------------------------------------------
        // PrintSignature
        // -----------------------------------------------------

        ADLFileFactory.printLicence(pw);

        // -----------------------------------------------------
        // Manage Fractal attributes arguments
        // -----------------------------------------------------
        String arguments = "";
        List<CtField<?>> fields = processedClass.getFields();
        Map<String, String> attributes = new HashMap<String, String>();

        /*
         * iterate over processedClass fields
         */
        for (CtField<?> field : fields) {

            /*
             * get the Attributes annotation
             */
            Attribute acAnnotation = field.getAnnotation(Attribute.class);

            /*
             * if the field is not annotated continue
             */
            if (acAnnotation == null)
                continue;

            if (acAnnotation.value() == null)
                attributes.put(field.getSimpleName(), "");
            else
                attributes.put(field.getSimpleName(), acAnnotation.value());
            if (acAnnotation.argument() != null
                    && acAnnotation.argument().length() > 0) {
                arguments += acAnnotation.argument() + ",";
                String argValue = "${" + acAnnotation.argument() + "}";
                attributes.put(field.getSimpleName(), argValue);
            }
        }
        if (arguments != "")
            arguments = arguments.substring(0, (arguments.length() - 1));

        // -----------------------------------------------------
        // Manage Fractal attributes arguments
        // -----------------------------------------------------
        Map<String, String> genericAttributes = new HashMap<String, String>();

        /*
         * iterate over processedClass fields
         */
        for (CtField<?> field : fields) {

            /*
             * get the @GenericAttribute annotation
             */
            GenericAttribute acAnnotation = field
                    .getAnnotation(GenericAttribute.class);

            /*
             * if the field is not annotated continue
             */
            if (acAnnotation == null)
                continue;

            if (acAnnotation.value() == null) {
                genericAttributes.put(field.getSimpleName(), "");
            } else {
                genericAttributes.put(field.getSimpleName(), acAnnotation.value());
            }

            if (acAnnotation.argument() != null
                    && acAnnotation.argument().length() > 0) {
                
                arguments += acAnnotation.argument() + ",";
                String argValue = "${" + acAnnotation.argument() + "}";
                genericAttributes.put(field.getSimpleName(), argValue);
            }
        }
        if (arguments != "")
            arguments = arguments.substring(0, (arguments.length() - 1));

        // -----------------------------------------------------
        // Get Fractal server interfaces
        // -----------------------------------------------------

        List<CtTypeReference<?>> itfs = new ArrayList<CtTypeReference<?>>();
        for (CtTypeReference<?> itf : processedClass.getSuperInterfaces()) {
            Interface itfAnnotation = itf.getAnnotation(Interface.class);
            if (itfAnnotation != null) {
                itfs.add(itf);
            }
        }
        CtTypeReference<?> superClass = processedClass.getSuperclass();
        if (superClass != null)
            if (superClass.getAnnotation(FractalComponent.class) != null)
                itfs.add(superClass);

        printBeginDefinition(itfs, arguments);

        // -----------------------------------------------------
        // Get Fractal client interfaces
        // -----------------------------------------------------

        // iterate over processedClass fields
        for (CtField<?> field : fields) {
            // get the Requires annotation
            Requires bindingAnnotation = field.getAnnotation(Requires.class);
            // if the field is not annotated continue
            if (bindingAnnotation == null)
                continue;
            Class<?> c = bindingAnnotation.signature();
            if (c.equals(Class.class))
                c = None.class;
            ADLFileFactory
                    .printItfSignature(pw, field, bindingAnnotation.name(), c,
                            bindingAnnotation.cardinality(), bindingAnnotation
                                    .contingency(), ADLFileFactory.ROLE_CLIENT);
        }
        // -----------------------------------------------------
        // Manage the local @Interface server itf annotations
        // -----------------------------------------------------

        Provides providesAnnotation = processedClass
                .getAnnotation(Provides.class);
        if (providesAnnotation != null) {
            Interface[] itfAnnotations = providesAnnotation.interfaces();
            for (Interface itfAnnotation : itfAnnotations) {
                ADLFileFactory.printItfSignature(pw, itfAnnotation.signature()
                        .getCanonicalName(), itfAnnotation.name(),
                        itfAnnotation.cardinality(), itfAnnotation
                                .contingency(), ADLFileFactory.ROLE_SERVER);
            }
        }

        // -----------------------------------------------------
        // Manage attribute
        // -----------------------------------------------------
        String attributeControllerSignature = processedClass.getQualifiedName()
                + AttributeTemplate.ATTRIBUTESUFFIX;

        printContentClass();
        printAttributes(attributeControllerSignature, attributes);

        // -----------------------------------------------------
        // Manage generic-attribute
        // -----------------------------------------------------
        String genericAttributeControllerSignature = "org.ow2.jasmine.jade.fractal.api.control.GenericAttributeController";

        printContentClass();
        printGenericAttributes(genericAttributeControllerSignature, genericAttributes);

        // -----------------------------------------------------
        // Print Controller Desc
        // -----------------------------------------------------

        pw.println("  <controller desc=\"" + fcAnnotation.controllerDesc()
                + "\"/>");

        // -----------------------------------------------------
        // closing file
        // -----------------------------------------------------

        printEndDefinition();
        pw.close();
    }

    private void printContentClass() {
        pw.println("  <content class=\"" + processedClass.getQualifiedName()
                + "\"/>");
    }

    private void printAttributes(String attributeControllerSignature,
            Map<String, String> attributes) {

        // <attributes signature="primitive.ServerAAttributeController">
        // <attribute value="&gt;&gt; " name="header"/>
        // <attribute value="${msg}" name="message"/>
        // </attributes>
        if (attributes.size() > 0) {
            pw.println("  <attributes signature=\""
                    + attributeControllerSignature + "\">");
            for (String attributeName : attributes.keySet()) {
                pw.println("    <attribute name=\"" + attributeName + "\""
                        + " value=\"" + "" + attributes.get(attributeName)
                        + "\"/>");
            }
            pw.println("  </attributes>");
        }
    }

    private void printGenericAttributes(
            String genericAttributeControllerSignature,
            Map<String, String> attributes) {

        if (attributes.size() > 0) {

            pw.println("  <attributes signature=\""
                    + genericAttributeControllerSignature + "\">");

            for (String attributeName : attributes.keySet()) {
                pw.println("    <attribute name=\"" + attributeName + "\""
                        + " value=\"" + "" + attributes.get(attributeName)
                        + "\"/>");
            }

            pw.println("  </attributes>");
        }
    }

    private void printBeginDefinition(List<CtTypeReference<?>> itfs,
            String arguments) {
        /* @Todo manage the arguments= option */
        if (arguments != "")
            arguments = "arguments=\"" + arguments + "\"";
        if (itfs.size() > 0) {
            String extendedADLFiles = "";
            for (CtTypeReference<?> itf : itfs) {
                extendedADLFiles += itf + ",";

            }
            extendedADLFiles = extendedADLFiles.substring(0, (extendedADLFiles
                    .length() - 1));
            pw.println("<definition name=\""
                    + processedClass.getQualifiedName() + "\" extends=\""
                    + extendedADLFiles + "\"" + " " + arguments + ">");
        } else
            pw.println("<definition name=\""
                    + processedClass.getQualifiedName() + "\"" + " "
                    + arguments + ">");
    }

    private void printEndDefinition() {
        pw.println("</definition>");
    }
}
