/***
 * Jade-Fractal
 *
 * Copyright (C) 2007 : INRIA - Domaine de Voluceau, Rocquencourt, B.P. 105, 
 * 78153 Le Chesnay Cedex - France 
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact: jade <AT> inrialpes <DOT> fr
 *
 * Author: SARDES project - http://sardes.inrialpes.fr
 *
 */
 
package org.ow2.jasmine.jade.fractal.julia.control.binding;

import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.Interface;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.objectweb.fractal.api.control.BindingController;
import org.objectweb.fractal.api.control.ContentController;
import org.objectweb.fractal.api.control.IllegalBindingException;
import org.objectweb.fractal.api.control.IllegalLifeCycleException;
import org.objectweb.fractal.api.control.LifeCycleController;
import org.objectweb.fractal.api.type.ComponentType;
import org.objectweb.fractal.api.type.InterfaceType;
import org.objectweb.fractal.julia.ChainedNoSuchInterfaceException;
import org.objectweb.fractal.util.Fractal;

import org.ow2.jasmine.jade.fractal.api.control.ReverseBindingController;
import org.ow2.jasmine.jade.fractal.util.FractalUtil;

/**
 * Provides an implementation of the
 * {@link org.objectweb.fractal.api.control.BindingController} interface which
 * manages the reverse bindings : any binding operation performed on the current
 * client component is notified to the reverse binding controller of the
 * relevant server component.
 * 
 * @author <a href="mailto:fabienne.boyer@inrialpes.fr">Fabienne Boyer</a></br>
 * contributor : <a href="mailto:julien.legrand@inrialpes.fr">Julien Legrand</a>
 */

public abstract class BindingWithReverseBindingControllerNotifierMixin
		implements BindingController {

	// -------------------------------------------------------------------------
	// Private constructor
	// -------------------------------------------------------------------------

	private BindingWithReverseBindingControllerNotifierMixin() {
	}

	// -------------------------------------------------------------------------
	// Fields and methods added and overriden by the mixin class
	// -------------------------------------------------------------------------

	/**
	 * Calls the overriden bindFc method and then register the reverse binding.
	 * 
	 * @param clientItfName
	 *            the name of a client interface of the component to which this
	 *            interface belongs.
	 * @param serverItf
	 *            a server interface.
	 * @throws NoSuchInterfaceException
	 *             if there is no such client interface.
	 * @throws IllegalBindingException
	 *             if the binding cannot be created.
	 * @throws IllegalLifeCycleException
	 *             if this component has a {@link LifeCycleController}
	 *             interface, but it is not in an appropriate state to perform
	 *             this operation.
	 */

	public void bindFc(String clientItfName, Object serverItf)
			throws NoSuchInterfaceException, IllegalBindingException,
			IllegalLifeCycleException {

		/*
		 * calls the overriden bindFc method
		 */
		_super_bindFc(clientItfName, serverItf);

		/*
		 * get the server component
		 */
		Component server = ((Interface) serverItf).getFcItfOwner();

		/*
		 * get the reverse-binding-controller interface of the server component
		 */
		ReverseBindingController srbc = null;
		try {
			srbc = FractalUtil.getReverseBindingController(server);
		} catch (NoSuchInterfaceException e) {
			return;
		}

		/*
		 * get the client interface type
		 */
		ComponentType cType = (ComponentType) _this_weaveableC.getFcType();
		InterfaceType cItfType;
		try {
			cItfType = cType.getFcInterfaceType(clientItfName);
		} catch (NoSuchInterfaceException e) {
			throw new ChainedNoSuchInterfaceException(null, _this_weaveableC,
					clientItfName);
		}

		/*
		 * get the content controller interface of the client component
		 */
		ContentController cc;
		try {
			cc = Fractal.getContentController(_this_weaveableC);
		} catch (NoSuchInterfaceException e) {
			cc = null;
		}

		/*
		 * register the reverse binding
		 */
		if (srbc != null) {
			if (cItfType.isFcClientItf()) {
				srbc.bindSource(clientItfName, _this_weaveableC
						.getFcInterface(clientItfName));
			} else {
				// the client interface is an internal interface
				srbc.bindSource(clientItfName, cc
						.getFcInternalInterface(clientItfName));
			}
		}
	}

	/**
	 * Calls the overriden method and then unregister the reverse binding.
	 * 
	 * @param clientItfName
	 *            the name of a client interface of the component to which this
	 *            interface belongs.
	 * @throws NoSuchInterfaceException
	 *             if there is no such client interface.
	 * @throws IllegalBindingException
	 *             if the binding cannot be removed.
	 * @throws IllegalLifeCycleException
	 *             if this component has a {@link LifeCycleController}
	 *             interface, but it is not in an appropriate state to perform
	 *             this operation.
	 */
	public void unbindFc(String clientItfName) throws NoSuchInterfaceException,
			IllegalBindingException, IllegalLifeCycleException {

		ReverseBindingController srbc = null;

		/*
		 * get the server component bound to the client interface
		 */
		Interface sItf = (Interface) (_this_lookupFc(clientItfName));
		Component scmp = sItf.getFcItfOwner();

		/*
		 * get the reverse-binding-controller interface of the server component
		 */
		try {
			srbc = FractalUtil.getReverseBindingController(scmp);
		} catch (NoSuchInterfaceException e) {
			return;
		}

		/*
		 * get the content controller interface of the current component
		 */
		ContentController cc;
		try {
			cc = Fractal.getContentController(_this_weaveableC);
		} catch (NoSuchInterfaceException e) {
			cc = null;
		}

		/*
		 * get the client interface type
		 */
		ComponentType cType = (ComponentType) _this_weaveableC.getFcType();
		InterfaceType cItfType;
		try {
			cItfType = cType.getFcInterfaceType(clientItfName);
		} catch (NoSuchInterfaceException e) {
			throw new ChainedNoSuchInterfaceException(null, _this_weaveableC,
					clientItfName);
		}

		/*
		 * unregister the reverse binding
		 */
		if (srbc != null) {
			if (cItfType.isFcClientItf()) {
				srbc.unbindSource(clientItfName, _this_weaveableC
						.getFcInterface(clientItfName));
			} else {
				// the client interface is internal
				srbc.unbindSource(clientItfName, cc
						.getFcInternalInterface(clientItfName));
			}
		}

		/*
		 * calls the overriden unbindFc method
		 */
		_super_unbindFc(clientItfName);
	}

	// -------------------------------------------------------------------------
	// Fields and methods required by the mixin class in the base class
	// -------------------------------------------------------------------------

	/**
	 * The <tt>weaveableC</tt> field required by this mixin. This field is
	 * supposed to reference the {@link Component} interface of the component to
	 * which this controller object belongs.
	 */

	public Component _this_weaveableC;

	/**
	 * The {@link BindingController#bindFc bindFc} method overriden by this
	 * mixin.
	 * 
	 * @param clientItfName
	 *            the name of a client interface of the component to which this
	 *            interface belongs.
	 * @param serverItf
	 *            a server interface.
	 * @throws NoSuchInterfaceException
	 *             if there is no such client interface.
	 * @throws IllegalBindingException
	 *             if the binding cannot be created.
	 * @throws IllegalLifeCycleException
	 *             if this component has a {@link LifeCycleController}
	 *             interface, but it is not in an appropriate state to perform
	 *             this operation.
	 */

	public abstract void _super_bindFc(String clientItfName, Object serverItf)
			throws NoSuchInterfaceException, IllegalBindingException,
			IllegalLifeCycleException;

	/**
	 * The {@link BindingController#unbindFc unbindFc} method overriden by this
	 * mixin.
	 * 
	 * @param clientItfName
	 *            the name of a client interface of the component to which this
	 *            interface belongs.
	 * @throws NoSuchInterfaceException
	 *             if there is no such client interface.
	 * @throws IllegalBindingException
	 *             if the binding cannot be removed.
	 * @throws IllegalLifeCycleException
	 *             if this component has a {@link LifeCycleController}
	 *             interface, but it is not in an appropriate state to perform
	 *             this operation.
	 */

	public abstract void _super_unbindFc(String clientItfName)
			throws NoSuchInterfaceException, IllegalBindingException,
			IllegalLifeCycleException;

	/**
	 * The {@link BindingController#listFc listFc} method required by this
	 * mixin.
	 * 
	 * @param clientItfName
	 *            the name of a client interface of the component to which this
	 *            interface belongs.
	 * @return the server interface to which the given interface is bound, or
	 *         <tt>
	 *      null</tt> if it is not bound.
	 * @throws NoSuchInterfaceException
	 *             if the component to which this interface belongs does not
	 *             have a client interface whose name is equal to the given
	 *             name.
	 */

	public abstract Object _this_lookupFc(String clientItfName)
			throws NoSuchInterfaceException;

}
