/***
 * Jade-Fractal
 *
 * Copyright (C) 2007 : INRIA - Domaine de Voluceau, Rocquencourt, B.P. 105, 
 * 78153 Le Chesnay Cedex - France 
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact: jade <AT> inrialpes <DOT> fr
 *
 * Author: SARDES project - http://sardes.inrialpes.fr
 *
 */
 
package org.ow2.jasmine.jade.fractal.julia.control.genericattribute;

import java.util.HashMap;
import java.util.Set;

import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.objectweb.fractal.api.control.ContentController;
import org.objectweb.fractal.api.control.NameController;
import org.objectweb.fractal.api.factory.InstantiationException;
import org.objectweb.fractal.julia.Controller;
import org.objectweb.fractal.julia.InitializationContext;

import org.ow2.jasmine.jade.fractal.api.control.GenericAttributeController;
import org.ow2.jasmine.jade.fractal.api.control.NoSuchAttributeException;

/**
 * Provide a basic implementation of the {@link GenericAttributeController}
 * interface. This mixin is used for composite component. Attributes are stored
 * in an hashtable
 * 
 * @author <a href="mailto:noel.depalma@inrialpes.fr">Noel de palma</a>
 * @author <a href="mailto:julien.legrand@inrialpes.fr">Julien Legrand</a>
 * 
 */
public abstract class GenericCompositeAttributeControllerMixin implements
		Controller, GenericAttributeController {

	/**
	 * The attributes.
	 */
	HashMap<String, String> attributes = null;

	// ------------------------------------------------------------------------
	// Private Constructor
	// ------------------------------------------------------------------------
	private GenericCompositeAttributeControllerMixin() {
	}

	// -------------------------------------------------------------------------
	// Fields and methods added and overriden by the mixin class
	// -------------------------------------------------------------------------
	public void initFcController(final InitializationContext ic)
			throws InstantiationException {

		_super_initFcController(ic);
	}

	// ------------------------------------------------------------------------
	// Implementation of GenericAttributeController interface
	// ------------------------------------------------------------------------

	/**
	 * retrieve an attribute identified by its name.
	 * 
	 * @param name
	 *            the name of the attribute to read.
	 * @return the String value of the attribute identified by its name. null if
	 *         the implementation does not implements
	 *         GenericAttributeController.
	 * @throws NoSuchAttributeException
	 *             if the given attribute is unknow.
	 */
	public String getAttribute(String name) throws NoSuchAttributeException {
		try {
			if (attributes == null)
				throw new NoSuchAttributeException("The attribute " + name
						+ " doesn't exists");

			return (String) (attributes.get(name));

		} catch (Exception e) {
			throw new NoSuchAttributeException(e.getMessage());
		}
	}

	/**
	 * set the value of an attribute identified by its name and notify the
	 * update to the metalevel (resp. the exec level).
	 * 
	 * @param name
	 *            the name of the attribute to read.
	 * @param value
	 *            the value of the attribute.
	 * @throws NoSuchAttributeException
	 *             if the given attribute is unknow.
	 */
	public void setAttribute(String name, String value)
			throws NoSuchAttributeException {

		try {
			if (!forwardSetAttribute(name, value)) {
				if (attributes == null)
					attributes = new HashMap<String, String>(20, 20);
				if (value != null)
					attributes.put(name, value);
			}
		} catch (Exception e) {
			throw new NoSuchAttributeException(e.getMessage());
		}
	}

	/**
	 * Returns the names of the attribute of the component to which this
	 * interface belongs.
	 * 
	 * @return the names of the attributes of the component to which this
	 *         interface belongs. return null if there is no attributes.
	 */

	public String[] listFcAtt() {
		if (attributes != null) {
			Set<String> set = attributes.keySet();
			if (set != null) {
				return (String[]) (set.toArray(new String[set.size()]));
			} else
				return null;
		} else
			return null;
	}

	/**
	 * Set the value of an attribute identified by its name. This method is used
	 * by the meta level (resp. the exec level) to notify an attribute update.
	 * 
	 * @param name
	 *            the name of the attribute to read.
	 * @param value
	 *            the value of the attribute.
	 * @throws NoSuchAttributeException
	 *             if the given attribute is unknow.
	 */
	public void setAttributeNotification(String name, String value)
			throws NoSuchAttributeException {
		try {
			if (!forwardSetAttribute(name, value)) {
				if (attributes == null)
					attributes = new HashMap<String, String>(20, 20);
				if (value != null)
					attributes.put(name, value);
			}
		} catch (Exception e) {
			throw new NoSuchAttributeException(e.getMessage());
		}
	}

	// -------------------------------------------------------------------------
	// Private methods
	// -------------------------------------------------------------------------

	/**
	 * Forward an attribute update to subcmp.
	 * 
	 * @param name
	 *            /path/attname. path is the cmp path to find the subcmp.
	 *            attname is the attribute name to be setted.
	 * 
	 * for example : if name="/deploymentService/applicationLauncher/adlFile" it
	 * will find the component deploymentService/applicationLauncher from this
	 * component. If found, it will set the adlFile attribute on
	 * applicationLauncher.
	 */
	private boolean forwardSetAttribute(String name, String value)
			throws NoSuchAttributeException {
		if (name.charAt(0) != '/')
			return false;
		// else we have to forward the attribute
		int nextToken = name.indexOf('/', 1);
		if (nextToken == -1 || nextToken == 1)
			throw new NoSuchAttributeException("Attribute Name : " + name
					+ " : Wrong format");

		String cmpname = name.substring(1, nextToken);// 1..nextToken-1
		String newattr = name.substring(nextToken);
		if ("/".equals(newattr))
			throw new NoSuchAttributeException("Attribute Name : " + name
					+ " : Wrong format");

		int primToken = newattr.indexOf('/', 1); // stop forwarding ?
		Component subcmp = searchSubCmp(cmpname);
		if (subcmp == null)
			throw new NoSuchAttributeException("Attribute Name : " + name
					+ " : subcmp " + cmpname + " not found");

		if (primToken == -1) {
			// trunc the attribute name for the next cmp (should be a primitive)
			newattr = newattr.substring(1);

		}

		try {
			((GenericAttributeController) (subcmp
					.getFcInterface("attribute-controller"))).setAttribute(
					newattr, value);
		} catch (Exception e) {
			throw new NoSuchAttributeException("Attribute Name : " + name
					+ " : cannot forward to subcmp " + cmpname);
		}
		// Logger.println("FORWARD SET ATTRIBUTE ON : "+cmpname+":"+newattr);
		return true;
	}

	/*
	 * search is this cmp have a subcmp with cmpname as name. return null, if
	 * there is no subcmp with this name.
	 */
	private Component searchSubCmp(String cmpname) {
		Component[] subcmp;
		String curName = null;

		subcmp = _this_weaveableCC.getFcSubComponents();
		for (int i = 0; i < subcmp.length; i++) {

			try {
				curName = ((NameController) (subcmp[i]
						.getFcInterface("name-controller"))).getFcName();
				if (cmpname.equals(curName))
					return subcmp[i];
			} catch (NoSuchInterfaceException e0) {
				// no name ctrler for this subcmp, ignore it
			}
		}

		return null;
	}

	// -------------------------------------------------------------------------
	// Fields and methods required by the mixin class in the base class
	// -------------------------------------------------------------------------

	/**
	 * 
	 */
	public ContentController _this_weaveableCC;

	public abstract void _super_initFcController(InitializationContext ic)
			throws InstantiationException;

}
