/***
 * Julia: France Telecom's implementation of the Fractal API
 * Copyright (C) 2001-2002 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact: Eric.Bruneton@rd.francetelecom.com
 *
 * Author: Eric Bruneton
 */

package org.ow2.jasmine.jade.fractal.julia.factory;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.objectweb.fractal.api.Type;
import org.objectweb.fractal.api.control.BindingController;
import org.objectweb.fractal.api.control.ContentController;
import org.objectweb.fractal.api.control.IllegalBindingException;
import org.objectweb.fractal.api.control.IllegalLifeCycleException;
import org.objectweb.fractal.api.factory.GenericFactory;
import org.objectweb.fractal.api.factory.InstantiationException;
import org.objectweb.fractal.api.type.ComponentType;
import org.objectweb.fractal.api.type.InterfaceType;
import org.objectweb.fractal.api.type.TypeFactory;
import org.objectweb.fractal.deployment.local.api.GarbageCollector;
import org.objectweb.fractal.deployment.local.api.GenericInstallingFactory;
import org.objectweb.fractal.deployment.local.api.Loader;
import org.objectweb.fractal.deployment.local.api.LocallyInstalledPackage;
import org.objectweb.fractal.deployment.local.api.OSGiInstaller;
import org.objectweb.fractal.deployment.local.api.PackageDescription;
import org.objectweb.fractal.julia.Controller;
import org.objectweb.fractal.julia.InitializationContext;
import org.objectweb.fractal.julia.factory.ChainedInstantiationException;
import org.objectweb.fractal.julia.loader.Tree;
import org.objectweb.fractal.rmi.registry.NamingService;
import org.objectweb.fractal.util.Fractal;

import org.ow2.jasmine.jade.fractal.api.control.DeploymentStateController;
import org.ow2.jasmine.jade.fractal.api.control.GenericInstallingFactoryContextController;
import org.ow2.jasmine.jade.fractal.api.control.GenericInstallingFactoryController;
import org.ow2.jasmine.jade.fractal.julia.loader.ClassLoaderWrapper;

/**
 * Provides a basic implementation of the {@link GenericFactory} interface. <br>
 * <br>
 * <b>Requirements</b>
 * <ul>
 * <li>the types used in the {@link #newFcInstance newFcInstance} method must
 * be instances of {@link ComponentType}.</li>
 * <li>the controller descriptors used in the {@link #newFcInstance
 * newFcInstance} method must be strings referencing "true descriptors", which
 * are loaded with the {@link #_this_weaveableL} loader. Here "true descriptors"
 * means controller descriptors as specified in {@link
 * org.objectweb.fractal.julia.asm.ContextClassGenerator}.</li>
 * </ul>
 * Contributors : <a href="mailto:jakub.kornas@inrialpes.fr">Jakub Kornas</a>,
 * <a href="mailto:julien.legrand@inrialpes.fr">Julien Legrand</a>
 */

public class JadeGenericFactoryMixin implements GenericInstallingFactory,
        BindingController, DeploymentStateController {

    /**
     *
     */
    private final String INSTALLER_ITF = "installer";

    /**
     *
     */
    private final String GC_ITF = "gc";

    /**
     *
     */
    private final String FRACTAL_RMI_REGISTRY_ITF = "registry";

    /**
     *
     */
    private OSGiInstaller installer = null;

    /**
     *
     */
    private GarbageCollector gc = null;

    /**
     *
     */
    private NamingService registry = null;

    /**
     *
     */
    private Map<Component, Loader> cmpsToLdr;

    /**
     *
     */
    private Set<Component> deployedCmps;

    // -------------------------------------------------------------------------
    // PUBLIC constructor (needed for bootstrap)
    // -------------------------------------------------------------------------

    public JadeGenericFactoryMixin() {
    }

    // -------------------------------------------------------------------------
    // Fields and methods added and overriden by the mixin class
    // -------------------------------------------------------------------------

    /**
     * Returns a tree representing the given component type. This tree has the
     * format specified in {@link
     * org.objectweb.fractal.julia.asm.ContextClassGenerator}
     *
     * @param type
     *            a component type, must be instance of {@link ComponentType}.
     * @return a tree representing the given component type.
     * @throws InstantiationException
     *             if the given type is not an instance of {@link ComponentType},
     *             or if it contains control interface types.
     */

    public Tree getFcTypeDescriptor(final Type type)
            throws InstantiationException {
        // checks the type system
        ComponentType compType;
        try {
            compType = (ComponentType) type;
        } catch (ClassCastException e) {
            throw new ChainedInstantiationException(e, null,
                    "The component type must be an instance of ComponentType");
        }

        // checks the component type
        InterfaceType[] itfTypes = compType.getFcInterfaceTypes();
        Tree[] itfTrees = new Tree[itfTypes.length];
        for (int i = 0; i < itfTypes.length; i++) {
            InterfaceType itfType = itfTypes[i];
            String itfName = itfType.getFcItfName();
            if (!itfName.equals("attribute-controller")) {
                if (itfName.equals("component")
                        || itfName.endsWith("-controller")) {
                    throw new ChainedInstantiationException(null, null,
                            "The component type must not contain control interface types");
                }
            }
            itfTrees[i] = new Tree(new Tree[] { new Tree(itfName),
                    new Tree(itfType.getFcItfSignature()),
                    new Tree(itfType.isFcClientItf() ? "true" : "false"),
                    new Tree(itfType.isFcOptionalItf() ? "true" : "false"),
                    new Tree(itfType.isFcCollectionItf() ? "true" : "false") });
        }

        // returns the type descriptor
        return new Tree(itfTrees);
    }

    /**
     * Returns the tree corresponding to the given controller descriptor. This
     * tree is found by using the {@link #_this_weaveableL} loader. It must have
     * the format specified in {@link
     * org.objectweb.fractal.julia.asm.ContextClassGenerator}
     *
     * @param controllerDesc
     *            a string referencing a true controller descriptor.
     * @return the tree corresponding to the given controller descriptor.
     * @throws InstantiationException
     *             if the tree cannot be loaded.
     */

    public Tree getFcControllerDescriptor(final Object controllerDesc)
            throws InstantiationException {
        Map definitions = new HashMap();
        definitions.put("attributeControllerInterface", new Tree("QUOTE"));
        definitions.put("interfaceName", new Tree("QUOTE"));
        try {
            return _this_weaveableL.evalTree(_this_weaveableL
                    .loadTree((String) controllerDesc), definitions);
        } catch (Exception e) {
            throw new ChainedInstantiationException(e, null,
                    "Cannot load the '" + controllerDesc
                            + "' controller descriptor");
        }
    }

    /**
     * Returns the tree corresponding to the given content descriptor. This tree
     * has the format specified in {@link
     * org.objectweb.fractal.julia.asm.ContextClassGenerator}
     *
     * @param contentDesc
     *            a content descriptor.
     * @return the tree corresponding to the given content descriptor.
     */

    public Tree getFcContentDescriptor(final Object contentDesc) {
        if (contentDesc instanceof String) {
            return new Tree((String) contentDesc);
        } else if (contentDesc != null && !(contentDesc instanceof Object[])) {
            return new Tree(contentDesc.getClass().getName());
        } else {
            return new Tree("EMPTY");
        }
    }

    // -------------------------------------------------------------------------
    // Fields and methods required by the mixin class in the base class
    // -------------------------------------------------------------------------

    /**
     * The <tt>weaveableL</tt> field required by this mixin. This field is
     * supposed to reference the {@link Loader} interface of the component to
     * which this controller object belongs.
     */

    public org.objectweb.fractal.julia.loader.Loader _this_weaveableL;

    /**
     * The <tt>weaveableTF</tt> field required by this mixin. This field is
     * supposed to reference the {@link TypeFactory} interface of the component
     * to which this controller object belongs.
     */

    public TypeFactory _this_weaveableTF;

    /**
     * The <tt>weaveableBC</tt> field required by this mixin. This field is
     * supposed to reference the {@link BindingController} interface of the
     * component to which this controller object belongs.
     */

    public BindingController _this_weaveableBC;

    /**
     * The <tt>weaveableGIF</tt> field required by this mixin. This field is
     * supposed to reference the {@link GenericInstallingFactory} interface of
     * the component to which this controller object belongs.
     */

    public GenericInstallingFactory _this_weaveableGIF;

    /**
     * The <tt>weaveableJCC</tt> field required by this mixin. This field is
     * supposed to reference the
     * {@link GenericInstallingFactoryContextController} interface of the
     * component to which this controller object belongs.
     */
    public GenericInstallingFactoryContextController _this_weaveableGIFCC;

    // ------------------------------------------------------------------------
    // Implementation of GenericFactory interface
    // ------------------------------------------------------------------------

    /*
     * (non-Javadoc)
     *
     * @see org.objectweb.fractal.api.factory.GenericFactory#newFcInstance(org.objectweb.fractal.api.Type,
     *      java.lang.Object, java.lang.Object)
     */
    public synchronized Component newFcInstance(final Type type,
            final Object controllerDesc, final Object contentDesc)
            throws InstantiationException {
        Object loader;
        Object controller;
        String juliaCfg = null;
        String juliaConfig = System.getProperty("julia.config");
        ClassLoader cl = null;

        boolean saveLoader = false;

        if (controllerDesc instanceof Object[]) {
            loader = getFcLoader(((Object[]) controllerDesc)[0]);
            controller = ((Object[]) controllerDesc)[1];
            saveLoader = true;
        } else if (controllerDesc instanceof Map) {
            Map params = (Map) controllerDesc;
            juliaCfg = params.get("juliaconfig").toString();
            controller = params.get("controller").toString();
            if (params.get("classloader") != null) {
                cl = (ClassLoader) params.get("classloader");
                loader = getFcLoader(cl);
            } else {
                loader = getFcLoader(null);
            }
        } else {
            loader = getFcLoader(null);
            controller = controllerDesc;
        }

        if (juliaCfg != null && cl != null) {
            Map<String, Object> hints = new HashMap<String, Object>();
            hints.put("classloader", cl);
            hints.put("julia.config", juliaCfg);
            try {
                _this_weaveableL.init(hints);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        // generates the component descriptor
        Tree typeTree = getFcTypeDescriptor(type);
        Tree controllerTree = getFcControllerDescriptor(controller);
        Tree contentTree = getFcContentDescriptor(contentDesc);
        Tree componentTree = new Tree(
                new Tree[] {
                        new Tree(
                                "org.objectweb.fractal.julia.asm.ContextClassGenerator"),
                        typeTree, controllerTree, contentTree });

        // creates an instance of the corresponding class
        // (a specific InitializationContext class for components of this kind)
        Class initializationContextClass;
        try {
            initializationContextClass = _this_weaveableL.loadClass(
                    componentTree, loader);
        } catch (Exception e) {
            throw new ChainedInstantiationException(e, null,
                    "Cannot load the specific InitializationContext sub class "
                            + "needed to create the component");
        }
        InitializationContext initializationContext;
        try {
            initializationContext = (InitializationContext) initializationContextClass
                    .newInstance();
        } catch (Exception e) {
            throw new ChainedInstantiationException(e, null,
                    "Cannot create the specific InitializationContext object "
                            + "needed to create the component");
        }

        // initializes the initialization context itself
        if (!(contentDesc instanceof String)) {
            initializationContext.content = contentDesc;
        }
        initializationContext.hints = _this_weaveableTF;
        initializationContext.create();

        // initializes all the controller objects of the component
        List controllers = initializationContext.controllers;
        for (int i = 0; i < controllers.size(); ++i) {
            Object o = controllers.get(i);
            if (o instanceof Controller) {
                ((Controller) o).initFcController(initializationContext);
            }
        }

        // returns the Component interface of the component
        try {
            Component c = (Component) initializationContext
                    .getInterface("component");
            Component result = (Component) c.getFcInterface("component");
            if (cmpsToLdr == null) {
                cmpsToLdr = new HashMap<Component, Loader>();
            }
            if (saveLoader) {
                Object l = getFcLoader(((Object[]) controllerDesc)[0]);
                if (l != null && l instanceof ClassLoader) {
                    cmpsToLdr.put(result, new ClassLoaderWrapper(
                            (ClassLoader) l));
                }
            }

            if (juliaCfg != null) {
                Map<String, Object> hints = new HashMap<String, Object>();
                // reconfigure the loader to the previous state
                ClassLoader juliaCl = getClass().getClassLoader();
                hints.put("classloader", juliaCl);
                hints.put("julia.config", juliaConfig);
                try {
                    _this_weaveableL.init(hints);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }

            setGenericFactory(result);

            /*
             * add the instantiated component in the new-component container if
             * it's not null. Beware ! This code can't be refactored in a
             * method. Due to class generation and probably the use of
             * synchronized method
             */
            try {

                Component newComponentContainer = _this_weaveableGIFCC
                        .getNewComponentContainer();

                if (newComponentContainer != null) {

                    ContentController cc = Fractal
                            .getContentController(newComponentContainer);

                    cc.addFcSubComponent(result);
                }

            } catch (Exception ignored) {
                // ignored.printStackTrace();
            }

            return result;

        } catch (NoSuchInterfaceException e) {
            throw new ChainedInstantiationException(e, null,
                    "The created component does not provide the Component interface");
        }
    }

    /**
     * @param loader
     * @return
     */
    public Object getFcLoader(final Object loader) {
        return loader;
    }

    // ------------------------------------------------------------------------
    // Implementation of GenericInstallingFactory interface
    // ------------------------------------------------------------------------

    /*
     * (non-Javadoc)
     *
     * @see org.objectweb.fractal.deployment.local.api.GenericInstallingFactory#newFcInstance(org.objectweb.fractal.api.Type,
     *      java.lang.Object, java.lang.Object,
     *      org.objectweb.fractal.deployment.local.api.PackageDescription)
     */
    public synchronized Component newFcInstance(Type type,
            Object controllerDesc, Object contentDesc,
            PackageDescription packageDescription)
            throws InstantiationException {

        LocallyInstalledPackage pkg = null;
        ClassLoader cl = null;
        Object ctrl = null;
        Loader ldr = null;

        if (controllerDesc instanceof Object[]) {

            cl = (ClassLoader) (((Object[]) controllerDesc)[0]);
            ctrl = (((Object[]) controllerDesc)[1]);
        } else if (installer != null) {

            Map properties = (Map) packageDescription.getPackageProperties();
            Boolean isStart = new Boolean((String) properties.get("isStart"));

            if (isStart == null) {
                pkg = installer.install(packageDescription);
            } else {
                pkg = installer.install(packageDescription, isStart);
            }

            if (pkg != null) {
                ldr = pkg.getLoader();
            }

            if (ldr != null) {
                cl = ((ClassLoaderWrapper) ldr).getClassLoader();
            } else {
                cl = getClass().getClassLoader();
                ldr = new ClassLoaderWrapper(cl);
            }
        } else {

            cl = getClass().getClassLoader();
            ldr = new ClassLoaderWrapper(cl);
        }

        if (controllerDesc instanceof Map) {
            Map params = (Map) controllerDesc;
            String juliaCfg = params.get("juliaconfig").toString();
            ctrl = params.get("controller").toString();
            String juliaConfig = System.getProperty("julia.config");
            try {
                Map<String, Object> hints = new HashMap<String, Object>();
                hints.put("classloader", cl);
                hints.put("julia.config", juliaCfg);
                _this_weaveableL.init(hints);

                hints = new HashMap<String, Object>();
                hints.put("classloader", cl);

                Component result = newComponent(type,
                        new Object[] { cl, ctrl }, contentDesc);

                // TODO: refactor (the GF controller should not have a set
                // method)
                setGenericFactory(result);
                cmpsToLdr.put(result, ldr);
                if (gc != null) {
                    gc.notifyComponentCreated(result, pkg);
                }

                // reconfigure the loader to the previous state
                ClassLoader juliaCl = getClass().getClassLoader();
                hints.put("classloader", juliaCl);
                hints.put("julia.config", juliaConfig);
                _this_weaveableL.init(hints);

                return result;
            } catch (NoSuchInterfaceException e) {
                e.printStackTrace();
            } catch (InstantiationException e) {
                e.printStackTrace();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        if (ctrl == null) {
            ctrl = controllerDesc;
        }
        Component result = newComponent(type, new Object[] { cl, ctrl },
                contentDesc);

        // TODO: refactor (the GF controller should not have a set method)
        setGenericFactory(result);
        cmpsToLdr.put(result, ldr);
        if (gc != null) {
            gc.notifyComponentCreated(result, pkg);
        }

        /*
         * add the instantiated component in the new-component container if it's
         * not null. Beware ! This code can't be refactored in a method. Due to
         * class generation and probably the use of synchronized method
         */
        try {

            Component newComponentContainer = _this_weaveableGIFCC
                    .getNewComponentContainer();

            if (newComponentContainer != null) {

                ContentController cc = Fractal
                        .getContentController(newComponentContainer);

                cc.addFcSubComponent(result);
            }

        } catch (Exception ignored) {
            // ignored.printStackTrace();
        }

        return result;
    }

    /*
     * (non-Javadoc)
     *
     * @see org.objectweb.fractal.deployment.local.api.GenericInstallingFactory#undeployFcComponent(org.objectweb.fractal.api.Component)
     */
    public synchronized void undeployFcComponent(Component cmp) {
        cmpsToLdr.remove(cmp);
        if (gc != null) {
            gc.notifyComponentDestroyed(cmp);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see org.objectweb.fractal.deployment.local.api.GenericInstallingFactory#getFcLoaderForComponent(org.objectweb.fractal.api.Component)
     */
    public synchronized Loader getFcLoaderForComponent(Component arg0) {
        Loader result = null;
        if (cmpsToLdr != null) {
            result = cmpsToLdr.get(arg0);
        }
        return result;
    }

    // ------------------------------------------------------------------------
    // Implementation of BindingController interface
    // ------------------------------------------------------------------------

    public void bindFc(String clientItfName, Object serverItf)
            throws NoSuchInterfaceException, IllegalBindingException,
            IllegalLifeCycleException {
        if (clientItfName.equals(this.INSTALLER_ITF)) {
            installer = (OSGiInstaller) serverItf;
        } else if (clientItfName.equals(this.GC_ITF)) {
            gc = (GarbageCollector) serverItf;
        } else if (clientItfName.equals(this.FRACTAL_RMI_REGISTRY_ITF)) {
            registry = (NamingService) serverItf;
        }
        // else if (clientItfName.equals("component")) {
        // myself = (Component) serverItf;
        // }
    }

    public String[] listFc() {
        return new String[] { INSTALLER_ITF, GC_ITF, FRACTAL_RMI_REGISTRY_ITF };
    }

    public Object lookupFc(String clientItfName)
            throws NoSuchInterfaceException {
        if (clientItfName.equals(INSTALLER_ITF)) {
            return installer;
        } else if (clientItfName.equals(GC_ITF)) {
            return gc;
        } else if (clientItfName.equals(FRACTAL_RMI_REGISTRY_ITF)) {
            return registry;
        }
        return null;
    }

    public void unbindFc(String clientItfName) throws NoSuchInterfaceException,
            IllegalBindingException, IllegalLifeCycleException {
        if (clientItfName.equals(this.INSTALLER_ITF)) {
            this.installer = null;
        } else if (clientItfName.equals(this.GC_ITF)) {
            this.gc = null;
        } else if (clientItfName.equals(this.FRACTAL_RMI_REGISTRY_ITF)) {
            this.registry = null;
        }
    }

    // -------------------------------------------------------------------------
    // Implementation of DeploymentStateController interface
    // -------------------------------------------------------------------------

    /*
     * (non-Javadoc)
     *
     * @see org.ow2.jasmine.jade.fractal.api.control.DeploymentStateController#addDeployedComponent(org.objectweb.fractal.api.Component)
     */
    public boolean addDeployedComponent(Component deployedComponent) {

        if (deployedCmps == null)
            deployedCmps = new HashSet<Component>();

        return deployedCmps.add(deployedComponent);

    }

    /*
     * (non-Javadoc)
     *
     * @see org.ow2.jasmine.jade.fractal.api.control.DeploymentStateController#removeDeployedComponent(org.objectweb.fractal.api.Component)
     */
    public boolean removeDeployedComponent(Component deployedComponent) {
        return deployedCmps.remove(deployedComponent);
    }

    /*
     * (non-Javadoc)
     *
     * @see org.ow2.jasmine.jade.fractal.api.control.DeploymentStateController#getDeployedComponent()
     */
    public Set<Component> getDeployedComponent() {

        if (deployedCmps == null || deployedCmps.isEmpty())
            return null;

        return deployedCmps;
    }

    // ------------------------------------------------------------------------
    // Private Methods
    // ------------------------------------------------------------------------

    /**
     * @param type
     * @param controllerDesc
     * @param contentDesc
     * @return
     * @throws InstantiationException
     */
    private Component newComponent(final Type type,
            final Object controllerDesc, final Object contentDesc)
            throws InstantiationException {
        Object loader;
        Object controller;

        if (controllerDesc instanceof Object[]) {
            loader = getFcLoader(((Object[]) controllerDesc)[0]);
            controller = ((Object[]) controllerDesc)[1];
        } else {
            loader = getFcLoader(null);
            controller = controllerDesc;
        }

        // generates the component descriptor
        Tree typeTree = getFcTypeDescriptor(type);
        Tree controllerTree = getFcControllerDescriptor(controller);
        Tree contentTree = getFcContentDescriptor(contentDesc);
        Tree componentTree = new Tree(
                new Tree[] {
                        new Tree(
                                "org.objectweb.fractal.julia.asm.ContextClassGenerator"),
                        typeTree, controllerTree, contentTree });

        // creates an instance of the corresponding class
        // (a specific InitializationContext class for components of this kind)
        Class initializationContextClass;
        try {
            initializationContextClass = _this_weaveableL.loadClass(
                    componentTree, loader);
        } catch (Exception e) {
            throw new ChainedInstantiationException(e, null,
                    "Cannot load the specific InitializationContext sub class "
                            + "needed to create the component");
        }
        InitializationContext initializationContext;
        try {
            initializationContext = (InitializationContext) initializationContextClass
                    .newInstance();
        } catch (Exception e) {
            throw new ChainedInstantiationException(e, null,
                    "Cannot create the specific InitializationContext object "
                            + "needed to create the component");
        }

        // initializes the initialization context itself
        if (!(contentDesc instanceof String)) {
            initializationContext.content = contentDesc;
        }
        initializationContext.hints = _this_weaveableTF;
        initializationContext.create();

        // initializes all the controller objects of the component
        List controllers = initializationContext.controllers;
        for (int i = 0; i < controllers.size(); ++i) {
            Object o = controllers.get(i);
            if (o instanceof Controller) {
                ((Controller) o).initFcController(initializationContext);
            }
        }

        // returns the Component interface of the component
        try {
            Component c = (Component) initializationContext
                    .getInterface("component");
            return (Component) c.getFcInterface("component");
        } catch (NoSuchInterfaceException e) {
            throw new ChainedInstantiationException(e, null,
                    "The created component does not provide the Component interface");
        }
    }

    /**
     * @param result
     */
    private void setGenericFactory(Component result) {
        try {
            ((GenericInstallingFactoryController) result
                    .getFcInterface("generic-installing-factory-controller"))
                    .setGenericFactory(_this_weaveableGIF);
        } catch (NoSuchInterfaceException ignored) {
        }
    }

}
