/***
 * Julia: France Telecom's implementation of the Fractal API
 * Copyright (C) 2001-2002 France Telecom R&D
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact: Eric.Bruneton@rd.francetelecom.com
 *
 * Author: Eric Bruneton
 */

package org.ow2.jasmine.jade.fractal.julia.loader;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.objectweb.fractal.julia.loader.Generated;
import org.objectweb.fractal.julia.loader.Initializable;
import org.objectweb.fractal.julia.loader.Loader;
import org.objectweb.fractal.julia.loader.Tree;

/**
 * Provides a basic implementation of the
 * {@link org.objectweb.fractal.julia.loader.Loader} interface. This
 * implementation loads the required classes from the class path. <br>
 * <p>
 * This class is an extension of the
 * {@link org.objectweb.fractal.julia.loader.BasicLoader}.
 * <p>
 * Contributors : <a href="mailto:jakub.kornas@inrialpes.fr">Jakub Kornas
 */

public class BasicLoader implements Loader {

	// -------------------------------------------------------------------------
	// PUBLIC constructor (needed for bootstrap)
	// -------------------------------------------------------------------------

	public BasicLoader() {
	}

	// -------------------------------------------------------------------------
	// Fields and methods added and overriden by the mixin class
	// -------------------------------------------------------------------------

	/**
	 * The value of the 'julia.config' system property. This property is a list
	 * of configuration file names.
	 */

	private String config;

	/**
	 * The value of the 'julia.loader.gen.check' system property. This property
	 * indicates whether or not the class parameters of the generated classes
	 * must be checked.
	 */

	private boolean checkClassParameters;

	/**
	 * The value of the 'julia.loader.gen.pkg' system property. This property
	 * indicates the package that contains the classes generated by Julia.
	 */

	private String genPkg;

	/**
	 * A map associating variable names to {@link Tree} values.
	 */

	private Map definitions;

	/**
	 * A transient stream used to parse trees.
	 */

	private InputStream is;

	/**
	 * Current line number in {@link #is is}.
	 */

	private int line;

	/**
	 * Last character read from {@link #is is}.
	 */

	private int car;

	/**
	 * A class loader used to evaluate the Tree
	 * 
	 */
	private ClassLoader cl = getClass().getClassLoader();

	// -------------------------------------------------------------------------
	// Implementation of the Loader interface
	// -------------------------------------------------------------------------

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.objectweb.fractal.julia.loader.Loader#init(java.util.Map)
	 */
	public void init(final Map context) throws Exception {
		if (context.get("classloader") != null) {
			cl = (ClassLoader) context.get("classloader");
			definitions = null;
		} else {
			cl = getClass().getClassLoader();
		}

		if (context.get("julia.config") != null) {
			config = (String) context.get("julia.config");
		} else if (System.getProperty("julia.config") != null) {
			config = System.getProperty("julia.config");
		} else {
			throw new Exception(
					"The julia.config system property is not defined");
		}

		if (context.get("julia.loader.check") != null) {
			checkClassParameters = true;
		} else if (System.getProperty("julia.loader.check") != null) {
			checkClassParameters = true;
		}

		genPkg = "org.objectweb.fractal.julia.generated";
		if (context.get("julia.loader.gen.pkg") != null) {
			genPkg = (String) context.get("julia.loader.gen.pkg");
		} else if (System.getProperty("julia.loader.gen.pkg") != null) {
			genPkg = System.getProperty("julia.loader.gen.pkg");
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.objectweb.fractal.julia.loader.Loader#loadTree(java.lang.String)
	 */
	public Tree loadTree(final String name) throws Exception {
		if (definitions == null) {
			definitions = new HashMap();
			int b = 0;
			int i;
			do {
				i = config.indexOf(',', b);
				String file;
				if (i == -1) {
					file = config.substring(b);
				} else {
					file = config.substring(b, i);
					b = i + 1;
				}
				InputStream is;
				try {
					is = getInputStream(file);
				} catch (IOException ioe) {
					is = cl.getResourceAsStream(file);
					if (is == null) {
						is = getClass().getResourceAsStream("/" + file);
					}
					if (is == null) {
						throw new Exception("Cannot find or open the '" + file
								+ "' file");
					}
				}
				try {
					line = 1;
					while (true) {
						Tree def;
						try {
							def = parseTree(is);
						} catch (Throwable e) {
							throw new Exception("File '" + file + "', line "
									+ line + ": " + e.getMessage());
						}
						if (def == null) {
							break;
						} else {
							definitions.put(def.getSubTree(0).toString(), def
									.getSubTree(1));
						}
					}
				} finally {
					try {
						is.close();
					} catch (IOException ignored) {
					}
				}
			} while (i != -1);
		}
		Tree t = (Tree) definitions.get(name);
		if (t == null) {
			throw new Exception(
					"Cannot find the '"
							+ name
							+ "' descriptor in the '"
							+ config
							+ "' file(s) specified in the julia.loader.config system property");
		}
		return t;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.objectweb.fractal.julia.loader.Loader#evalTree(org.objectweb.fractal.julia.loader.Tree,
	 *      java.util.Map)
	 */
	public Tree evalTree(final Tree tree, final Map context) throws Exception {
		if (tree.getSize() == 0) {
			String var = tree.toString();
			if (var.startsWith("'")) {
				var = var.substring(1);
				Tree val = (Tree) context.get(var);
				if (val == null) {
					val = loadTree(var);
				}
				if (val == null) {
					throw new Exception(var);
				}
				if (!val.toString().equals("QUOTE")) {
					return evalTree(val, context);
				}
			}
			return tree;
		} else {
			boolean ok = true;
			Tree[] subTrees = tree.getSubTrees();
			for (int i = 0; i < subTrees.length; ++i) {
				Tree oldSubTree = subTrees[i];
				Tree newSubTree = evalTree(oldSubTree, context);
				if (newSubTree != oldSubTree) {
					if (ok) {
						Tree[] newSubTrees = new Tree[subTrees.length];
						System.arraycopy(subTrees, 0, newSubTrees, 0,
								subTrees.length);
						subTrees = newSubTrees;
						ok = false;
					}
					subTrees[i] = newSubTree;
				}
			}
			return (ok ? tree : new Tree(subTrees));
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.objectweb.fractal.julia.loader.Loader#newObject(org.objectweb.fractal.julia.loader.Tree,
	 *      java.lang.Object)
	 */
	public Object newObject(final Tree objectDescriptor, final Object loader)
			throws Exception {
		if (objectDescriptor.getSize() == 0) {
			return loadClass(objectDescriptor.toString(), loader).newInstance();
		} else {
			Object o;
			o = loadClass(objectDescriptor.getSubTree(0), loader).newInstance();
			if (objectDescriptor.getSize() > 1) {
				Tree[] trees = objectDescriptor.getSubTrees();
				Tree[] args = new Tree[trees.length - 1];
				System.arraycopy(trees, 1, args, 0, args.length);
				((Initializable) o).initialize(new Tree(args));
			}
			return o;
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.objectweb.fractal.julia.loader.Loader#loadClass(java.lang.String,
	 *      java.lang.Object)
	 */
	public Class loadClass(final String name, final Object loader)
			throws ClassNotFoundException {
		return Class.forName(name);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.objectweb.fractal.julia.loader.Loader#loadClass(org.objectweb.fractal.julia.loader.Tree,
	 *      java.lang.Object)
	 */
	public Class loadClass(final Tree classDescriptor, final Object loader)
			throws ClassNotFoundException {
		if (classDescriptor.getSize() == 0) {
			return loadClass(classDescriptor.toString(), loader);
		}
		int n = 0;
		String s = genPkg + ".C"
				+ Integer.toHexString(classDescriptor.hashCode()) + "_";
		while (true) {
			String name = s + n;
			Class c;
			try {
				c = loadClass(name, loader);
			} catch (ClassNotFoundException e) {
				return generateClass(name, classDescriptor, loader);
			}
			if (!checkClassParameters) {
				return c;
			}
			try {
				Generated gen = (Generated) c.newInstance();
				if (gen != null
						&& classDescriptor.equals(gen
								.getFcGeneratorParameters())) {
					return c;
				}
			} catch (Exception ignored) {
			}
			++n;
		}
	}

	// -------------------------------------------------------------------------
	// Utility methods: tree parsing
	// -------------------------------------------------------------------------

	/*
	 * Convenience method used for J2ME conversion (FileInputStream not
	 * available in CLDC API).
	 */

	private InputStream getInputStream(String file) throws IOException {
		return new FileInputStream(file);
	}

	/**
	 * Creates a tree by parsing the given input stream. This method can be
	 * called several times on the same stream, in order to parse several
	 * consecutive trees, but <i>only</i> if the there is at least one space,
	 * tab or return character between these trees.
	 * 
	 * @param is
	 *            the input stream to be parsed. This input stream must use a
	 *            LISP like syntax, with single line comments beginning with
	 *            '#', as in the following example:
	 * 
	 * <pre>
	 *        ( # foo
	 *          foo
	 *          # bar
	 *          (bar foo)
	 *        )
	 * </pre>
	 * 
	 * @return the tree parsed from the given input stream, or <tt>null</tt>
	 *         if the end of the stream has been reached.
	 * @throws IOException
	 *             if an I/O exception occurs during parsing.
	 * @throws Exception
	 *             if a syntax error is found.
	 */

	private Tree parseTree(final InputStream is) throws Exception {
		try {
			this.is = is;
			read();
			parseSpaces();
			if (car == -1) {
				return null;
			} else {
				return parseTree();
			}
		} finally {
			this.is = null;
		}
	}

	/**
	 * Recursive method to parse a tree. The first character of the tree to be
	 * parsed is supposed to have already been read, and available in
	 * {@link #car car}. After parsing, the character immediately following the
	 * parsed tree is also supposed to have already been parsed, and available
	 * in {@link #car car}.
	 * 
	 * @return a tree parsed from the {@link #is is} stream.
	 * @throws IOException
	 *             if an I/O exception occurs during parsing.
	 * @throws Exception
	 *             if a syntax error is found.
	 */

	private Tree parseTree() throws IOException, Exception {
		int c = car;
		if (c == -1) {
			throw new Exception("Unexpected end of file");
		} else if (c == ')') {
			throw new Exception("Unmatched closing parenthesis");
		} else if (c == '(') {
			// parses a tree of the form "(subTree1 ... subTreeN)"
			read();
			List subTrees = new ArrayList();
			while (true) {
				c = parseSpaces();
				if (c == ')') {
					read();
					return new Tree((Tree[]) subTrees.toArray(new Tree[subTrees
							.size()]));
				} else {
					subTrees.add(parseTree());
				}
			}
		} else {
			// parses a tree of the form "tree"
			StringBuffer buf = new StringBuffer();
			while (true) {
				buf.append((char) c);
				c = read();
				if (c == -1 || c == ' ' || c == '\t' || c == '\n' || c == '\r'
						|| c == '#' || c == '(' || c == ')') {
					car = c;
					return new Tree(buf.toString());
				}
			}
		}
	}

	/**
	 * Parses spaces and comments until a non space character is found.
	 * 
	 * @return the first non space character found, which is also stored in
	 *         {@link #car car}.
	 * @throws IOException
	 *             if I/O exception occurs during parsing.
	 */

	private int parseSpaces() throws IOException {
		int c = car;
		while (c == ' ' || c == '\t' || c == '\n' || c == '\r' || c == '#') {
			if (c == '#') {
				// parses a single line comment
				do {
					c = read();
				} while (c != '\n' && c != '\r');
			}
			c = read();
		}
		return car = c;
	}

	/**
	 * Reads a char in {@link #is} and updates {@link #line} if needed.
	 * 
	 * @return a char read from {@link #is}.
	 * @throws IOException
	 *             if I/O exception occurs during reading.
	 */

	private int read() throws IOException {
		car = is.read();
		if (car == '\n' || car == '\r') {
			++line;
		}
		return car;
	}

	// -------------------------------------------------------------------------
	// Utility methods: class generation
	// -------------------------------------------------------------------------

	/**
	 * Generates the class whose descriptor is given, with the given name. The
	 * generated class must implement the {@link Generated} interface, and its
	 * {@link Generated#getFcGeneratorParameters getFcGeneratorParameters}
	 * method must return <tt>classDescriptor.toString()</tt>. The default
	 * implementation of this method throws an exception.
	 * 
	 * @param name
	 *            the name of the class to be generated.
	 * @param classDescriptor
	 *            the descriptor of the class to be generated. This descriptor
	 *            <i>must</i> be of the form "(objectDescriptor arg1 ... argN)"
	 *            (see {@link #loadClass(Tree,Object) loadClass}).
	 * @param loader
	 *            an optional class loader to load auxiliary classes.
	 * @return a class named <tt>name</tt> and whose content is described by
	 *         the given class descriptor.
	 * @throws ClassNotFoundException
	 *             if a problem occurs during the generation of the class.
	 */

	protected Class generateClass(final String name,
			final Tree classDescriptor, final Object loader)
			throws ClassNotFoundException {
		throw new ClassNotFoundException(name);
	}

	// -------------------------------------------------------------------------
	// Fields and methods required by the mixin class in the base class
	// -------------------------------------------------------------------------

}
