/***
 * Reflex-Fractal
 *
 * Copyright (C) 2007 : INRIA - Domaine de Voluceau, Rocquencourt, B.P. 105,
 * 78153 Le Chesnay Cedex - France
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact: jade <AT> inrialpes <DOT> fr
 *
 * Author: SARDES project - http://sardes.inrialpes.fr
 *
 */

package org.ow2.jasmine.jade.reflex.adl;

import java.util.Map;

import org.objectweb.deployment.scheduling.component.api.FactoryProviderTask;
import org.objectweb.deployment.scheduling.component.api.InstanceProviderTask;
import org.objectweb.deployment.scheduling.core.api.Scheduler;
import org.objectweb.fractal.adl.ADLException;
import org.objectweb.fractal.adl.BasicTaskMap;
import org.objectweb.fractal.adl.Compiler;
import org.objectweb.fractal.adl.Definition;
import org.objectweb.fractal.adl.Factory;
import org.objectweb.fractal.adl.Loader;
import org.objectweb.fractal.adl.Node;
import org.objectweb.fractal.adl.TaskMap;
import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.objectweb.fractal.api.control.BindingController;
import org.objectweb.fractal.api.factory.InstantiationException;
import org.objectweb.fractal.util.Fractal;

import org.ow2.jasmine.jade.reflex.api.ReflexFractal;
import org.ow2.jasmine.jade.reflex.api.control.ReflexController;
import org.ow2.jasmine.jade.reflex.util.Logger;
import org.ow2.jasmine.jade.reflex.util.Reflex;

/**
 * Basic implementation of the {@link Factory} interface. This implementation
 * uses a {@link Loader} to load ADL definitions, a {@link Compiler} to compile
 * them, and a {@link Scheduler} to execute the compiled tasks.
 */

public class ReflexBasicFactory implements BindingController, Factory {

    /**
     * Name of the client interface bound to the {@link Loader} used by this
     * factory.
     */

    public final static String LOADER_BINDING = "loader";

    /**
     * Name of the client interface bound to the {@link Compiler} used by this
     * factory.
     */

    public final static String COMPILER_BINDING = "compiler";

    /**
     * Name of the client interface bound to the {@link Scheduler} used by this
     * factory.
     */

    public final static String SCHEDULER_BINDING = "scheduler";

    /**
     * The {@link Loader} used by this factory.
     */

    public Loader loader;

    /**
     * The {@link Compiler} used by this factory.
     */

    public Compiler compiler;

    /**
     * The {@link Scheduler} used by this factory.
     */

    public Scheduler scheduler;

    // --------------------------------------------------------------------------
    // Implementation of the BindingController interface
    // --------------------------------------------------------------------------

    public String[] listFc() {
        return new String[] { LOADER_BINDING, COMPILER_BINDING,
                SCHEDULER_BINDING };
    }

    public Object lookupFc(final String itf) {
        if (itf.equals(LOADER_BINDING)) {
            return loader;
        } else if (itf.equals(COMPILER_BINDING)) {
            return compiler;
        } else if (itf.equals(SCHEDULER_BINDING)) {
            return scheduler;
        }
        return null;
    }

    public void bindFc(final String itf, final Object value) {
        if (itf.equals(LOADER_BINDING)) {
            loader = (Loader) value;
        } else if (itf.equals(COMPILER_BINDING)) {
            compiler = (Compiler) value;
        } else if (itf.equals(SCHEDULER_BINDING)) {
            scheduler = (Scheduler) value;
        }
    }

    public void unbindFc(final String itf) {
        if (itf.equals(LOADER_BINDING)) {
            loader = null;
        } else if (itf.equals(COMPILER_BINDING)) {
            compiler = null;
        } else if (itf.equals(SCHEDULER_BINDING)) {
            scheduler = null;
        }
    }

    // -------------------------------------------------------------------------
    // Implementation of the Facfory interface
    // -------------------------------------------------------------------------

    /*
     * (non-Javadoc)
     *
     * @see org.objectweb.fractal.adl.Factory#newComponentType(java.lang.String,
     *      java.util.Map)
     */
    public Object newComponentType(String name, Map context)
            throws ADLException {
        Definition d = loader.load(name, context);
        TaskMap m = new BasicTaskMap();
        compiler.compile(d, m, context);
        try {
            m.getTask("type", d).execute(context);
        } catch (Exception e) {
            Logger.println("Cannot execute the task: " + d.getName());
            Logger.println(e);
            throw new ADLException("Cannot execute the task: " + d.getName(), (Node) d, e);
        }
        return ((FactoryProviderTask) m.getTask("type", d)).getFactory();
    }

    /*
     * (non-Javadoc)
     *
     * @see org.objectweb.fractal.adl.Factory#newComponent(java.lang.String,
     *      java.util.Map)
     */
    public Object newComponent(String name, Map context) throws ADLException {
        Definition d = loader.load(name, context);
        TaskMap m = new BasicTaskMap();
        compiler.compile(d, m, context);
        scheduler.schedule(m.getTasks(), context);

        /*
         * add the meta component to the Meta Component Container
         */


        /*
         * get the component created
         */
        Component res = (Component) ((InstanceProviderTask) m.getTask("create",
                d)).getInstance();
        /*
         * get the Fractal bootstrap
         */
        Component bootstrap = null;
        if (context != null) {
            bootstrap = (Component) context.get("bootstrap");
        }
        if (bootstrap == null) {
            // if (loader != null) {
            // Map ctxt = new HashMap();
            // ctxt.put("classloader", loader);
            // bootstrap = ReflexFractal.getBootstrapComponent(ctxt);
            // } else {
            try {
                bootstrap = ReflexFractal.getBootstrapComponent();
            } catch (InstantiationException e) {
                Logger.println("Cannot get the reflexive bootstrap component");
                Logger.println(e);
                throw new ADLException("Cannot get the reflexive bootstrap component", (Node) d, e);
            }
            // }
        }

        Component metaComponentContainer = null;
        try {
            ReflexController rc = Reflex.getReflexController(bootstrap);
            // Component metaComponentContainer = rc.getMetaComponentContainer();
            metaComponentContainer = Reflex.getReflexController(
                    Reflex.getDualComponent(bootstrap)).getMetaComponentContainer();
        } catch (NoSuchInterfaceException e) {
            Logger.println("Cannot get the ReflexController");
            Logger.println(e);
            throw new ADLException("Cannot get the ReflexController", (Node) d, e);
        }

        boolean isMeta = false;
        if (context.containsKey("meta"))
            isMeta = (Boolean) context.get("meta");

        Component metaRes = null;
        if (isMeta)
            metaRes = res;
        else
            try {
                metaRes = Reflex.getReflexController(res).getCmpRef();
            } catch (NoSuchInterfaceException e) {
                Logger.println("Cannot get the ReflexController");
                Logger.println(e);
                throw new ADLException("Cannot get the ReflexController", (Node) d, e);
            }

        if (metaComponentContainer != null && metaRes != null)
            try {
                Fractal.getContentController(metaComponentContainer)
                        .addFcSubComponent(metaRes);
            } catch (Exception e) {
                Logger.println("Cannot add the ref of the meta component to the meta component of bootstrap");
                Logger.println(e);
                throw new ADLException("Cannot add the ref of the meta component to the meta component of bootstrap", (Node) d, e);
            }

        return res;
    }
}
