/***
 * Reflex-Fractal
 *
 * Copyright (C) 2007 : INRIA - Domaine de Voluceau, Rocquencourt, B.P. 105, 
 * 78153 Le Chesnay Cedex - France 
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact: jade <AT> inrialpes <DOT> fr
 *
 * Author: SARDES project - http://sardes.inrialpes.fr
 *
 */
 
package org.ow2.jasmine.jade.reflex.control.attribute;

import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.control.ContentController;
import org.objectweb.fractal.api.factory.InstantiationException;
import org.objectweb.fractal.julia.Controller;
import org.objectweb.fractal.julia.InitializationContext;

import org.ow2.jasmine.jade.fractal.api.control.GenericAttributeController;
import org.ow2.jasmine.jade.fractal.api.control.NoSuchAttributeException;
import org.ow2.jasmine.jade.reflex.api.control.GenericAttributeNotificationController;
import org.ow2.jasmine.jade.reflex.api.control.ReflexController;
import org.ow2.jasmine.jade.reflex.util.DebugReflex;
import org.ow2.jasmine.jade.reflex.util.Logger;
import org.ow2.jasmine.jade.reflex.util.Reflex;

/**
 * Provides an implementation of the {@link GenericAttributeController}
 * interface which manages the causal connection with the dual (i.e
 * meta/execution) level : Any attribute setting operation performed on the
 * current component is forwarded to the attribute controller of dual component.
 * 
 * In this implementation of Attribute Controller, attributes are stored in an
 * hashtable.
 * 
 * @author <a href="mailto:noel.depalma@inrialpes.fr">Noel de palma</a>
 * 
 * Modified 1/6/5 F. Boyer: Merge this class with the
 * MetaGenericAttributeController class Use _this_weaveableOptRC to get the
 * ReflexController of the current component
 */
public abstract class ReflexCompositeGenericAttributeControllerMixin implements
        GenericAttributeController, GenericAttributeNotificationController {

    // -------------------------------------------------------------------------
    // Private constructor
    // -------------------------------------------------------------------------
    private ReflexCompositeGenericAttributeControllerMixin() {
    }

    // -------------------------------------------------------------------------
    // Fields and methods added and overriden by the mixin class
    // -------------------------------------------------------------------------

    public void initFcController(final InitializationContext ic)
            throws InstantiationException {
        _super_initFcController(ic);
    }

    /**
     * set the value of an attribute identified by its name and and then
     * notifies the dual level of the attribute setting.
     * 
     * @param name
     *            the name of the attribute to set.
     * @param value
     *            the value of the attribute.
     * @throws NoSuchAttributeException
     *             if the given attribute is unknow.
     */
    public void setAttribute(String name, String value)
            throws NoSuchAttributeException {

        Logger.println(DebugReflex.attribute,
                "[GenericAttributeController] setAttribute : " + name + " = "
                        + value);
        /*
         * set the attribute
         */
        _super_setAttribute(name, value);
        /*
         * notify the dual level : get the attribute-notification-controller
         * interface of the dual component
         */
        if (_this_weaveableOptRC != null) {
            if (_this_weaveableOptRC.getIsNotifiable()) {
                try {
                    GenericAttributeNotificationController dualCtrl = Reflex
                            .getGenericAttributeNotificationController(_this_weaveableOptRC
                                    .getCmpRef());

                    dualCtrl.setAttributeNotification(name, value);

                } catch (Exception e) {
                    Logger
                            .println(DebugReflex.warning,
                                    "[Warning][GenericAttributeController] setAttribute : cannot update dual level");
                }
            }

        } else {
            Logger
                    .println(DebugReflex.warning,
                            "[Warning][GenericAttributeController] setAttribute : no dual level");
        }
    }

    /**
     * Set the value of an attribute identified by its name.
     * 
     * @param name
     *            the name of the attribute to set.
     * @param value
     *            the value of the attribute.
     * @throws NoSuchAttributeException
     *             if the given attribute is unknown.
     */
    public void setAttributeNotification(String name, String value)
            throws NoSuchAttributeException {

        Logger.println(DebugReflex.attribute,
                "[Notification][GenericAttributeController] setAttribute : "
                        + name + " = " + value);

        _super_setAttribute(name, value);
    }

    // -------------------------------------------------------------------------
    // Fields and methods required by the mixin class in the base class
    // -------------------------------------------------------------------------

    /**
     * The <tt>weaveableOptRC</tt> field required by this mixin. This field is
     * supposed to reference the {@link ReflexController} interface of the
     * component to which this controller object belongs.
     */
    public ReflexController _this_weaveableOptRC;

    /**
     * The <tt>weaveableCC</tt> field required by this mixin. This field is
     * supposed to reference the {@link ContentController} interface of the
     * component to which this controller object belongs.
     */
    public ContentController _this_weaveableCC;

    /**
     * The <tt>weaveableC</tt> field required by this mixin. This field is
     * supposed to reference the {@link Component} interface of the component to
     * which this controller object belongs.
     */
    public Component _this_weaveableC;

    /**
     * The {@link Controller#initFcController initFcController} method overriden
     * by this mixin
     */
    public abstract void _super_initFcController(InitializationContext ic)
            throws InstantiationException;

    /**
     * The {@link GenericCompositeAttributeController#setAttribute setAttribute}
     * method overriden by this mixin.
     */
    public abstract void _super_setAttribute(String name, String value);

}
