/***
 * Reflex-Fractal
 *
 * Copyright (C) 2007 : INRIA - Domaine de Voluceau, Rocquencourt, B.P. 105, 
 * 78153 Le Chesnay Cedex - France 
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact: jade <AT> inrialpes <DOT> fr
 *
 * Author: SARDES project - http://sardes.inrialpes.fr
 *
 */
 
package org.ow2.jasmine.jade.reflex.control.binding;

import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.Interface;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.objectweb.fractal.api.control.BindingController;
import org.objectweb.fractal.api.control.IllegalBindingException;
import org.objectweb.fractal.api.control.IllegalLifeCycleException;
import org.objectweb.fractal.api.control.LifeCycleController;
import org.objectweb.fractal.api.control.NameController;
import org.objectweb.fractal.util.Fractal;

import org.ow2.jasmine.jade.reflex.api.control.BindingNotificationController;
import org.ow2.jasmine.jade.reflex.api.control.ReflexController;
import org.ow2.jasmine.jade.reflex.util.DebugReflex;
import org.ow2.jasmine.jade.reflex.util.Logger;
import org.ow2.jasmine.jade.reflex.util.Reflex;

/**
 * Provides an implementation of the
 * {@link org.objectweb.fractal.api.control.BindingController} interface which
 * manages the causal connection with the dual level (i.e meta/execution) : any
 * binding operation performed on the current component is notified to the
 * binding controller of dual component.
 * 
 * @author <a href="mailto:fabienne.boyer@inrialpes.fr">Fabienne Boyer</a>
 */

public abstract class ReflexBindingMixin implements BindingController,
        BindingNotificationController {

    // -------------------------------------------------------------------------
    // Private constructor
    // -------------------------------------------------------------------------

    private ReflexBindingMixin() {
    }

    // -------------------------------------------------------------------------
    // Fields and methods added and overriden by the mixin class
    // -------------------------------------------------------------------------

    // ------------------------------------------------------------------------
    // Implementation of BindingController interface
    // ------------------------------------------------------------------------

    /**
     * Calls the overriden bindFc method and then notifies the dual level of the
     * binding.
     * 
     * @param clientItfName
     *            the name of a client interface of the component to which this
     *            interface belongs.
     * @param serverItf
     *            a server interface.
     * @throws NoSuchInterfaceException
     *             if there is no such client interface.
     * @throws IllegalBindingException
     *             if the binding cannot be created.
     * @throws IllegalLifeCycleException
     *             if this component has a {@link LifeCycleController}
     *             interface, but it is not in an appropriate state to perform
     *             this operation.
     */

    public void bindFc(String clientItfName, Object serverItf)
            throws NoSuchInterfaceException, IllegalBindingException,
            IllegalLifeCycleException {

        Logger.println(DebugReflex.binding,
                "[BindingController] bindFc : clt -| clt-itf -> srv-itf |- srv = "
                        + _this_weaveableOptNC.getFcName()
                        + " -| "
                        + clientItfName
                        + " -> "
                        + ((Interface) serverItf).getFcItfName()
                        + " |- "
                        + Fractal.getNameController(
                                ((Interface) serverItf).getFcItfOwner())
                                .getFcName());

        // calls the overriden bindFc method
        _super_bindFc(clientItfName, serverItf);

        // notify the dual level
        if (_this_weaveableOptRC != null) {
            if (_this_weaveableOptRC.getIsNotifiable()) {
                try {
                    // get the binding-notification-controller interface of the
                    // dual
                    // component

                    Component c = _this_weaveableOptRC.getCmpRef();

                    BindingNotificationController dualCtrl = Reflex
                            .getBindingNotificationController(c);

                    dualCtrl.bindFcNotification(clientItfName, Reflex
                            .getDualInterface((Interface) serverItf));

                } catch (Exception e) {
                    Logger
                            .println(DebugReflex.warning,
                                    "[Warning][BindingController] bindFc : cannot update dual level");
                }
            }
        } else {
            Logger.println(DebugReflex.warning,
                    "[Warning][BindingController] bindFc : no dual level");
        }
    }

    /**
     * Calls the overriden method and then notifies the dual level of the
     * unbinding.
     * 
     * @param clientItfName
     *            the name of a client interface of the component to which this
     *            interface belongs.
     * @throws NoSuchInterfaceException
     *             if there is no such client interface.
     * @throws IllegalBindingException
     *             if the binding cannot be removed.
     * @throws IllegalLifeCycleException
     *             if this component has a {@link LifeCycleController}
     *             interface, but it is not in an appropriate state to perform
     *             this operation.
     */
    public void unbindFc(String clientItfName) throws NoSuchInterfaceException,
            IllegalBindingException, IllegalLifeCycleException {

        Logger.println(DebugReflex.binding,
                "[BindingController] unbindFc : client = " + clientItfName);

        // calls the overriden unbindFc method
        _super_unbindFc(clientItfName);

        // notify the dual level
        if (_this_weaveableOptRC != null) {
            if (_this_weaveableOptRC.getIsNotifiable()) {
                try {
                    // get the binding-notification-controller interface of the
                    // dual
                    // component
                    BindingNotificationController dualCtrl = Reflex
                            .getBindingNotificationController(_this_weaveableOptRC
                                    .getCmpRef());

                    dualCtrl.unbindFcNotification(clientItfName);
                } catch (Exception e) {
                    Logger
                            .println(DebugReflex.warning,
                                    "[Warning][BindingController] unbindFc : cannot update dual level");
                }
            }
        } else {
            Logger.println(DebugReflex.warning,
                    "[Warning][BindingController] unbindFc : no dual level");
        }
    }

    // ------------------------------------------------------------------------
    // Implementation of BindingNotificationController interface
    // ------------------------------------------------------------------------

    /**
     * Calls the overriden bindFc method.
     * 
     * @param clientItfName
     *            the name of a client interface of the component to which this
     *            interface belongs.
     * @param serverItf
     *            a server interface.
     * @throws NoSuchInterfaceException
     *             if there is no such client interface.
     * @throws IllegalBindingException
     *             if the binding cannot be created.
     * @throws IllegalLifeCycleException
     *             if this component has a {@link LifeCycleController}
     *             interface, but it is not in an appropriate state to perform
     *             this operation.
     */
    public void bindFcNotification(String clientItfName, Object serverItf)
            throws NoSuchInterfaceException, IllegalBindingException,
            IllegalLifeCycleException {

        Logger
                .println(
                        DebugReflex.binding,
                        "[Notification][BindingController] bindFc : clt -| clt-itf -> srv-itf |- srv = "
                                + _this_weaveableOptNC.getFcName()
                                + " -| "
                                + clientItfName
                                + " -> "
                                + ((Interface) serverItf).getFcItfName()
                                + " |- "
                                + Fractal
                                        .getNameController(
                                                ((Interface) serverItf)
                                                        .getFcItfOwner())
                                        .getFcName());

        // calls the overriden bindFc method
        _super_bindFc(clientItfName, serverItf);
    }

    /**
     * Calls the overriden unbindFc method.
     * 
     * @param clientItfName
     *            the name of a client interface of the component to which this
     *            interface belongs.
     * @throws NoSuchInterfaceException
     *             if there is no such client interface.
     * @throws IllegalBindingException
     *             if the binding cannot be removed.
     * @throws IllegalLifeCycleException
     *             if this component has a {@link LifeCycleController}
     *             interface, but it is not in an appropriate state to perform
     *             this operation.
     */
    public void unbindFcNotification(String clientItfName)
            throws NoSuchInterfaceException, IllegalBindingException,
            IllegalLifeCycleException {

        Logger.println(DebugReflex.binding,
                "[Notification][BindingController] unbindFc : client = "
                        + clientItfName);

        // calls the overriden unbindFc method
        _super_unbindFc(clientItfName);
    }

    // -------------------------------------------------------------------------
    // Fields and methods required by the mixin class in the base class
    // -------------------------------------------------------------------------

    /**
     * The {@link Component} interface of the component to which this controller
     * object belongs.
     */

    public NameController _this_weaveableOptNC;

    /**
     * The <tt>weaveableOptRC</tt> field required by this mixin. This field is
     * supposed to reference the {@link ReflexController} interface of the
     * component to which this controller object belongs.
     */

    public ReflexController _this_weaveableOptRC;

    /**
     * The {@link BindingController#bindFc bindFc} method overriden by this
     * mixin.
     * 
     * @param clientItfName
     *            the name of a client interface of the component to which this
     *            interface belongs.
     * @param serverItf
     *            a server interface.
     * @throws NoSuchInterfaceException
     *             if there is no such client interface.
     * @throws IllegalBindingException
     *             if the binding cannot be created.
     * @throws IllegalLifeCycleException
     *             if this component has a {@link LifeCycleController}
     *             interface, but it is not in an appropriate state to perform
     *             this operation.
     */

    public abstract void _super_bindFc(String clientItfName, Object serverItf)
            throws NoSuchInterfaceException, IllegalBindingException,
            IllegalLifeCycleException;

    /**
     * The {@link BindingController#unbindFc unbindFc} method overriden by this
     * mixin.
     * 
     * @param clientItfName
     *            the name of a client interface of the component to which this
     *            interface belongs.
     * @throws NoSuchInterfaceException
     *             if there is no such client interface.
     * @throws IllegalBindingException
     *             if the binding cannot be removed.
     * @throws IllegalLifeCycleException
     *             if this component has a {@link LifeCycleController}
     *             interface, but it is not in an appropriate state to perform
     *             this operation.
     */

    public abstract void _super_unbindFc(String clientItfName)
            throws NoSuchInterfaceException, IllegalBindingException,
            IllegalLifeCycleException;
}
