/***
 * Reflex-Fractal
 *
 * Copyright (C) 2007 : INRIA - Domaine de Voluceau, Rocquencourt, B.P. 105, 
 * 78153 Le Chesnay Cedex - France 
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact: jade <AT> inrialpes <DOT> fr
 *
 * Author: SARDES project - http://sardes.inrialpes.fr
 *
 */
 
package org.ow2.jasmine.jade.reflex.control.content;

import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.objectweb.fractal.api.control.ContentController;
import org.objectweb.fractal.api.control.IllegalContentException;
import org.objectweb.fractal.api.control.IllegalLifeCycleException;
import org.objectweb.fractal.api.control.LifeCycleController;
import org.objectweb.fractal.util.Fractal;

import org.ow2.jasmine.jade.reflex.api.control.ContentNotificationController;
import org.ow2.jasmine.jade.reflex.api.control.ReflexController;
import org.ow2.jasmine.jade.reflex.util.DebugReflex;
import org.ow2.jasmine.jade.reflex.util.Logger;
import org.ow2.jasmine.jade.reflex.util.Reflex;

/**
 * Provides an implementation of the
 * {@link org.objectweb.fractal.api.control.ContentController} interface which
 * manages the causal connection with the dual level (i.e meta/execution) : any
 * content operation performed on the current component is notified to the
 * content controller of dual component.
 * 
 * @author <a href="mailto:fabienne.boyer@inrialpes.fr">Fabienne Boyer</a>
 */

public abstract class ReflexContentMixin implements ContentController,
		ContentNotificationController {

	// -------------------------------------------------------------------------
	// Private constructor
	// -------------------------------------------------------------------------

	private ReflexContentMixin() {
	}

	// -------------------------------------------------------------------------
	// Fields and methods added and overriden by the mixin class
	// -------------------------------------------------------------------------

	// ------------------------------------------------------------------------
	// Implementation of ContentController interface
	// ------------------------------------------------------------------------

	/**
	 * Calls the overriden addFcSubComponent method and then notifies the dual
	 * level of the content modification.
	 * 
	 * @param subComp
	 *            the component to be added inside this component.
	 * @throws IllegalContentException
	 *             if the given component cannot be added inside this component.
	 * @throws IllegalLifeCycleException
	 *             if this component has a {@link LifeCycleController}
	 *             interface, but it is not in an appropriate state to perform
	 *             this operation.
	 */

	public void addFcSubComponent(final Component subComponent)
			throws IllegalContentException, IllegalLifeCycleException {

		Logger.println(DebugReflex.content,
				"[ContentController] addFcSubComponent");

		// calls the overriden addFcSubComponent method
		_super_addFcSubComponent(subComponent);

		// notify the dual level
		if (_this_weaveableOptRC != null) {
			if (_this_weaveableOptRC.getIsNotifiable()) {
				try {
					// get the content-notification-controller interface of the
					// dual
					// component
					ContentNotificationController dualCtrl = Reflex
							.getContentNotificationController(_this_weaveableOptRC
									.getCmpRef());
					// notify the dual controller
					dualCtrl.addFcSubComponentNotification(Reflex
							.getDualComponent(subComponent));

				} catch (IllegalContentException ignored) {
					/*
					 * case of a cycle
					 */
				} catch (Exception e) {
					Logger
							.println(DebugReflex.warning,
									"[Warning][ContentController] addFcSubComponent : cannot update dual level");
				}
			}
		} else {
			Logger
					.println(DebugReflex.warning,
							"[Warning][ContentController] addFcSubComponent : no dual level");
		}
	}

	/**
	 * Calls the overriden method and then notifies the dual level of the
	 * content modification.
	 * 
	 * @param subComponent
	 *            the component to be removed from this component.
	 * @throws IllegalContentException
	 *             if the given component cannot be removed from this component.
	 * @throws IllegalLifeCycleException
	 *             if this component has a {@link LifeCycleController}
	 *             interface, but it is not in an appropriate state to perform
	 *             this operation.
	 */

	public void removeFcSubComponent(final Component subComponent)
			throws IllegalContentException, IllegalLifeCycleException {

		Logger.println(DebugReflex.content,
				"[ContentController] removeFcSubComponent");

		// calls the overriden removeFcSubComponent method
		_super_removeFcSubComponent(subComponent);

		// notify the dual level
		if (_this_weaveableOptRC != null) {
			if (_this_weaveableOptRC.getIsNotifiable()) {
				try {
					// get the content-notification-controller interface of the
					// dual
					// component
					ContentNotificationController dualCtrl = Reflex
							.getContentNotificationController(_this_weaveableOptRC
									.getCmpRef());

					// notify the dual controller
					dualCtrl.removeFcSubComponentNotification(Reflex
							.getDualComponent(subComponent));
				} catch (Exception e) {
					Logger
							.println(DebugReflex.warning,
									"[Warning][ContentController] removeFcSubComponent : cannot update dual level");
				}
			}
		} else {
			Logger
					.println(DebugReflex.warning,
							"[Warning][ContentController] removeFcSubComponent : cannot update dual level");
		}
	}

	// ------------------------------------------------------------------------
	// Implementation of ContentNotificationController interface
	// ------------------------------------------------------------------------

	/**
	 * Calls the overriden addFcSubComponent method
	 * 
	 * @param subComp
	 *            the component to be added inside this component.
	 * @throws IllegalContentException
	 *             if the given component cannot be added inside this component.
	 * @throws IllegalLifeCycleException
	 *             if this component has a {@link LifeCycleController}
	 *             interface, but it is not in an appropriate state to perform
	 *             this operation.
	 */
	public void addFcSubComponentNotification(Component subComponent)
			throws IllegalContentException, IllegalLifeCycleException {

		Logger.println(DebugReflex.content,
				"[Notification][ContentController] addFcSubComponent");

		/*
		 * FIXME old-dirty hack : avoid to have managed_resources in SR
		 */

		try {

			Component parents[] = Fractal.getSuperController(subComponent)
					.getFcSuperComponents();

			for (Component parent : parents) {

				if ("system_representation".equals(Fractal.getNameController(
						parent).getFcName())) {

					Fractal.getContentController(parent).removeFcSubComponent(
							subComponent);
					break;
				}
			}

		} catch (NoSuchInterfaceException ignored) {
			// ignored.printStackTrace();
		} catch (IllegalLifeCycleException ignored) {
			// ignored.printStackTrace();
		} catch (IllegalContentException ignored) {
			// ignored.printStackTrace();
		}

		// calls the overriden addFcSubComponent method
		_super_addFcSubComponent(subComponent);
	}

	/**
	 * Calls the overriden removeFcSubComponent method
	 * 
	 * @param subComp
	 *            the component to be removed inside this component.
	 * @throws IllegalContentException
	 *             if the given component cannot be removed inside this
	 *             component.
	 * @throws IllegalLifeCycleException
	 *             if this component has a {@link LifeCycleController}
	 *             interface, but it is not in an appropriate state to perform
	 *             this operation.
	 */
	public void removeFcSubComponentNotification(Component subComponent)
			throws IllegalContentException, IllegalLifeCycleException {

		Logger.println(DebugReflex.content,
				"[Notification][ContentController] removeFcSubComponent");

		// calls the overriden removeFcSubComponent method
		_super_removeFcSubComponent(subComponent);
	}

	// -------------------------------------------------------------------------
	// Fields and methods required by the mixin class in the base class
	// -------------------------------------------------------------------------

	/**
	 * The <tt>weaveableOptRC</tt> field required by this mixin. This field is
	 * supposed to reference the {@link ReflexController} interface of the
	 * component to which this controller object belongs.
	 */

	public ReflexController _this_weaveableOptRC;

	/**
	 * The {@link ContentController#addFcSubComponent addFcSubComponent} method
	 * overriden by this mixin.
	 * 
	 * @param subComponent
	 *            the component to be added inside this component.
	 * @throws IllegalContentException
	 *             if the given component cannot be added inside this component.
	 * @throws IllegalLifeCycleException
	 *             if this component has a {@link LifeCycleController}
	 *             interface, but it is not in an appropriate state to perform
	 *             this operation.
	 */

	public abstract void _super_addFcSubComponent(Component subComponent)
			throws IllegalContentException, IllegalLifeCycleException;

	/**
	 * The {@link ContentController#removeFcSubComponent removeFcSubComponent}
	 * method overriden by this mixin.
	 * 
	 * @param subComponent
	 *            the component to be removed from this component.
	 * @throws IllegalContentException
	 *             if the given component cannot be removed from this component.
	 * @throws IllegalLifeCycleException
	 *             if this component has a {@link LifeCycleController}
	 *             interface, but it is not in an appropriate state to perform
	 *             this operation.
	 */

	public abstract void _super_removeFcSubComponent(Component subComponent)
			throws IllegalContentException, IllegalLifeCycleException;
}
