/***
 * Reflex-Fractal
 *
 * Copyright (C) 2007 : INRIA - Domaine de Voluceau, Rocquencourt, B.P. 105, 
 * 78153 Le Chesnay Cedex - France 
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact: jade <AT> inrialpes <DOT> fr
 *
 * Author: SARDES project - http://sardes.inrialpes.fr
 *
 */
 
package org.ow2.jasmine.jade.reflex.control.lifecycle;

import org.objectweb.fractal.api.control.IllegalLifeCycleException;
import org.objectweb.fractal.api.control.LifeCycleController;
import org.objectweb.fractal.julia.Controller;
import org.objectweb.fractal.julia.control.lifecycle.LifeCycleCoordinator;

import org.ow2.jasmine.jade.reflex.api.control.LifeCycleNotificationController;
import org.ow2.jasmine.jade.reflex.api.control.ReflexController;
import org.ow2.jasmine.jade.reflex.util.DebugReflex;
import org.ow2.jasmine.jade.reflex.util.Logger;
import org.ow2.jasmine.jade.reflex.util.Reflex;

/**
 * Provides an implementation of the
 * {@link org.objectweb.fractal.api.control.LifeCycleController} interface which
 * manages the causal connection with the dual level (i.e meta/execution): any
 * lifecycle operation performed on the current component is notified to the
 * lifecycle controller of dual component.
 * 
 * @author <a href="mailto:fabienne.boyer@inrialpes.fr">Fabienne Boyer</a>
 */

public abstract class ReflexLifeCycleMixin implements Controller,
        LifeCycleCoordinator, LifeCycleNotificationController {

    private ReflexLifeCycleMixin() {
    }

    // ------------------------------------------------------------------------
    // Implementation of LifecycleCoordinator interface
    // ------------------------------------------------------------------------

    /**
     * Calls the overriden startFc method and then notifies the dual level of
     * the lifecycle operation.
     * 
     * @throws IllegalLifeCycleException
     *             if this component has a {@link LifeCycleController}
     *             interface, but it is not in an appropriate state to perform
     *             this operation.
     */
    public void startFc() throws IllegalLifeCycleException {

        Logger.println(DebugReflex.lifecycle, "[LifeCycleController] startFc");

        // calls the overriden startFc method
        _super_startFc();

        if (_this_weaveableOptRC != null) {
            if (_this_weaveableOptRC.getIsNotifiable()) {
                try {

                    // get the lifecycle-notification-controller interface of
                    // the
                    // dual component
                    LifeCycleNotificationController dualCtrl = Reflex
                            .getLifeCycleNotificationController(_this_weaveableOptRC
                                    .getCmpRef());

                    // notify the dual level
                    dualCtrl.startFcNotification();
                } catch (Exception e) {
                    Logger
                            .println(DebugReflex.warning,
                                    "[Warning][LifeCycleController] startFc : cannot update dual level");
                }
            }
        } else {
            Logger.println(DebugReflex.warning,
                    "[Warning][LifeCycleController] startFc : no dual level");
        }
    }

    /**
     * Calls the overriden stopFc method and then notifies the dual level of the
     * lifecycle operation.
     * 
     * @throws IllegalLifeCycleException
     *             if this component has a {@link LifeCycleController}
     *             interface, but it is not in an appropriate state to perform
     *             this operation.
     */
    public void stopFc() throws IllegalLifeCycleException {

        Logger.println(DebugReflex.lifecycle, "[LifeCycleController] stopFc");

        // calls the overriden stopFcmethod
        _super_stopFc();

        if (_this_weaveableOptRC != null) {
            if (_this_weaveableOptRC.getIsNotifiable()) {
                try {
                    // get the lifecycle-notification-controller interface of
                    // the
                    // dual component
                    LifeCycleNotificationController dualCtrl = Reflex
                            .getLifeCycleNotificationController(_this_weaveableOptRC
                                    .getCmpRef());

                    // notify the dual level
                    dualCtrl.stopFcNotification();
                } catch (Exception e) {
                    Logger
                            .println(DebugReflex.warning,
                                    "[Warning][LifeCycleController] stopFc : cannot update dual level");
                }
            }
        } else {
            Logger.println(DebugReflex.warning,
                    "[Warning][LifeCycleController] stopFc : no dual level");
        }
    }

    // ------------------------------------------------------------------------
    // Implementation of LifecycleNotificationController interface
    // ------------------------------------------------------------------------

    /**
     * Calls the overriden startFc method.
     * 
     * @throws IllegalLifeCycleException
      *             if this component has a {@link LifeCycleController}
     *             interface, but it is not in an appropriate state to perform
     *             this operation.
     */
    public void startFcNotification() throws IllegalLifeCycleException {

        Logger.println(DebugReflex.lifecycle,
                "[Notification][LifeCycleController] startFc");
        
        _super_startFc();

    }

    /**
     * Calls the overriden stopFc method.
     * 
     * @throws IllegalLifeCycleException
     *             if this component has a {@link LifeCycleController}
     *             interface, but it is not in an appropriate state to perform
     *             this operation.
     */
    public void stopFcNotification() throws IllegalLifeCycleException {

        Logger.println(DebugReflex.lifecycle,
                "[Notification][LifeCycleController] stopFc");
        
        _super_stopFc();

    }

    // -------------------------------------------------------------------------
    // Fields and methods required by the mixin class in the base class
    // -------------------------------------------------------------------------

    /**
     * The <tt>weaveableOptRC</tt> field required by this mixin. This field is
     * supposed to reference the {@link ReflexController} interface of the
     * component to which this controller object belongs.
     */

    public ReflexController _this_weaveableOptRC;

    /**
     * The {@link LifeCycleController#startFc startFc} method overriden by this
     * mixin.
     * 
     * @throws IllegalLifeCycleException
     *             if a problem occurs.
     */

    public abstract void _super_startFc() throws IllegalLifeCycleException;

    /**
     * The {@link LifeCycleController#startFc stopFc} method overriden by this
     * mixin.
     * 
     * @throws IllegalLifeCycleException
     *             if a problem occurs.
     */

    public abstract void _super_stopFc() throws IllegalLifeCycleException;
}