/***
 * Reflex-Fractal
 *
 * Copyright (C) 2007 : INRIA - Domaine de Voluceau, Rocquencourt, B.P. 105, 
 * 78153 Le Chesnay Cedex - France 
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact: jade <AT> inrialpes <DOT> fr
 *
 * Author: SARDES project - http://sardes.inrialpes.fr
 *
 */
 
package org.ow2.jasmine.jade.reflex.factory;

import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.objectweb.fractal.api.Type;
import org.objectweb.fractal.api.control.BindingController;
import org.objectweb.fractal.api.control.ContentController;
import org.objectweb.fractal.api.control.NameController;
import org.objectweb.fractal.api.factory.GenericFactory;
import org.objectweb.fractal.api.factory.InstantiationException;
import org.objectweb.fractal.api.type.ComponentType;
import org.objectweb.fractal.api.type.InterfaceType;
import org.objectweb.fractal.deployment.local.api.GenericInstallingFactory;
import org.objectweb.fractal.deployment.local.api.PackageDescription;
import org.objectweb.fractal.julia.InitializationContext;
import org.objectweb.fractal.rmi.registry.NamingService;
import org.objectweb.fractal.util.Fractal;

import org.ow2.jasmine.jade.reflex.api.control.ReflexController;
import org.ow2.jasmine.jade.reflex.api.factory.GenericFactoryNotification;
import org.ow2.jasmine.jade.reflex.util.DebugReflex;
import org.ow2.jasmine.jade.reflex.util.Environment;
import org.ow2.jasmine.jade.reflex.util.Logger;
import org.ow2.jasmine.jade.reflex.util.Reflex;

/**
 * Provides an implementation of the
 * {@link org.objectweb.fractal.api.factory.GenericFactory} interface which
 * manages the causal connection with the dual (i.e meta/execution) level : any
 * newFcInstance operation performed on the current factory is forwarded to the
 * factory of the dual level.
 * 
 * @author <a href="mailto:fabienne.boyer@inrialpes.fr">Fabienne Boyer</a>
 * contributor : <a href="mailto:julien.legrand@inrialpes.fr">Julien Legrand</a>
 * 
 * More informations about this class: - define this class as a mixin which can
 * be added to any GenericFactory in order to make it reflexive - A "reflexive"
 * factory is a standard factory except that it has a specific additionnal
 * controller (ReflexController) which allows to get the reference of its dual
 * factory - A "reflexive" factory ensures that for any component it creates, it
 * notifies its dual factory such that a dual component is automatically
 * created. - the level of the current factory is set according to the
 * reflex-fractal.level system property - at instanciation time: - the factory
 * automatically register itself under the naming service (by using as name the
 * concatenation of the name given by the reflex-fractal.name system property
 * and the level), - the factory automatically connects itself with its dual
 * factory
 * 
 */
public abstract class ReflexGenericFactoryMixin implements GenericFactory,
		GenericInstallingFactory, GenericFactoryNotification {

	private boolean isInitialized = false;

	private NamingService ns = null;

	// -------------------------------------------------------------------------
	// PUBLIC constructor (needed for bootstrap)
	// -------------------------------------------------------------------------

	private ReflexGenericFactoryMixin() {
	}

	// -------------------------------------------------------------------------
	// Fields and methods added and overriden by the mixin class
	// -------------------------------------------------------------------------

	public void initFcController(final InitializationContext ic)
			throws InstantiationException {

		// calls the overriden initFcController method
		_super_initFcController(ic);
	}

	// -------------------------------------------------------------------------
	// Implementation of GenericFactory interface
	// -------------------------------------------------------------------------

	/**
	 * Calls the overriden newFcInstance method and then notifies the dual
	 * factory of the newFcInstance operation.
	 * 
	 * @param type
	 *            an arbitrary component type.
	 * @param controllerDesc
	 *            a description of the controller part of the component to be
	 *            created. This description is implementation specific. If it is
	 *            <tt>null</tt> then a "default" controller part will be used.
	 * @param contentDesc
	 *            a description of the content part of the component to be
	 *            created. This description is implementation specific. It may
	 *            be <tt>null</tt> to create component with an empty initial
	 *            content. It may also be, in Java, the fully qualified name of
	 *            a Java class, to create primitive components.
	 * @return the {@link Component} interface of the created component.
	 * @throws InstantiationException
	 *             if the component cannot be created.
	 */

	public Component newFcInstance(final Type type,
			final Object controllerDesc, final Object contentDesc)
			throws InstantiationException {

		if (ns == null) {
			init();
			isInitialized = true;
		}

		Component cmp = null;

		// if (contentDesc != null)
		Logger.println(DebugReflex.genericfactory,
				"[ReflexGenericFactory] newFcInstance : " + contentDesc);

		// if this factory operates in reflex mode
		if (_this_weaveableOptRC != null) {

			try {

				/*
				 * get the controller descriptor
				 */
				String cd = null;
				if (controllerDesc instanceof Object[]) {
					cd = (String) (((Object[]) controllerDesc)[1]);
				} else {
					cd = (String) controllerDesc;
				}

				/*
				 * generate the controller descriptor
				 */
				String newControllerDesc = cd;
				if (cd.startsWith("jade-")) {
					newControllerDesc = Reflex.controllersPrefix
							+ newControllerDesc;
				}

				/*
				 * create a component associated to reflexive controllers
				 */
				if (controllerDesc instanceof Object[]) {
					cmp = _super_newFcInstance(type,
							new Object[] { ((Object[]) controllerDesc)[0],
									newControllerDesc }, contentDesc);
				} else {
					cmp = _super_newFcInstance(type, newControllerDesc,
							contentDesc);
				}

				try {

					// Set the contentDesc and controllerDesc attributes
					ReflexController rc = Reflex.getReflexController(cmp);

					rc.setReflexAttribute("controllerDesc", (String) (cd));
					rc
							.setReflexAttribute("contentDesc",
									(String) (contentDesc));

					// rc.setIsNotifiable(_this_weaveableOptRC.getIsNotifiable());

					/*
					 * get the auto property
					 */
					Boolean auto = new Boolean(Environment
							.getProperty(Environment.reflexAuto));

					if (auto) {

						if (_this_weaveableOptRC.getCmpRef() != null) {

							/*
							 * get the notification controller interface of the
							 * dual factory
							 */
							GenericFactoryNotification dualCtrl = Reflex
									.getGenericFactoryNotification(_this_weaveableOptRC
											.getCmpRef());
							/*
							 * notify the dual factory of the newFcInstance
							 * operation
							 */
							Component dualCmp = null;
							// .newFcInstanceNotification(type, cd, null);
							if (controllerDesc instanceof Object[]) {
								dualCmp = dualCtrl.newFcInstanceNotification(
										type, new Object[] {
												((Object[]) controllerDesc)[0],
												newControllerDesc }, null);
							} else {
								dualCmp = dualCtrl.newFcInstanceNotification(
										type, newControllerDesc, null);
							}
							/*
							 * set the component's dual reference
							 */
							rc.setCmpRef(dualCmp);
							/*
							 * set the dual reference of the dual component
							 */
							Reflex.getReflexController(dualCmp).setCmpRef(cmp);
							/*
							 * add the new component as a sub-component
							 */
							// Component metaComponentContainer = Reflex
							// .getReflexController(
							// _this_weaveableOptRC.getCmpRef())
							// .getMetaComponentContainer();
							//
							// if (metaComponentContainer == null) {
							// ContentNotificationController cc = Reflex
							// .getContentNotificationController(_this_weaveableC);
							// cc.addFcSubComponentNotification(cmp);
							// } else {
							// Fractal.getContentController(
							// metaComponentContainer)
							// .addFcSubComponent(cmp);
							// }
						} else {
							Logger
									.println(DebugReflex.warning,
											"[Warning][ReflexGenericFactor] newFcInstance : No dual factory found");
						}
					}
				} catch (Exception noReflexControllerFound) {
					// noReflexControllerFound.printStackTrace();
				}
			} catch (Exception e) {

				e.printStackTrace();

				Logger.println(DebugReflex.warning,
						"[Warning][ReflexGenericFactory] newFcInstance : cannot update dual level"
								+ e);
			}
		} else {
			/*
			 * calls the overriden newFcInstance method with the standard
			 * controllers
			 */
			cmp = _super_newFcInstance(type, controllerDesc, contentDesc);
		}

		Logger.println(DebugReflex.genericfactory,
				"[ReflexGenericFactory] newFcInstance : cmp = "
						+ cmp.toString());
		return cmp;
	}

	// -------------------------------------------------------------------------
	// Implementation of GenericInstallingFactory interface
	// -------------------------------------------------------------------------

	public Component newFcInstance(Type type, Object controllerDesc, Object contentDesc,
			PackageDescription pkg) throws InstantiationException {

		if (ns == null) {
			init();
			isInitialized = true;
		}

		Component cmp = null;

		// if (contentDesc != null)
		Logger.println(DebugReflex.genericfactory,
				"[ReflexGenericFactory] newFcInstance : " + contentDesc);

		// if this factory operates in reflex mode
		if (_this_weaveableOptRC != null) {

			try {

				/*
				 * get the controller descriptor
				 */
				String cd = null;
				if (controllerDesc instanceof Object[]) {
					cd = (String) (((Object[]) controllerDesc)[1]);
				} else {
					cd = (String) controllerDesc;
				}

				/*
				 * generate the controller descriptor
				 */
				String newControllerDesc = cd;
				if (cd.startsWith("jade-")) {
					newControllerDesc = Reflex.controllersPrefix
							+ newControllerDesc;
				}

				/*
				 * create a component associated to reflexive controllers
				 */
				if (controllerDesc instanceof Object[]) {
					cmp = _super_newFcInstance(type,
							new Object[] { ((Object[]) controllerDesc)[0],
									newControllerDesc }, contentDesc, pkg);
				} else {
					cmp = _super_newFcInstance(type, newControllerDesc,
							contentDesc, pkg);
				}

				try {

					// Set the contentDesc and controllerDesc attributes
					ReflexController rc = Reflex.getReflexController(cmp);

					rc.setReflexAttribute("controllerDesc", (String) (cd));
					rc
							.setReflexAttribute("contentDesc",
									(String) (contentDesc));

					// rc.setIsNotifiable(_this_weaveableOptRC.getIsNotifiable());

					/*
					 * get the auto property
					 */
					Boolean auto = new Boolean(Environment
							.getProperty(Environment.reflexAuto));

					if (auto) {

						if (_this_weaveableOptRC.getCmpRef() != null) {

							/*
							 * get the notification controller interface of the
							 * dual factory
							 */
							GenericFactoryNotification dualCtrl = Reflex
									.getGenericFactoryNotification(_this_weaveableOptRC
											.getCmpRef());
							/*
							 * notify the dual factory of the newFcInstance
							 * operation
							 */
							Component dualCmp = null;
							// .newFcInstanceNotification(type, cd, null);
							if (controllerDesc instanceof Object[]) {
								dualCmp = dualCtrl.newFcInstanceNotification(
										type, new Object[] {
												((Object[]) controllerDesc)[0],
												newControllerDesc }, null);
							} else {
								dualCmp = dualCtrl.newFcInstanceNotification(
										type, newControllerDesc, null);
							}
							/*
							 * set the component's dual reference
							 */
							rc.setCmpRef(dualCmp);
							/*
							 * set the dual reference of the dual component
							 */
							Reflex.getReflexController(dualCmp).setCmpRef(cmp);
							/*
							 * add the new component as a sub-component
							 */
							// Component metaComponentContainer = Reflex
							// .getReflexController(
							// _this_weaveableOptRC.getCmpRef())
							// .getMetaComponentContainer();
							//
							// if (metaComponentContainer == null) {
							// ContentNotificationController cc = Reflex
							// .getContentNotificationController(_this_weaveableC);
							// cc.addFcSubComponentNotification(cmp);
							// } else {
							// Fractal.getContentController(
							// metaComponentContainer)
							// .addFcSubComponent(cmp);
							// }
						} else {
							Logger
									.println(DebugReflex.warning,
											"[Warning][ReflexGenericFactor] newFcInstance : No dual factory found");
						}
					}
				} catch (Exception noReflexControllerFound) {
					// noReflexControllerFound.printStackTrace();
				}
			} catch (Exception e) {

				e.printStackTrace();

				Logger.println(DebugReflex.warning,
						"[Warning][ReflexGenericFactory] newFcInstance : cannot update dual level"
								+ e);
			}
		} else {
			/*
			 * calls the overriden newFcInstance method with the standard
			 * controllers
			 */
			cmp = _super_newFcInstance(type, controllerDesc, contentDesc);
		}

		Logger.println(DebugReflex.genericfactory,
				"[ReflexGenericFactory] newFcInstance : cmp = "
						+ cmp.toString());
		return cmp;
	}

//	public void undeployFcComponent(Component arg0) {
//		// TODO Auto-generated method stub
//
//	}

	// -------------------------------------------------------------------------
	// Implementation of GenericFactoryNotification interface
	// -------------------------------------------------------------------------

	/**
	 * Calls the overriden newFcInstance method.
	 * 
	 * @param type
	 *            an arbitrary component type.
	 * @param controllerDesc
	 *            a description of the controller part of the component to be
	 *            created. This description is implementation specific. If it is
	 *            <tt>null</tt> then a "default" controller part will be used.
	 * @param contentDesc
	 *            a description of the content part of the component to be
	 *            created. This description is implementation specific. It may
	 *            be <tt>null</tt> to create component with an empty initial
	 *            content. It may also be, in Java, the fully qualified name of
	 *            a Java class, to create primitive components.
	 * @return the {@link Component} interface of the created component.
	 * @throws InstantiationException
	 *             if the component cannot be created.
	 */

	public Component newFcInstanceNotification(Type type,
			Object controllerDesc, Object contentDesc)
			throws org.objectweb.fractal.api.factory.InstantiationException {

		if (!isInitialized) {
			init();
			isInitialized = true;
		}

		Component cmp = null;

		Logger.println(DebugReflex.genericfactory,
				"[Notification][ReflexGenericFactory] newFcInstance : "
						+ contentDesc);

		/*
		 * if this factory operates in reflex mode
		 */
		if (_this_weaveableOptRC != null) {

			try {

				/*
				 * get the controller descriptor
				 */
				String cd = null;
				if (controllerDesc instanceof Object[]) {
					cd = (String) (((Object[]) controllerDesc)[1]);
				} else {
					cd = (String) controllerDesc;
				}

				/*
				 * generate the controller descriptor
				 */
				String newControllerDesc = cd;
				if (cd.startsWith("jade-")) {
					newControllerDesc = Reflex.controllersPrefix
							+ newControllerDesc;
				}

				/*
				 * calls the overriden newFcInstance method with the reflexive
				 */
				if (controllerDesc instanceof Object[]) {
					cmp = _super_newFcInstance(type,
							new Object[] { ((Object[]) controllerDesc)[0],
									newControllerDesc }, contentDesc);
				} else {
					cmp = _super_newFcInstance(type, newControllerDesc,
							contentDesc);
				}

				System.out.println("Type : ");

				for (InterfaceType it : ((ComponentType) type)
						.getFcInterfaceTypes()) {
					System.out.println("\t - " + it.getFcItfName() + "("
							+ it.getFcItfSignature() + ")");
				}

				try {
					/*
					 * Set the contentDesc and controllerDesc attributes
					 */
					ReflexController rc = Reflex.getReflexController(cmp);

					rc.setReflexAttribute("controllerDesc",
							(String) (controllerDesc));
					rc
							.setReflexAttribute("contentDesc",
									(String) (contentDesc));

					// rc.setIsNotifiable(_this_weaveableOptRC.getIsNotifiable());

					/*
					 * add the new component as a sub-component
					 */

					/*
					 * This is not the good method because all components will
					 * be added as sub-cmp to the SR. But we just want that the
					 * composite component will.
					 */
					try {
						Component metaComponentContainer = _this_weaveableOptRC
								.getMetaComponentContainer();

						System.out.println(metaComponentContainer.toString());

						if (metaComponentContainer != null) {
							Fractal
									.getContentController(
											metaComponentContainer)
									.addFcSubComponent(cmp);
						}
					} catch (Exception ignored) {
					}

				} catch (Exception noReflexControllerFound) {
					// ignored
				}
			} catch (Exception e) {
				Logger.println(DebugReflex.on,
						"[ReflexGenericFactory] newFcInstanceNotification : cannot create instance "
								+ e);
				/*
				 * DEBUG begin 7 sept. 06 jlegrand
				 */
				e.printStackTrace();
				/*
				 * end
				 */
			}
		} else {
			Logger
					.println(DebugReflex.warning,
							"[Warning][ReflexGenericFactory] newFcInstanceNotification : no dual level");

			/*
			 * calls the overriden newFcInstance method with the standard
			 * controllers
			 */
			cmp = _super_newFcInstance(type, controllerDesc, contentDesc);
		}

		Logger.println(DebugReflex.genericfactory,
				"[ReflexGenericFactory] newFcInstance : cmp = "
						+ cmp.toString());
		return cmp;
	}

	// -------------------------------------------------------------------------
	// Private methods
	// -------------------------------------------------------------------------

	private void init() throws InstantiationException {
		/*
		 * if the factory operates in reflexive mode
		 */
		if (_this_weaveableOptRC != null) {

			try {

				/*
				 * get the dual jadenode
				 */
				String dualName = Environment
						.getProperty(Environment.reflexDualNameLabel);

				/*
				 * if dual-name is not defined, the bootstrap doesn't duplicate
				 * components
				 */
				if (dualName == null) {

					try {
						Logger
								.println(DebugReflex.genericfactory,
										"[ReflexGenericFactory] init: collocated meta & exec factories");
						/*
						 * set the dual references for the factory and for the
						 * dual factory (dual(cmp) = cmp)
						 */
						_this_weaveableOptRC.setCmpRef(Reflex
								.getComponent(_this_weaveableC));

						// _this_weaveableOptRC.setCmpRef(_this_weaveableC);

					} catch (Exception e) {
						Logger.println(DebugReflex.on,
								"[ReflexGenericFactory] init : cannot set dual references for factories"
										+ e);
						throw new InstantiationException(
								"Pb encountered when setting dual references for factories");
					}

				}

				/*
				 * The bootstrap duplicate components ant needs to get a ref of
				 * Jadeboot bootstrap
				 */
				else {

					/*
					 * get the Fractal RMI registry
					 */
					NamingService ns = (NamingService) _this_weaveableBC
							.lookupFc("registry");
					if (ns == null) {

						Logger
								.println(DebugReflex.on,
										"[ReflexGenericFactory] init : cannot get registry");
						throw new InstantiationException(
								"The registry is unreachable");
					}

					/*
					 * get the reference of the dual factory if it is already
					 * started
					 */
					Logger.println(DebugReflex.genericfactory,
							"[ReflexGenericFactory] : looking for the Fractal factory of "
									+ dualName);

					Component dualJadeNode = ns.lookup(dualName);
					Component controllers = null;
					ContentController cc = Fractal
							.getContentController(dualJadeNode);
					for (Component sc : cc.getFcSubComponents()) {
						try {
							NameController nc = Fractal.getNameController(sc);
							if ("controllers".equals(nc.getFcName())) {
								controllers = sc;
								continue;
							}
						} catch (NoSuchInterfaceException ignored) {
						}
					}

					Component dualBootstrap = null;
					cc = Fractal.getContentController(controllers);
					for (Component sc : cc.getFcSubComponents()) {
						try {
							NameController nc = Fractal.getNameController(sc);
							if (nc.getFcName().endsWith("_factory")) {
								dualBootstrap = sc;
								continue;
							}
						} catch (NoSuchInterfaceException ignored) {
						}
					}

					if (dualBootstrap != null) {
						try {
							Logger
									.println(DebugReflex.genericfactory,
											"[ReflexGenericFactory] init: setting dual references for factories");
							/*
							 * set the dual references for the factory and for
							 * the dual factory
							 */
							_this_weaveableOptRC.setCmpRef(dualBootstrap);

							/*
							 * DEBUG begin 24 mai 2006 jlegrand
							 */
							// ReflexController rcdual = Reflex
							// .getReflexController(dualBootstrap);
							// rcdual
							// .setCmpRef(Reflex
							// .getComponent((Component) _this_weaveableC));
							// rcdual.setCmpRef(_this_weaveableC);
							/*
							 * end
							 */

						} catch (Exception e) {
							Logger.println(DebugReflex.on,
									"[ReflexGenericFactory] init : cannot set dual references for factories"
											+ e);
							throw new InstantiationException(
									"Pb encountered when setting dual references for factories");
						}
					} else {
						Logger
								.println(
										DebugReflex.on,
										"[ReflexGenericFactory] init : cannot find the dual factory defined by the name "
												+ dualName);
						// throw new InstantiationException(
						// "Pb encountered when searching dual factory");
					}
				}
			} catch (Exception e) {
				Logger.println(DebugReflex.on, "[ReflexGenericFactory] init : "
						+ e);
			}
		}
	}

	// -------------------------------------------------------------------------
	// Fields and methods required by the mixin class in the base class
	// -------------------------------------------------------------------------

	/**
	 * The <tt>weaveableC</tt> field required by this mixin. This field is
	 * supposed to reference the {@link Component} interface of the component to
	 * which this controller object belongs.
	 */
	public Component _this_weaveableC;

	/**
	 * The <tt>weaveableBC</tt> field required by this mixin. This field is
	 * supposed to reference the {@link BindingController} interface of the
	 * component to which this controller object belongs.
	 */

	public BindingController _this_weaveableBC;

	/**
	 * The <tt>weaveableOptRC</tt> field required by this mixin. This field is
	 * supposed to reference the {@link ReflexController} interface of the
	 * component to which this controller object belongs.
	 */
	public ReflexController _this_weaveableOptRC;

	/**
	 * The {@link GenericFactory#newFcInstance newFcInstance} method overriden
	 * by this mixin.
	 * 
	 * @param type
	 *            an arbitrary component type.
	 * @param controllerDesc
	 *            a description of the controller part of the component to be
	 *            created. This description is implementation specific. If it is
	 *            <tt>null</tt> then a "default" controller part will be used.
	 * @param contentDesc
	 *            a description of the content part of the component to be
	 *            created. This description is implementation specific. It may
	 *            be <tt>null</tt> to create component with an empty initial
	 *            content. It may also be, in Java, the fully qualified name of
	 *            a Java class, to create primitive components.
	 * @return the {@link Component} interface of the created component.
	 * @throws InstantiationException
	 *             if the component cannot be created.
	 */

	public abstract Component _super_newFcInstance(Type type,
			Object controllerDesc, Object contentDesc)
			throws InstantiationException;
	
	public abstract Component _super_newFcInstance(Type type,
			Object controllerDesc, Object contentDesc, PackageDescription pkg)
			throws InstantiationException;

	public abstract void _super_initFcController(InitializationContext ic)
			throws InstantiationException;

}
