/***
 * Reflex-Fractal
 *
 * Copyright (C) 2007 : INRIA - Domaine de Voluceau, Rocquencourt, B.P. 105, 
 * 78153 Le Chesnay Cedex - France 
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact: jade <AT> inrialpes <DOT> fr
 *
 * Author: SARDES project - http://sardes.inrialpes.fr
 *
 */
 
package org.ow2.jasmine.jade.reflex.factory;

import java.util.Map;

import org.objectweb.fractal.api.Component;
import org.objectweb.fractal.api.NoSuchInterfaceException;
import org.objectweb.fractal.api.Type;
import org.objectweb.fractal.api.control.BindingController;
import org.objectweb.fractal.api.control.ContentController;
import org.objectweb.fractal.api.control.NameController;
import org.objectweb.fractal.api.factory.GenericFactory;
import org.objectweb.fractal.api.factory.InstantiationException;
import org.objectweb.fractal.deployment.local.api.GenericInstallingFactory;
import org.objectweb.fractal.deployment.local.api.PackageDescription;
import org.objectweb.fractal.julia.InitializationContext;
import org.objectweb.fractal.rmi.registry.NamingService;
import org.objectweb.fractal.util.Fractal;

import org.ow2.jasmine.jade.fractal.api.control.GenericInstallingFactoryContextController;
import org.ow2.jasmine.jade.fractal.util.FractalUtil;
import org.ow2.jasmine.jade.reflex.api.control.ReflexController;
import org.ow2.jasmine.jade.reflex.api.factory.GenericInstallingFactoryNotification;
import org.ow2.jasmine.jade.reflex.util.DebugReflex;
import org.ow2.jasmine.jade.reflex.util.Environment;
import org.ow2.jasmine.jade.reflex.util.Logger;
import org.ow2.jasmine.jade.reflex.util.Reflex;

/**
 * Provides an implementation of the
 * {@link org.objectweb.fractal.deployment.local.api.GenericInstallingFactory} interface which
 * manages the causal connection with the dual (i.e meta/execution) level : any
 * newFcInstance operation performed on the current factory is forwarded to the
 * factory of the dual level.
 * 
 * @author <a href="mailto:julien.legrand@inrialpes.fr">Julien Legrand</a>
 * 
 * More informations about this class: - define this class as a mixin which can
 * be added to any GenericFactory in order to make it reflexive - A "reflexive"
 * factory is a standard factory except that it has a specific additionnal
 * controller (ReflexController) which allows to get the reference of its dual
 * factory - A "reflexive" factory ensures that for any component it creates, it
 * notifies its dual factory such that a dual component is automatically
 * created. - the level of the current factory is set according to the
 * reflex-fractal.level system property - at instanciation time: - the factory
 * automatically register itself under the naming service (by using as name the
 * concatenation of the name given by the reflex-fractal.name system property
 * and the level), - the factory automatically connects itself with its dual
 * factory
 * 
 */
public abstract class ReflexGenericInstallingFactoryMixin implements
		GenericInstallingFactory, GenericInstallingFactoryNotification {

	/**
	 * 
	 */
	// private boolean isInitialized = false;
	/**
	 * 
	 */
	private NamingService ns = null;

	// -------------------------------------------------------------------------
	// Private constructor (needed for bootstrap)
	// -------------------------------------------------------------------------

	private ReflexGenericInstallingFactoryMixin() {
	}

	// -------------------------------------------------------------------------
	// Fields and methods added and overriden by the mixin class
	// -------------------------------------------------------------------------

	public void initFcController(final InitializationContext ic)
			throws InstantiationException {

		// calls the overriden initFcController method
		_super_initFcController(ic);
	}

	// -------------------------------------------------------------------------
	// Implementation of GenericInstallingFactory interface
	// -------------------------------------------------------------------------

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.objectweb.fractal.deployment.local.api.GenericInstallingFactory#undeployFcComponent(org.objectweb.fractal.api.Component)
	 */
	public void undeployFcComponent(Component component) {

		if (_this_weaveableOptRC != null) {

			try {

				init();

				/*
				 * undeploy the component associated to reflexive controllers
				 */
				_super_undeployFcComponent(component);

				if (_this_weaveableOptRC.getCmpRef() != null) {

					/*
					 * get the notification controller interface of the dual
					 * factory
					 */
					GenericInstallingFactoryNotification dualCtrl = Reflex
							.getGenericInstallingFactoryNotification(_this_weaveableOptRC
									.getCmpRef());
					/*
					 * notify the dual factory of the undeployFcComponent
					 * operation
					 */
					dualCtrl.undeployFcComponentNotification(Reflex
							.getDualComponent(component));
				}
			} catch (Exception e) {

//				e.printStackTrace();

				Logger
						.println(
								DebugReflex.warning,
								"[Warning][ReflexGenericInstallingFactory] undeployFcComponent : cannot update dual level"
										+ e);
			}
		} else {
			/*
			 * calls the overriden undeployFcComponent method with the standard
			 * controllers
			 */
			_super_undeployFcComponent(component);
		}

		Logger.println(DebugReflex.genericfactory,
				"[ReflexGenericInstallingFactory] undeployFcComponent : cmp = "
						+ component.toString());

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.objectweb.fractal.deployment.local.api.GenericInstallingFactory#newFcInstance(org.objectweb.fractal.api.Type,
	 *      java.lang.Object, java.lang.Object,
	 *      org.objectweb.fractal.deployment.local.api.PackageDescription)
	 */
	public Component newFcInstance(Type type, Object controllerDesc,
			Object contentDesc, PackageDescription pkg)
			throws InstantiationException {

		init();

		Component cmp = null;

		Logger.println(DebugReflex.genericfactory,
				"[ReflexGenericInstallingFactory] newFcInstance : "
						+ contentDesc);

		/*
		 * if this factory operates in reflex mode
		 */
		if (_this_weaveableOptRC != null) {

			try {

				/*
				 * get the controller descriptor
				 */
				String cd = getControllerDesc(controllerDesc);

				/*
				 * generate the controller descriptor
				 */
				String newControllerDesc = getNewControllerDesc(cd);

				/*
				 * create a component associated to reflexive controllers
				 */
				cmp = createComponent(type, controllerDesc, contentDesc, pkg,
						newControllerDesc);

				try {

					/*
					 * Set the contentDesc and controllerDesc attributes
					 */
					ReflexController rc = Reflex.getReflexController(cmp);

					rc.setReflexAttribute("controllerDesc", (String) cd);
					rc.setReflexAttribute("contentDesc", (String) contentDesc);

					/*
					 * get the auto property
					 */
					if (new Boolean(Environment
							.getProperty(Environment.reflexAuto))) {

						Component dualBootstrap = _this_weaveableOptRC
								.getCmpRef();

						if (dualBootstrap != null) {

							/*
							 * get the notification controller interface of the
							 * dual factory
							 */
							GenericInstallingFactoryNotification dualCtrl = Reflex
									.getGenericInstallingFactoryNotification(dualBootstrap);

							/*
							 * set the meta component container to
							 * meta-managed_resources
							 */
							Component previousNewComponentContainer = setNewComponentContainer(dualBootstrap);

							/*
							 * notify the dual factory of the newFcInstance
							 * operation
							 */
							Component dualCmp = createDualComponent(type,
									controllerDesc, pkg, newControllerDesc,
									dualCtrl);

							/*
							 * reset meta-component container
							 */
							FractalUtil
									.getGenericInstallingFactoryContextController(
											dualBootstrap)
									.setNewComponentContainer(
											previousNewComponentContainer);

							/*
							 * set the component's dual reference
							 */
							rc.setCmpRef(dualCmp);
							/*
							 * set the dual reference of the dual component
							 */
							Reflex.getReflexController(dualCmp).setCmpRef(cmp);

						} else {
							Logger
									.println(DebugReflex.warning,
											"[Warning][ReflexGenericFactor] newFcInstance : No dual factory found");
						}
					}
				} catch (Exception noReflexControllerFound) {
					// noReflexControllerFound.printStackTrace();
				}
			} catch (Exception e) {

//				e.printStackTrace();

				Logger
						.println(
								DebugReflex.warning,
								"[Warning][ReflexGenericInstallingFactory] newFcInstance : cannot update dual level"
										+ e);
			}
		} else {
			/*
			 * calls the overriden newFcInstance method with the standard
			 * controllers
			 */
			cmp = _super_newFcInstance(type, controllerDesc, contentDesc);
		}

		Logger.println(DebugReflex.genericfactory,
				"[ReflexGenericInstallingFactory] newFcInstance : cmp = "
						+ cmp.toString());
		Logger.println(DebugReflex.genericfactory,
				"--------------------------------------------");

		return cmp;
	}

	// -------------------------------------------------------------------------
	// Implementation of GenericFactory interface
	// -------------------------------------------------------------------------

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.objectweb.fractal.api.factory.GenericFactory#newFcInstance(org.objectweb.fractal.api.Type,
	 *      java.lang.Object, java.lang.Object)
	 */
	public Component newFcInstance(Type type, Object controllerDesc,
			Object contentDesc) throws InstantiationException {

		init();

		Component cmp = null;

		Logger.println(DebugReflex.genericfactory,
				"[ReflexGenericInstallingFactory] newFcInstance : "
						+ contentDesc);

		/*
		 * if this factory operates in reflex mode
		 */
		if (_this_weaveableOptRC != null) {

			try {

				/*
				 * get the controller descriptor
				 */
				String cd = null;
				cd = getControllerDesc(controllerDesc);

				/*
				 * generate the controller descriptor
				 */
				String newControllerDesc = getNewControllerDesc(cd);

				/*
				 * create a component associated to reflexive controllers
				 */
				cmp = createComponent(type, controllerDesc, contentDesc,
						newControllerDesc);

				try {

					/*
					 * Set the contentDesc and controllerDesc attributes
					 */
					ReflexController rc = Reflex.getReflexController(cmp);

					rc.setReflexAttribute("controllerDesc", (String) (cd));
					rc
							.setReflexAttribute("contentDesc",
									(String) (contentDesc));

					/*
					 * get the auto property
					 */
					if (new Boolean(Environment
							.getProperty(Environment.reflexAuto))) {

						Component dualBootstrap = _this_weaveableOptRC
								.getCmpRef();

						if (dualBootstrap != null) {

							/*
							 * get the notification controller interface of the
							 * dual factory
							 */
							GenericInstallingFactoryNotification dualCtrl = Reflex
									.getGenericInstallingFactoryNotification(dualBootstrap);

							/*
							 * set the meta component container to
							 * meta-managed_resources
							 */
							Component previousNewComponentContainer = setNewComponentContainer(dualBootstrap);

							/*
							 * notify the dual factory of the newFcInstance
							 * operation
							 */
							Component dualCmp = createDualComponent(type,
									controllerDesc, newControllerDesc, dualCtrl);

							/*
							 * reset meta-component container
							 */
							FractalUtil
									.getGenericInstallingFactoryContextController(
											dualBootstrap)
									.setNewComponentContainer(
											previousNewComponentContainer);

							/*
							 * set the component's dual reference
							 */
							rc.setCmpRef(dualCmp);
							/*
							 * set the dual reference of the dual component
							 */
							Reflex.getReflexController(dualCmp).setCmpRef(cmp);

						} else {
							Logger
									.println(DebugReflex.warning,
											"[Warning][ReflexGenericFactor] newFcInstance : No dual factory found");
						}
					}
				} catch (Exception noReflexControllerFound) {
					// noReflexControllerFound.printStackTrace();
				}
			} catch (Exception e) {

				e.printStackTrace();

				Logger
						.println(
								DebugReflex.warning,
								"[Warning][ReflexGenericInstallingFactory] newFcInstance : cannot update dual level"
										+ e);
			}
		} else {
			/*
			 * calls the overriden newFcInstance method with the standard
			 * controllers
			 */
			cmp = _super_newFcInstance(type, controllerDesc, contentDesc);
		}

		Logger.println(DebugReflex.genericfactory,
				"[ReflexGenericInstallingFactory] newFcInstance : cmp = "
						+ cmp.toString());
		Logger.println(DebugReflex.genericfactory,
				"--------------------------------------------");

		return cmp;

	}

	// -------------------------------------------------------------------------
	// Implementation of GenericInstallingFactoryNotification interface
	// -------------------------------------------------------------------------

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.ow2.jasmine.jade.reflex.api.factory.GenericInstallingFactoryNotification#undeployFcComponentNotification(org.objectweb.fractal.api.Component)
	 */
	public void undeployFcComponentNotification(Component cmp) {

		Logger.println(DebugReflex.lifecycle,
				"[Notification][GenericInstallingFactory] undeployFcComponent");

		_super_undeployFcComponent(cmp);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.ow2.jasmine.jade.reflex.api.factory.GenericInstallingFactoryNotification#newFcInstanceNotification(org.objectweb.fractal.api.Type,
	 *      java.lang.Object, java.lang.Object,
	 *      org.objectweb.fractal.deployment.local.api.PackageDescription)
	 */
	public Component newFcInstanceNotification(Type type,
			Object controllerDesc, Object contentDesc,
			PackageDescription packageDescription)
			throws InstantiationException {

		init();

		Component cmp = null;

		Logger.println(DebugReflex.genericfactory,
				"[Notification][ReflexGenericInstallingFactory] newFcInstance : "
						+ contentDesc);

		/*
		 * if this factory operates in reflex mode
		 */
		if (_this_weaveableOptRC != null) {

			try {

				/*
				 * get the controller descriptor
				 */
				String cd = getControllerDesc(controllerDesc);

				/*
				 * generate the controller descriptor
				 */
				String newControllerDesc = getNewControllerDesc(cd);

				/*
				 * mark package to notify the installer only to install it
				 * without starting it.
				 */
				Map<String, String> properties = (Map<String, String>) packageDescription
						.getPackageProperties();
				properties.put("isStart", "false");

				cmp = createComponent(type, controllerDesc, contentDesc,
						packageDescription, newControllerDesc);

				try {
					/*
					 * Set the contentDesc and controllerDesc attributes
					 */
					ReflexController rc = Reflex.getReflexController(cmp);

					rc.setReflexAttribute("controllerDesc",
							(String) (controllerDesc));
					rc
							.setReflexAttribute("contentDesc",
									(String) (contentDesc));

				} catch (Exception noReflexControllerFound) {
					// ignored
				}
			} catch (Exception e) {
				Logger
						.println(
								DebugReflex.on,
								"[ReflexGenericInstallingFactory] newFcInstanceNotification : cannot create instance "
										+ e);
			}
		} else {
			Logger
					.println(
							DebugReflex.warning,
							"[Warning][ReflexGenericInstallingFactory] newFcInstanceNotification : no dual level");

			/*
			 * calls the overriden newFcInstance method with the standard
			 * controllers
			 */
			cmp = _super_newFcInstance(type, controllerDesc, contentDesc,
					packageDescription);
		}

		Logger.println(DebugReflex.genericfactory,
				"[ReflexGenericInstallingFactory] newFcInstance : cmp = "
						+ cmp.toString());

		return cmp;

	}

	// -------------------------------------------------------------------------
	// Implementation of GenericFactoryNotification interface
	// -------------------------------------------------------------------------

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.ow2.jasmine.jade.reflex.api.factory.GenericFactoryNotification#newFcInstanceNotification(org.objectweb.fractal.api.Type,
	 *      java.lang.Object, java.lang.Object)
	 */
	public Component newFcInstanceNotification(Type type,
			Object controllerDesc, Object contentDesc)
			throws InstantiationException {

		init();

		Component cmp = null;

		Logger.println(DebugReflex.genericfactory,
				"[Notification][ReflexGenericInstallingFactory] newFcInstance : "
						+ contentDesc);

		/*
		 * if this factory operates in reflex mode
		 */
		if (_this_weaveableOptRC != null) {

			try {

				/*
				 * get the controller descriptor
				 */
				String cd = getControllerDesc(controllerDesc);

				/*
				 * generate the controller descriptor
				 */
				String newControllerDesc = getNewControllerDesc(cd);

				cmp = createComponent(type, controllerDesc, contentDesc,
						newControllerDesc);

				try {
					/*
					 * Set the contentDesc and controllerDesc attributes
					 */
					ReflexController rc = Reflex.getReflexController(cmp);

					rc.setReflexAttribute("controllerDesc",
							(String) (controllerDesc));
					rc
							.setReflexAttribute("contentDesc",
									(String) (contentDesc));

				} catch (Exception noReflexControllerFound) {
					// ignored
				}
			} catch (Exception e) {
				Logger
						.println(
								DebugReflex.on,
								"[ReflexGenericInstallingFactory] newFcInstanceNotification : cannot create instance "
										+ e);
			}
		} else {
			Logger
					.println(
							DebugReflex.warning,
							"[Warning][ReflexGenericInstallingFactory] newFcInstanceNotification : no dual level");

			/*
			 * calls the overriden newFcInstance method with the standard
			 * controllers
			 */
			cmp = _super_newFcInstance(type, controllerDesc, contentDesc);
		}

		Logger
				.println(
						DebugReflex.genericfactory,
						"[Notification][ReflexGenericInstallingFactory] newFcInstanceNotification : cmp = "
								+ cmp.toString());
		return cmp;

	}

	// -------------------------------------------------------------------------
	// Private methods
	// -------------------------------------------------------------------------

	/**
	 * @throws InstantiationException
	 */
	private void init() throws InstantiationException {

		if (ns == null) {
			/*
			 * if the factory operates in reflexive mode
			 */
			if (_this_weaveableOptRC != null) {

				try {

					/*
					 * get the dual jadenode
					 */
					String dualName = Environment
							.getProperty(Environment.reflexDualNameLabel);

					/*
					 * if dual-name is not defined, the bootstrap doesn't
					 * duplicate components
					 */
					if (dualName == null) {

						try {
							Logger
									.println(DebugReflex.genericfactory,
											"[ReflexGenericInstallingFactory] init: collocated meta & exec factories");
							/*
							 * set the dual references for the factory and for
							 * the dual factory (dual(cmp) = cmp)
							 */
							_this_weaveableOptRC.setCmpRef(Reflex
									.getComponent(_this_weaveableC));

						} catch (Exception e) {
							Logger
									.println(
											DebugReflex.on,
											"[ReflexGenericInstallingFactory] init : cannot set dual references for factories"
													+ e);
							throw new InstantiationException(
									"Pb encountered when setting dual references for factories");
						}

					}

					/*
					 * The bootstrap duplicate components ant needs to get a ref
					 * of Jadeboot bootstrap
					 */
					else {

						/*
						 * get the Fractal RMI registry
						 */
						NamingService ns = (NamingService) _this_weaveableBC
								.lookupFc("registry");
						if (ns == null) {

							Logger
									.println(DebugReflex.on,
											"[ReflexGenericInstallingFactory] init : cannot get registry");
							throw new InstantiationException(
									"The registry is unreachable");
						}

						/*
						 * get the reference of the dual factory if it is
						 * already started
						 */
						Logger.println(DebugReflex.genericfactory,
								"[ReflexGenericInstallingFactory] : looking for the Fractal factory of "
										+ dualName);

						Component dualJadeNode = ns.lookup(dualName);
						Component controllers = null;
						ContentController cc = Fractal
								.getContentController(dualJadeNode);
						for (Component sc : cc.getFcSubComponents()) {
							try {
								NameController nc = Fractal
										.getNameController(sc);
								if ("controllers".equals(nc.getFcName())) {
									controllers = sc;
									continue;
								}
							} catch (NoSuchInterfaceException ignored) {
							}
						}

						Component dualBootstrap = null;
						cc = Fractal.getContentController(controllers);
						for (Component sc : cc.getFcSubComponents()) {
							try {
								NameController nc = Fractal
										.getNameController(sc);
								if (nc.getFcName().endsWith("_factory")) {
									dualBootstrap = sc;
									continue;
								}
							} catch (NoSuchInterfaceException ignored) {
							}
						}

						if (dualBootstrap != null) {
							try {
								Logger
										.println(DebugReflex.genericfactory,
												"[ReflexGenericInstallingFactory] init: setting dual references for factories");
								/*
								 * set the dual references for the factory and
								 * for the dual factory
								 */
								_this_weaveableOptRC.setCmpRef(dualBootstrap);

							} catch (Exception e) {
								Logger
										.println(
												DebugReflex.on,
												"[ReflexGenericInstallingFactory] init : cannot set dual references for factories"
														+ e);
								throw new InstantiationException(
										"Pb encountered when setting dual references for factories");
							}

							// isInitialized = true;

						} else {
							Logger
									.println(
											DebugReflex.on,
											"[ReflexGenericInstallingFactory] init : cannot find the dual factory defined by the name "
													+ dualName);
						}
					}
				} catch (Exception e) {
					Logger.println(DebugReflex.on,
							"[ReflexGenericInstallingFactory] init : " + e);
				}
			}
		}
	}

	/**
	 * @param type
	 * @param controllerDesc
	 * @param newControllerDesc
	 * @param dualCtrl
	 * @return
	 * @throws InstantiationException
	 */
	private Component createDualComponent(Type type, Object controllerDesc,
			String newControllerDesc,
			GenericInstallingFactoryNotification dualCtrl)
			throws InstantiationException {
		Component dualCmp;
		if (controllerDesc instanceof Object[]) {

			dualCmp = dualCtrl.newFcInstanceNotification(type, new Object[] {
					((Object[]) controllerDesc)[0], newControllerDesc }, null);
		} else {

			dualCmp = dualCtrl.newFcInstanceNotification(type,
					newControllerDesc, null);
		}
		return dualCmp;
	}

	/**
	 * @param type
	 * @param controllerDesc
	 * @param contentDesc
	 * @param newControllerDesc
	 * @return
	 * @throws InstantiationException
	 */
	private Component createComponent(Type type, Object controllerDesc,
			Object contentDesc, String newControllerDesc)
			throws InstantiationException {
		Component cmp;
		if (controllerDesc instanceof Object[]) {
			cmp = _super_newFcInstance(type, new Object[] {
					((Object[]) controllerDesc)[0], newControllerDesc },
					contentDesc);
		} else {
			cmp = _super_newFcInstance(type, newControllerDesc, contentDesc);
		}
		return cmp;
	}

	/**
	 * @param type
	 * @param controllerDesc
	 * @param pkg
	 * @param newControllerDesc
	 * @param dualCtrl
	 * @return
	 * @throws InstantiationException
	 */
	private Component createDualComponent(Type type, Object controllerDesc,
			PackageDescription pkg, String newControllerDesc,
			GenericInstallingFactoryNotification dualCtrl)
			throws InstantiationException {
		Component dualCmp;
		if (controllerDesc instanceof Object[]) {

			dualCmp = dualCtrl.newFcInstanceNotification(type, new Object[] {
					((Object[]) controllerDesc)[0], newControllerDesc }, null,
					pkg);
		} else {

			dualCmp = dualCtrl.newFcInstanceNotification(type,
					newControllerDesc, null, pkg);
		}
		return dualCmp;
	}

	/**
	 * @param dualBootstrap
	 */
	private Component setNewComponentContainer(Component dualBootstrap) {

		Component dualBootstrapNewComponentContainerBackup = null;

		try {

			Component newComponentContainer = _this_weaveableGIFCC
					.getNewComponentContainer();

			GenericInstallingFactoryContextController dbgifc = FractalUtil
					.getGenericInstallingFactoryContextController(dualBootstrap);

			dualBootstrapNewComponentContainerBackup = dbgifc
					.getNewComponentContainer();

			if (newComponentContainer != null) {

				Component dualNewComponentContainer = Reflex
						.getDualComponent(newComponentContainer);

				if (dualNewComponentContainer != null) {

					Reflex.getReflexController(dualBootstrap)
							.setMetaComponentContainer(
									dualNewComponentContainer);

					dbgifc.setNewComponentContainer(dualNewComponentContainer);
				}
			}
		} catch (NoSuchInterfaceException e) {
		}

		return dualBootstrapNewComponentContainerBackup;
	}

	/**
	 * @param type
	 * @param controllerDesc
	 * @param contentDesc
	 * @param pkg
	 * @param newControllerDesc
	 * @return
	 * @throws InstantiationException
	 */
	private Component createComponent(Type type, Object controllerDesc,
			Object contentDesc, PackageDescription pkg, String newControllerDesc)
			throws InstantiationException {
		Component cmp;
		if (controllerDesc instanceof Object[]) {
			cmp = _super_newFcInstance(type, new Object[] {
					((Object[]) controllerDesc)[0], newControllerDesc },
					contentDesc, pkg);
		} else {
			cmp = _super_newFcInstance(type, newControllerDesc, contentDesc,
					pkg);
		}
		return cmp;
	}

	/**
	 * @param controllerDesc
	 * @return
	 */
	private String getNewControllerDesc(String controllerDesc) {
		if (controllerDesc.startsWith("jade-")) {
			return Reflex.controllersPrefix + controllerDesc;
		} else {
			return controllerDesc;
		}
	}

	/**
	 * @param controllerDesc
	 * @return
	 */
	private String getControllerDesc(Object controllerDesc) {
		String cd = null;
		if (controllerDesc instanceof Object[]) {
			cd = (String) (((Object[]) controllerDesc)[1]);
		} else {
			cd = (String) controllerDesc;
		}
		return cd;
	}

	// -------------------------------------------------------------------------
	// Fields and methods required by the mixin class in the base class
	// -------------------------------------------------------------------------

	/**
	 * The <tt>weaveableC</tt> field required by this mixin. This field is
	 * supposed to reference the {@link Component} interface of the component to
	 * which this controller object belongs.
	 */
	public Component _this_weaveableC;

	/**
	 * The <tt>weaveableBC</tt> field required by this mixin. This field is
	 * supposed to reference the {@link BindingController} interface of the
	 * component to which this controller object belongs.
	 */

	public BindingController _this_weaveableBC;

	/**
	 * The <tt>weaveableOptRC</tt> field required by this mixin. This field is
	 * supposed to reference the {@link ReflexController} interface of the
	 * component to which this controller object belongs.
	 */
	public ReflexController _this_weaveableOptRC;

	/**
	 * The <tt>weaveableGIFCC</tt> field required by this mixin. This field is
	 * supposed to reference the
	 * {@link GenericInstallingFactoryContextController} interface of the
	 * component to which this controller object belongs.
	 */
	public GenericInstallingFactoryContextController _this_weaveableGIFCC;

	/**
	 * The {@link GenericFactory#newFcInstance newFcInstance} method overriden
	 * by this mixin.
	 * 
	 * @param type
	 *            an arbitrary component type.
	 * @param controllerDesc
	 *            a description of the controller part of the component to be
	 *            created. This description is implementation specific. If it is
	 *            <tt>null</tt> then a "default" controller part will be used.
	 * @param contentDesc
	 *            a description of the content part of the component to be
	 *            created. This description is implementation specific. It may
	 *            be <tt>null</tt> to create component with an empty initial
	 *            content. It may also be, in Java, the fully qualified name of
	 *            a Java class, to create primitive components.
	 * @return the {@link Component} interface of the created component.
	 * @throws InstantiationException
	 *             if the component cannot be created.
	 */

	public abstract Component _super_newFcInstance(Type type,
			Object controllerDesc, Object contentDesc)
			throws InstantiationException;

	public abstract Component _super_newFcInstance(Type type,
			Object controllerDesc, Object contentDesc, PackageDescription pkg)
			throws InstantiationException;

	public abstract void _super_undeployFcComponent(Component component);

	public abstract void _super_initFcController(InitializationContext ic)
			throws InstantiationException;

}
