/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008 Bull S.A.S.
 * Copyright (C) 2008 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: WorkerProgressBean.java 2758 2008-11-21 14:28:15Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.api.entities.deployment;

import java.io.Serializable;

import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.OneToOne;
import javax.persistence.Version;

import org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean.ActionState;
import org.ow2.jasmine.jadort.api.entities.topology.WorkerBean;

/**
 * Bean that represents the progress of the operations a worker.
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
@Entity(name = "JaDOrT_WorkerProgressBean")
public class WorkerProgressBean implements Serializable {

    /**
     * Auto-generated serial version ID.
     */
    private static final long serialVersionUID = -2851071984012556925L;

    /**
     * Auto-generated identifier of the WorkerProgressBean in the database.
     */
    @Id
    @GeneratedValue
    private Integer id;

    /**
     * Auto-generated version of the WorkerProgressBean. Used internally for
     * checking integrity.
     */
    @Version
    @SuppressWarnings("unused")
    private Integer version;

    /**
     * Different possible progress states of a workerProgress.<br>
     * <br>
     * We don't use an enum here to preserve compatibility with IIOP runtimes on
     * Java EE 5. You can check out the bug report and comments on
     * http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=6277781 for details.
     */
    public static final class WorkerProgressState implements Serializable {

        private static final long serialVersionUID = 2591831000876771633L;

        public static final WorkerProgressState INITIAL = new WorkerProgressState("INITIAL");

        public static final WorkerProgressState START_OK = new WorkerProgressState("START_OK");

        public static final WorkerProgressState STOP_OK = new WorkerProgressState("STOP_OK");

        private static final WorkerProgressState[] values = new WorkerProgressState[] {WorkerProgressState.INITIAL,
            WorkerProgressState.START_OK, WorkerProgressState.STOP_OK};

        private String workerProgressState;

        /**
         * @param workerProgressState WorkerProgressState.
         */
        private WorkerProgressState(final String workerProgressState) {
            this.workerProgressState = workerProgressState;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean equals(final Object object) {
            if ((object != null) && (object instanceof WorkerProgressState)) {
                WorkerProgressState workerProgressState = (WorkerProgressState) object;
                if (workerProgressState.workerProgressState.equals(this.workerProgressState)) {
                    return true;
                }
            }

            return false;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public int hashCode() {
            return this.workerProgressState.hashCode();
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String toString() {
            return this.workerProgressState;
        }

        /**
         * @return All WorkerProgressState values.
         */
        public static WorkerProgressState[] values() {
            return WorkerProgressState.values;
        }

        /**
         * Get the WorkerProgressState for a string.
         * 
         * @param string WorkerProgressState string.
         * @return WorkerProgressState for that string.
         */
        public static WorkerProgressState valueOf(final String string) {
            for (WorkerProgressState workerProgressState : WorkerProgressState.values) {
                if (workerProgressState.workerProgressState.equals(string)) {
                    return workerProgressState;
                }
            }

            throw new IllegalArgumentException("Unknown value: " + string);
        }
    }

    /**
     * The associated worker to the workerProgress.
     */
    // Cascade for all except CascadeType.REMOVE, since removing a
    // WorkerProgressBean doesn't imply that the associated WorkerBean
    // (created as part of a migration operation) should be deleted!
    @OneToOne(cascade = {CascadeType.MERGE, CascadeType.PERSIST, CascadeType.REFRESH})
    private WorkerBean worker;

    /**
     * Progress percent of the workerProgress.
     */
    private int progress;

    /**
     * Deployment log
     */
    private String log;

    /**
     * Deployment progress state
     */
    private WorkerProgressState workerProgressState;

    /**
     * Deployment state
     */
    private ActionState actionState;

    /**
     * Default WorkerProgressBean constructor
     */
    public WorkerProgressBean() {
        this.progress = 0;
        this.actionState = ActionState.WAITING;
        this.workerProgressState = WorkerProgressState.INITIAL;
    }

    /**
     * Construct a WorkerProgressBean with its associated worker
     * 
     * @param worker The worker associated ti the depoyment
     */
    public WorkerProgressBean(final WorkerBean worker) {
        this();
        this.worker = worker;
    }

    /**
     * Get the workerProgress id
     * 
     * @return the workerProgress id
     */
    public Integer getId() {
        return this.id;
    }

    /**
     * Set the workerProgress id
     * 
     * @param id the workerProgress id
     */
    public void setId(final Integer id) {
        this.id = id;
    }

    /**
     * Associated worker getter
     * 
     * @return The associated worker
     */
    public WorkerBean getWorker() {
        return this.worker;
    }

    /**
     * Change the associated worker
     * 
     * @param worker New associated worker
     */
    public void setWorker(final WorkerBean worker) {
        this.worker = worker;
    }

    /**
     * Returns the name of the worker associated with this workerProgress
     * 
     * @return the name of the worker associated with this workerProgress
     */
    public String getWorkerName() {
        return this.worker.getName();
    }

    /**
     * Return the workerProgress state in String format :
     * <ul>
     * <li>WAITING --> "Waiting"
     * <li>RUNNING --> "Running"
     * <li>FINISHED_OK -- > "Done OK"
     * <li>FINISHED_ERROR --> "Error !"
     * </ul>
     * 
     * @return Deployment state in String format.
     */
    public String getState() {
        if (this.actionState.equals(ActionState.WAITING)) {
            return "Waiting";
        } else if (this.actionState.equals(ActionState.RUNNING)) {
            return "Running";
        } else if (this.actionState.equals(ActionState.FINISHED_OK)) {
            return "Done OK";
        } else if (this.actionState.equals(ActionState.FINISHED_ERROR)) {
            return "Error !";
        } else {
            return null;
        }
    }

    /**
     * Do not use this method, needs to be here to satisfy ActionScript
     * 
     * @param state
     */
    public void setState(final String state) {
        // Empty method, needs to be here to satisfy ActionScript
    }

    /**
     * Percent of progress getter
     * 
     * @return The workerProgress progress (percents)
     */
    public int getProgress() {
        return this.progress;
    }

    /**
     * Deployment progress setter.
     * 
     * @param progress The progress percent to set.
     */
    public void setProgress(final int progress) {
        if (progress < 0 || progress > 100) {
            throw new IllegalArgumentException("Progress is a percentage (between 0 and 100)");
        }

        this.progress = progress;
    }

    /**
     * Get the workerProgress log
     * 
     * @return Deployment log
     */
    public String getLog() {
        return this.log;
    }

    /**
     * Set the workerProgress log
     * 
     * @param log The log to set for this workerProgress
     */
    public void setLog(final String log) {
        this.log = log;
    }

    /**
     * Add log to the workerProgress log.
     * 
     * @param string The log to add for this workerProgress
     */
    public void appendToLog(final String string) {
        if (string != null && string.length() > 0) {
            if (this.log == null) {
                this.log = string;
            } else {
                this.log += '\n' + string;
            }
        }
    }

    /**
     * Get the workerProgress progress state
     * 
     * @return Deployment progress state
     */
    public WorkerProgressState getWorkerProgressState() {
        return this.workerProgressState;
    }

    /**
     * Set the workerProgress progress state
     * 
     * @param workerProgressState the workerProgress progress state to set
     */
    public void setWorkerProgressState(final WorkerProgressState workerProgressState) {
        this.workerProgressState = workerProgressState;
    }

    /**
     * Get workerProgress state.
     * 
     * @return Deployment state.
     */
    public ActionState getActionState() {
        return this.actionState;
    }

    /**
     * Set workerProgress state.
     * 
     * @param actionState Deployment state to set.
     */
    public void setActionState(final ActionState actionState) {
        this.actionState = actionState;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(this.getClass().getSimpleName());
        sb.append("[worker='");
        sb.append(this.worker);
        sb.append("']");
        return sb.toString();
    }

    /**
     * {@inheritDoc}
     * 
     * @return true if <code>obj</code> is a {@link WorkerProgressBean} and the
     *         {@link WorkerProgressBean#getId()} is the same for both objects,
     *         false otherwise.
     */
    @Override
    public boolean equals(final Object obj) {
        if (obj != null && obj instanceof WorkerProgressBean) {
            WorkerProgressBean other = (WorkerProgressBean) obj;
            if (this.id != null && other.id != null) {
                return this.id.equals(other.id);
            }
        }

        // Else not same type or some parts are null
        return false;
    }

}
