/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Copyright (C) 2008-2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: IJadortService.java 2965 2009-01-12 12:14:51Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.api;

import java.io.File;
import java.io.Serializable;
import java.net.URL;
import java.util.List;

import org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean;
import org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean;
import org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean;
import org.ow2.jasmine.jadort.api.entities.deployment.WorkerProgressBean;
import org.ow2.jasmine.jadort.api.entities.topology.GroupBean;
import org.ow2.jasmine.jadort.api.entities.topology.ServerBean;
import org.ow2.jasmine.jadort.api.entities.topology.WorkerBean;

/**
 * IJadortService interface defines all methods used for the serverProgress
 * orchestration
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
public interface IJadortService {

    /**
     * Operation types.<br>
     * <br>
     * We don't use an enum here to preserve compatibility with IIOP runtimes on
     * Java EE 5. You can check out the bug report and comments on
     * http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=6277781 for details.
     */
    public static final class OperationType implements Serializable {

        private static final long serialVersionUID = -7508039622857282263L;

        public static final OperationType MIGRATE = new OperationType("MIGRATE");

        public static final OperationType MAINTAIN = new OperationType("MAINTAIN");

        private static final OperationType[] values = new OperationType[] {OperationType.MIGRATE, OperationType.MAINTAIN};

        private String operationType;

        /**
         * @param operationType OperationType.
         */
        private OperationType(final String operationType) {
            this.operationType = operationType;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean equals(final Object object) {
            if ((object != null) && (object instanceof OperationType)) {
                OperationType operationType = (OperationType) object;
                if (operationType.operationType.equals(this.operationType)) {
                    return true;
                }
            }

            return false;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public int hashCode() {
            return this.operationType.hashCode();
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String toString() {
            return this.operationType;
        }

        /**
         * @return All OperationType values.
         */
        public static OperationType[] values() {
            return OperationType.values;
        }

        /**
         * Get the OperationType for a string.
         * 
         * @param string OperationType string.
         * @return OperationType for that string.
         */
        public static OperationType valueOf(final String string) {
            for (OperationType operationType : OperationType.values) {
                if (operationType.operationType.equals(string)) {
                    return operationType;
                }
            }

            throw new IllegalArgumentException("Unknown value: " + string);
        }
    }

    /**
     * Returns the current Step of the current operation
     * 
     * @return The current step.
     */
    OperationStateBean.Step getCurrentStep() throws JadortServiceException;

    /**
     * Returns the list of operations
     * 
     * @return List of Operations.
     */
    List<OperationStateBean> getOperationsList() throws JadortServiceException;

    /**
     * Delete an Operation from the operation list
     * 
     * @param operation the operation to be deleted
     */
    void deleteOperation(final OperationStateBean operation) throws JadortServiceException;

    /**
     * Select an Operation from the operation list
     * 
     * @param operation the operation to be selected
     */
    void selectOperation(final OperationStateBean operation) throws JadortServiceException;

    /**
     * Create a new operation
     * 
     * @param newOperationName the name of the new operation that will be
     *        created
     */
    void createNewOperation(final String newOperationName) throws JadortServiceException;

    /**
     * Returns the current operation
     * 
     * @return The current operation.
     */
    OperationStateBean getCurrentOperation() throws JadortServiceException;

    /**
     * Tests if it is allowed to go to the next step
     * 
     * @return true id calling {@link IJadortService#next()} is allowed, false
     *         otherwise.
     */
    boolean canGoToNextStep() throws JadortServiceException;

    /**
     * goes to next step
     * 
     * @see IJadortService#canGoToNextStep()
     */
    void next() throws JadortServiceException;

    /**
     * Tests if it is allowed to go the previous step
     * 
     * @return true id calling {@link IJadortService#previous()} is allowed,
     *         false otherwise.
     */
    boolean canGoToPreviousStep() throws JadortServiceException;

    /**
     * Goes back to the previous step.
     * 
     * @see IJadortService#canGoToPreviousStep()
     */
    void previous() throws JadortServiceException;

    /**
     * Loads the topology from a .xml file.
     * 
     * @param xmlTopoFile the file path of the .xml file uploaded by the user,
     *        this file contains the topology description
     */
    void loadTopology(final File xmlTopoFile) throws JadortServiceException;

    /**
     * Returns the list of groups obtained from the loaded topology in the
     * current operation
     * 
     * @return The list of groups
     */
    List<GroupBean> getGroups() throws JadortServiceException;

    /**
     * Select a group from the group list
     * 
     * @param selectedGroup the group to be selected
     */
    void selectGroup(final GroupBean selectedGroup, final OperationType operationType) throws JadortServiceException;

    /**
     * Returns the list of server ServerProgressBean in the current operation.
     * The servers of these ServerProgressBean objects will have their number of
     * active sessions up to date.
     * 
     * @return The list of ServerProgressBean
     */
    List<ServerProgressBean> getServerProgressList() throws JadortServiceException;

    /**
     * Returns the list of server WorkerProgressBean in the current operation
     * 
     * @return The list of WorkerProgressBean
     */
    List<WorkerProgressBean> getWorkerProgressList() throws JadortServiceException;

    /**
     * Returns the selected Group in the current operation
     * 
     * @return the selected group
     */
    GroupBean getSelectedGroup() throws JadortServiceException;

    /**
     * @return The number of active sessions on all servers:
     *         <ul>
     *         <li>If migrating, it corresponds to the number of active sessions
     *         on the old version. If no old version, this is set to 0.</li>
     *         <li>If maintaining, it corresponds to the number of active
     *         sessions on all applications on the server.</li>
     *         </ul>
     *         This value is re-set every time the object is refreshed on the
     *         stateful bean, and not persisted.
     */
    int getActiveSessions() throws JadortServiceException;

    /**
     * This function allows to restart a task on a server (the server will
     * restart the last task)
     * 
     * @param server the server on which the task will be restarted
     */
    void restartServer(final ServerBean server) throws JadortServiceException;

    /**
     * this function checks if no problem on the server and all problems are
     * resolved on the server.
     * 
     * @return true-> OK the problem server is resolved and it can be ignored ,
     *         false -> KO the problem still
     * @param server the server that will be checked
     * @see IJadortService#ignoreServer()
     */
    boolean checkServer(final ServerBean server) throws JadortServiceException;

    /**
     * this function forces the server to be ignored and continues this task
     * 
     * @param server the server that will be ignored
     */
    void ignoreServer(final ServerBean server) throws JadortServiceException;

    /**
     * this function forces the server to be aborted and continues this task
     * 
     * @param server the server that will be aborted
     */
    void abortServer(final ServerBean server) throws JadortServiceException;

    /**
     * Create a new application
     * 
     * @param url URL where to fetch the application. Can be local, remote
     *        (HTTP, FTP, ...)
     */
    void createApplication(final URL url) throws JadortServiceException;

    /**
     * Get the current application
     * 
     * @return the ApplicationBean that will be installed.
     */
    ApplicationBean getApplication() throws JadortServiceException;

    /**
     * Select the servers to maintain
     * 
     * @param selectedServers the servers to maintain
     */
    void selectServers(List<ServerBean> selectedServers) throws JadortServiceException;

    /**
     * This function allows to get the list of servers. These servers will have
     * their number of active sessions up to date.
     */
    List<ServerBean> getServers() throws JadortServiceException;

    /**
     * This function allows to get the list of selected servers when maintain
     * servers
     */
    List<ServerBean> getSelectedServers() throws JadortServiceException;

    /**
     * This function allows to restart a task on a worker (the worker will
     * restart the last task)
     * 
     * @param worker the worker on which the task will be restarted
     */
    void restartWorker(final WorkerBean worker) throws JadortServiceException;

    /**
     * this function checks if no problem on the worker and all problems are
     * resolved on the worker.
     * 
     * @return true-> OK the problem server is resolved and it can be ignored ,
     *         false -> KO the problem still
     * @param worker the worker that will be checked
     * @see IJadortService#ignoreWorker()
     */
    boolean checkWorker(final WorkerBean worker) throws JadortServiceException;

    /**
     * this function forces the worker to be ignored and continues this task
     * 
     * @param worker the worker that will be ignored
     */
    void ignoreWorker(final WorkerBean worker) throws JadortServiceException;

    /**
     * this function forces the worker to be aborted and continues this task
     * 
     * @param worker the worker that will be aborted
     */
    void abortWorker(final WorkerBean worker) throws JadortServiceException;

}
