/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Copyright (C) 2008-2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ServerProgressBean.java 2965 2009-01-12 12:14:51Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.api.entities.deployment;

import java.io.Serializable;

import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.OneToOne;
import javax.persistence.Version;

import org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean.ActionState;
import org.ow2.jasmine.jadort.api.entities.topology.ServerBean;

/**
 * Bean that represents the progress of the operations a server.
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
@Entity(name = "JaDOrT_ServerProgressBean")
public class ServerProgressBean implements Serializable {

    /**
     * Auto-generated serial version ID.
     */
    private static final long serialVersionUID = -2851071984012556925L;

    /**
     * Auto-generated identifier of the ServerProgressBean in the database.
     */
    @Id
    @GeneratedValue
    private Integer id;

    /**
     * Auto-generated version of the ServerProgressBean. Used internally for
     * checking integrity.
     */
    @Version
    @SuppressWarnings("unused")
    private Integer version;

    /**
     * Different possible progress states of a serverProgress.<br>
     * <br>
     * We don't use an enum here to preserve compatibility with IIOP runtimes on
     * Java EE 5. You can check out the bug report and comments on
     * http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=6277781 for details.
     */
    public static final class ServerProgressState implements Serializable {

        private static final long serialVersionUID = 6491848606792309527L;

        public static final ServerProgressState INITIAL = new ServerProgressState("INITIAL");

        public static final ServerProgressState UPLOAD_OK = new ServerProgressState("UPLOAD_OK");

        public static final ServerProgressState DEPLOY_OK = new ServerProgressState("DEPLOY_OK");

        public static final ServerProgressState SET_DEFAULT_OK = new ServerProgressState("SET_DEFAULT_OK");

        public static final ServerProgressState UNDEPLOY_OK = new ServerProgressState("UNDEPLOY_OK");

        public static final ServerProgressState ERASE_OK = new ServerProgressState("ERASE_OK");

        public static final ServerProgressState START_OK = new ServerProgressState("START_OK");

        public static final ServerProgressState MAINTAIN_OK = new ServerProgressState("MAINTAIN_OK");

        public static final ServerProgressState STOP_OK = new ServerProgressState("STOP_OK");

        public static final ServerProgressState DISABLE_APPLICATIONS_OK = new ServerProgressState("DISABLE_APPLICATIONS_OK");

        private static final ServerProgressState[] values = new ServerProgressState[] {ServerProgressState.INITIAL,
            ServerProgressState.UPLOAD_OK, ServerProgressState.DEPLOY_OK, ServerProgressState.SET_DEFAULT_OK,
            ServerProgressState.UNDEPLOY_OK, ServerProgressState.ERASE_OK, ServerProgressState.START_OK,
            ServerProgressState.MAINTAIN_OK, ServerProgressState.STOP_OK, ServerProgressState.DISABLE_APPLICATIONS_OK};

        private String serverProgressState;

        /**
         * @param serverProgressState ServerProgressState.
         */
        private ServerProgressState(final String serverProgressState) {
            this.serverProgressState = serverProgressState;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean equals(final Object object) {
            if ((object != null) && (object instanceof ServerProgressState)) {
                ServerProgressState serverProgressState = (ServerProgressState) object;
                if (serverProgressState.serverProgressState.equals(this.serverProgressState)) {
                    return true;
                }
            }

            return false;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public int hashCode() {
            return this.serverProgressState.hashCode();
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String toString() {
            return this.serverProgressState;
        }

        /**
         * @return All ServerProgressState values.
         */
        public static ServerProgressState[] values() {
            return ServerProgressState.values;
        }

        /**
         * Get the ServerProgressState for a string.
         * 
         * @param string ServerProgressState string.
         * @return ServerProgressState for that string.
         */
        public static ServerProgressState valueOf(final String string) {
            for (ServerProgressState serverProgressState : ServerProgressState.values) {
                if (serverProgressState.serverProgressState.equals(string)) {
                    return serverProgressState;
                }
            }

            throw new IllegalArgumentException("Unknown value: " + string);
        }
    }

    /**
     * The associated server to the serverProgress.
     */
    // Cascade for all except CascadeType.REMOVE, since removing a
    // ServerProgressBean doesn't imply that the associated ServerBean
    // (created as part of a GroupBean) should be deleted!
    @OneToOne(cascade = {CascadeType.MERGE, CascadeType.PERSIST, CascadeType.REFRESH})
    private ServerBean server;

    /**
     * Old application name. This is the previously deployed application on the
     * server.
     */
    private String oldApplication;

    /**
     * New application name. The application the serverProgress is currently
     * deploying.
     */
    private String newApplication;

    /**
     * Progress percent of the serverProgress.
     */
    private int progress;

    /**
     * Deployment log
     */
    private String log;

    /**
     * Deployment progress state
     */
    private ServerProgressState serverProgressState;

    /**
     * Deployment state
     */
    private ActionState actionState;

    /**
     * Default ServerProgressBean constructor
     */
    public ServerProgressBean() {
        this.progress = 0;
        this.actionState = ActionState.WAITING;
        this.serverProgressState = ServerProgressState.INITIAL;
    }

    /**
     * Construct a ServerProgressBean with its associated server
     * 
     * @param server The server associated ti the depoyment
     */
    public ServerProgressBean(final ServerBean server) {
        this();
        this.server = server;
    }

    /**
     * Get the serverProgress id
     * 
     * @return the serverProgress id
     */
    public Integer getId() {
        return this.id;
    }

    /**
     * Set the serverProgress id
     * 
     * @param id the serverProgress id
     */
    public void setId(final Integer id) {
        this.id = id;
    }

    /**
     * Server getter
     * 
     * @return The associated server
     */
    public ServerBean getServer() {
        return this.server;
    }

    /**
     * Change the associated server
     * 
     * @param server New associated server
     */
    public void setServer(final ServerBean server) {
        this.server = server;
    }

    /**
     * Returns the name of the server associated with this serverProgress
     * 
     * @return the name of the server associated with this serverProgress
     */
    public String getServerName() {
        return this.server.getName();
    }

    /**
     * Return the serverProgress state in String format :
     * <ul>
     * <li>WAITING --> "Waiting"
     * <li>RUNNING --> "Running"
     * <li>FINISHED_OK -- > "Done OK"
     * <li>FINISHED_ERROR --> "Error !"
     * </ul>
     * 
     * @return Deployment state in String format.
     */
    public String getState() {
        if (this.actionState.equals(ActionState.WAITING)) {
            return "Waiting";
        } else if (this.actionState.equals(ActionState.RUNNING)) {
            return "Running";
        } else if (this.actionState.equals(ActionState.FINISHED_OK)) {
            return "Done OK";
        } else if (this.actionState.equals(ActionState.FINISHED_ERROR)) {
            return "Error !";
        } else {
            return null;
        }
    }

    /**
     * Do not use this method, needs to be here to satisfy ActionScript
     * 
     * @param state
     */
    public void setState(final String state) {
        // Empty method, needs to be here to satisfy ActionScript
    }

    /**
     * Old application name getter
     * 
     * @return Old application name
     */
    public String getOldApplication() {
        return this.oldApplication;
    }

    /**
     * Old application name setter.
     * 
     * @param oldApplication Old application name
     */
    public void setOldApplication(final String oldApplication) {
        this.oldApplication = oldApplication;
    }

    /**
     * @return true if there's an old application version present and the old
     *         application is not the same as the new one.
     */
    public boolean hasOldVersion() {
        return this.oldApplication != null && !this.oldApplication.equals(this.newApplication);
    }

    /**
     * New application name getter.
     * 
     * @return New application name
     */
    public String getNewApplication() {
        return this.newApplication;
    }

    /**
     * New application name setter.
     * 
     * @param newApplication New application name
     */
    public void setNewApplication(final String newApplication) {
        this.newApplication = newApplication;
    }

    /**
     * Percent of progress getter
     * 
     * @return The serverProgress progress (percents)
     */
    public int getProgress() {
        return this.progress;
    }

    /**
     * Deployment progress setter.
     * 
     * @param progress The progress percent to set.
     */
    public void setProgress(final int progress) {
        if (progress < 0 || progress > 100) {
            throw new IllegalArgumentException("Progress is a percentage (between 0 and 100)");
        }

        this.progress = progress;
    }

    /**
     * Get the serverProgress log
     * 
     * @return Deployment log
     */
    public String getLog() {
        return this.log;
    }

    /**
     * Set the serverProgress log
     * 
     * @param log The log to set for this serverProgress
     */
    public void setLog(final String log) {
        this.log = log;
    }

    /**
     * Add log to the serverProgress log.
     * 
     * @param string The log to add for this serverProgress
     */
    public void appendToLog(final String string) {
        if (string != null && string.length() > 0) {
            if (this.log == null) {
                this.log = string;
            } else {
                this.log += '\n' + string;
            }
        }
    }

    /**
     * Get the serverProgress progress state
     * 
     * @return Deployment progress state
     */
    public ServerProgressState getServerProgressState() {
        return this.serverProgressState;
    }

    /**
     * Set the serverProgress progress state
     * 
     * @param serverProgressState the serverProgress progress state to set
     */
    public void setServerProgressState(final ServerProgressState serverProgressState) {
        this.serverProgressState = serverProgressState;
    }

    /**
     * Get serverProgress state.
     * 
     * @return Deployment state.
     */
    public ActionState getActionState() {
        return this.actionState;
    }

    /**
     * Set serverProgress state.
     * 
     * @param actionState Deployment state to set.
     */
    public void setActionState(final ActionState actionState) {
        this.actionState = actionState;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(this.getClass().getSimpleName());
        sb.append("[server='");
        sb.append(this.server);
        sb.append("']");
        return sb.toString();
    }

    /**
     * {@inheritDoc}
     * 
     * @return true if <code>obj</code> is a {@link ServerProgressBean} and the
     *         {@link ServerProgressBean#getId()} is the same for both objects,
     *         false otherwise.
     */
    @Override
    public boolean equals(final Object obj) {
        if (obj != null && obj instanceof ServerProgressBean) {
            ServerProgressBean other = (ServerProgressBean) obj;
            if (this.id != null && other.id != null) {
                return this.id.equals(other.id);
            }
        }

        // Else not same type or some parts are null
        return false;
    }

}
