/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Copyright (C) 2008-2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ProgressBean.java 3401 2009-04-14 14:47:53Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.api.entities.deployment;

import java.io.Serializable;

import org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean.ActionState;

/**
 * Common interface for all progress beans.
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
public abstract class ProgressBean<ProgressStateType extends ProgressState> implements Serializable {

    /**
     * Auto-generated serial version ID.
     */
    private static final long serialVersionUID = -4589119608958079980L;

    /**
     * Get the deployment log
     * 
     * @return Deployment log
     */
    public abstract String getLog();

    /**
     * Set the deployment log
     * 
     * @param log Deployment log
     */
    public abstract void setLog(final String log);

    /**
     * Append to the current deployment log
     * 
     * @param string Deployment log string to append to the current log
     */
    public synchronized void appendToLog(final String string) {
        if (string != null && string.length() > 0) {
            String log = this.getLog();
            if (log == null) {
                log = string;
            } else {
                log += '\n' + string;
            }
            this.setLog(log);
        }
    }

    /**
     * Get the deployment's progress state
     * 
     * @return Deployment progress state
     */
    public abstract ProgressStateType getProgressState();

    /**
     * Set the deployment's progress state
     * 
     * @param progressState Deployment progress state
     */
    public abstract void setProgressState(final ProgressStateType progressState);

    /**
     * Get the deployment's action state
     * 
     * @return Deployment action state
     */
    public abstract ActionState getActionState();

    /**
     * Set the deployment's action state
     * 
     * @param actionState Deployment action state
     */
    public abstract void setActionState(final ActionState actionState);

    /**
     * Return the {@link ProgressBean#getActionState()} in String format :
     * <ul>
     * <li>{@link ActionState#WAITING} becomes "Waiting"
     * <li>{@link ActionState#RUNNING} becomes "Running"
     * <li>{@link ActionState#FINISHED_OK} becomes "Done OK"
     * <li>{@link ActionState#FINISHED_ERROR} becomes "Error !"
     * </ul>
     * 
     * @return {@link ProgressBean#getActionState()} in String format.
     */
    public String getState() {
        ActionState actionState = this.getActionState();
        if (actionState.equals(ActionState.WAITING)) {
            return "Waiting";
        } else if (actionState.equals(ActionState.RUNNING)) {
            return "Running";
        } else if (actionState.equals(ActionState.FINISHED_OK)) {
            return "Done OK";
        } else if (actionState.equals(ActionState.FINISHED_ERROR)) {
            return "Error !";
        } else {
            return null;
        }
    }

    /**
     * Do not use this method, needs to be here to satisfy ActionScript
     * 
     * @param state
     */
    public void setState(final String state) {
        // Empty method, needs to be here to satisfy ActionScript
    }

    /**
     * @return The ProgressBean's progress (percents)
     */
    public abstract int getProgress();

    /**
     * @param progressPercent The ProgressBean's progress (percents)
     */
    public abstract void setProgress(final int progressPercent);

}
