/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008 Bull S.A.S.
 * Copyright (C) 2008 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JonasServerAction.java 2938 2009-01-08 10:04:54Z chahinem $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.service.action;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;

import javax.management.Attribute;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXServiceURL;

import org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean;
import org.ow2.jasmine.jadort.api.entities.topology.ConnectorBean;
import org.ow2.jasmine.jadort.api.entities.topology.ServerBean;
import org.ow2.util.ee.deploy.api.deployer.DeployerException;

/**
 * Action for the JOnAS 5 server.
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
public class JonasServerAction extends ServerAction {

    private static final int BUFFER_SIZE = 1024;

    private static final int AUTODEPLOY_TIMEOUT = 60;

    private String name;

    private ConnectorBean serverConnector;

    private ConnectorBean managerConnector;

    private ObjectName j2eeServer;

    private ObjectName versioningService;

    private MBeanServerConnection mbscnx = null;

    private Set<String> autoDeploy = Collections.synchronizedSet(new HashSet<String>());

    protected class VersionInformation {
        public String policy;

        public ObjectName mbean;

        public VersionInformation(final String policy, final ObjectName mbean) {
            this.policy = policy;
            this.mbean = mbean;
        }
    }

    protected class ApplicationInformation {
        public Map<String, VersionInformation> versions;

        public Set<ObjectName> managers;

        public ApplicationInformation() {
            this.versions = new HashMap<String, VersionInformation>();
            this.managers = new HashSet<ObjectName>();
        }
    }

    protected JonasServerAction(final ServerBean server) {
        this.name = server.getName();
        this.serverConnector = server.getServerConnector();
        this.managerConnector = server.getManagerConnector();
        this.appendToLog("Created JonasServerAction for server '" + this.name + "'");
    }

    /**
     * Checks connectivity with the server JMX connector.
     */
    @SuppressWarnings("unchecked")
    protected void checkJMXConnection() throws IOException, MalformedObjectNameException {
        String url = this.serverConnector.getConnectorUrl();
        String username = this.serverConnector.getUsername();
        String password = this.serverConnector.getPassword();

        if (this.mbscnx != null) {
            try {
                this.mbscnx.getMBeanCount();
            } catch (IOException e) {
                this.mbscnx = null;
                this.appendToLog("Connection dropped, reconnecting to JMX server on URL '" + url + "'");
            }
        }

        if (this.mbscnx == null) {
            this.appendToLog("Trying to connect to JMX server on URL '" + url + "'");
            Map<String, Object> env = new HashMap<String, Object>();
            if (username != null && password != null) {
                String[] creds = {username, password};
                env.put(JMXConnector.CREDENTIALS, creds);
            }
            JMXConnector c = JMXConnectorFactory.connect(new JMXServiceURL(url), env);
            this.mbscnx = c.getMBeanServerConnection();
            this.j2eeServer = ((Set<ObjectName>) this.mbscnx.queryNames(new ObjectName("*:j2eeType=J2EEServer,*"), null))
                .iterator().next();
            this.versioningService = null;
            try {
                ObjectName versioning = ((Set<ObjectName>) this.mbscnx.queryNames(new ObjectName(
                    "*:type=service,name=versioning"), null)).iterator().next();
                if (((Boolean) this.mbscnx.getAttribute(versioning, "VersioningEnabled")).booleanValue()) {
                    this.versioningService = versioning;
                }
            } catch (Exception ignored) {
                // Versioning service not present
            }
            this
                .appendToLog("JMX server connection OK for server '" + this.name + "', J2EEServer is '" + this.j2eeServer + "'");
        }
    }

    /**
     * Gets the list of deployed applications from the server.
     * 
     * @return Map of applications. The map should be read the following way:
     *         <ul>
     *         <li>Keys are the module name (EAR or standalone)
     *         <li>Values are the version details for the corresponding
     *         versioned module (map of versions + policy as well as all MBeans
     *         used for switching between versions). If the module is not
     *         versioned, its value is null.
     *         </ul>
     */
    @SuppressWarnings("unchecked")
    protected Map<String, ApplicationInformation> getApplicationsList() throws Exception {
        Map<String, ApplicationInformation> applications = new HashMap<String, ApplicationInformation>();

        // Non-versioned EJB-JARs and WARs, including those that are in EARs.
        Set<ObjectName> nonVersionedEjbsAndWars = new HashSet<ObjectName>();

        // Key = real ObjectName, Value = manager ObjectName. This therefore
        // implies that there'll be multiple keys pointing to the same value if
        // there are multiple versions of an application that're deployed.
        Map<ObjectName, ObjectName> versionedEjbsAndWars = new HashMap<ObjectName, ObjectName>();

        // Get the list of all deployed applications, use the obtained
        // information in order to fill nonVersionedEjbsAndWars and
        // versionedEjbsAndWars.
        {
            Set<ObjectName> realEjbMBeans = new HashSet<ObjectName>();
            Set<ObjectName> virtualEjbMBeans = new HashSet<ObjectName>();
            for (ObjectName ejbjar : (Set<ObjectName>) this.mbscnx.queryNames(new ObjectName("*:j2eeType=EJBModule,*"), null)) {
                if ("true".equals(ejbjar.getKeyProperty("virtualContext"))) {
                    virtualEjbMBeans.add(ejbjar);
                } else {
                    realEjbMBeans.add(ejbjar);
                }
            }
            for (ObjectName virtualEjbMBean : virtualEjbMBeans) {
                String baseName = virtualEjbMBean.getKeyProperty("name").substring("VirtualContainer-".length());
                for (ObjectName ejbMBean : realEjbMBeans) {
                    // If the EJB-JAR is in an EAR, then the JNDI prefix will
                    // be set using the EAR's (J2EEApplication) name. Therefore
                    // check for the naming base in both attributes.
                    if (ejbMBean.getKeyProperty("name").startsWith(baseName)) {
                        versionedEjbsAndWars.put(ejbMBean, virtualEjbMBean);
                    } else {
                        String appName = ejbMBean.getKeyProperty("J2EEApplication");
                        if (appName != null && !appName.equals("none") && appName.startsWith(baseName)) {
                            versionedEjbsAndWars.put(ejbMBean, virtualEjbMBean);
                        }
                    }
                }
            }

            Set<ObjectName> realWarMBeans = new HashSet<ObjectName>();
            Set<ObjectName> virtualWarMBeans = new HashSet<ObjectName>();
            for (ObjectName war : (Set<ObjectName>) this.mbscnx.queryNames(new ObjectName("*:j2eeType=WebModule,*"), null)) {
                if ("true".equals(war.getKeyProperty("virtualContext"))) {
                    virtualWarMBeans.add(war);
                } else {
                    realWarMBeans.add(war);
                }
            }
            for (ObjectName virtualWarMBean : virtualWarMBeans) {
                String baseName = virtualWarMBean.getKeyProperty("name");
                for (ObjectName warMBean : realWarMBeans) {
                    if (warMBean.getKeyProperty("name").startsWith(baseName)) {
                        versionedEjbsAndWars.put(warMBean, virtualWarMBean);
                    }
                }
            }
            realEjbMBeans.removeAll(versionedEjbsAndWars.keySet());
            realWarMBeans.removeAll(versionedEjbsAndWars.keySet());
            nonVersionedEjbsAndWars.addAll(realEjbMBeans);
            nonVersionedEjbsAndWars.addAll(realWarMBeans);
        }

        // Process all versioned items
        for (Map.Entry<ObjectName, ObjectName> versionedEjbOrWar : versionedEjbsAndWars.entrySet()) {
            for (Map.Entry<String, String> context : ((Map<String, String>) this.mbscnx.getAttribute(versionedEjbOrWar
                .getValue(), "Contexts")).entrySet()) {
                ObjectName j2eeApplication;
                String applicationName = versionedEjbOrWar.getKey().getKeyProperty("J2EEApplication");
                if (applicationName == null || applicationName.equals("none") || applicationName.equals("null")) {
                    // Not an EAR, therefore applicationName is the WAR or
                    // EJB-JAR archive's name
                    applicationName = versionedEjbOrWar.getKey().getKeyProperty("name");
                    j2eeApplication = versionedEjbOrWar.getKey();

                    // Filter out bizarre characters from applicationName
                    int doubleSlash = applicationName.indexOf("//");
                    if (doubleSlash >= 0) {
                        int nextSlash = applicationName.indexOf('/', doubleSlash + 2);
                        if (nextSlash >= 0) {
                            // URL like //localhost/applicationName
                            applicationName = applicationName.substring(nextSlash + 1);
                        }
                    }

                    if (applicationName.length() == 0) {
                        applicationName = "[ ROOT ]";
                    }
                } else {
                    Set<ObjectName> j2eeApplications = this.mbscnx.queryNames(new ObjectName("*:j2eeType=J2EEApplication,name="
                        + applicationName + ",*"), null);
                    if (j2eeApplications.size() == 0) {
                        // Application not found... Deployment issue?
                        continue;
                    }
                    j2eeApplication = j2eeApplications.iterator().next();
                }

                URL j2eeApplicationURL = new URL(this.getPath(j2eeApplication));
                Object[] opParams = new Object[] {j2eeApplicationURL};
                String[] opSignature = new String[] {URL.class.getName()};
                String versionID = (String) this.mbscnx.invoke(this.versioningService, "getVersionID", opParams, opSignature);
                if (versionID != null && versionID.startsWith("-version")) {
                    applicationName = applicationName.replace(versionID, "");
                    String versionNumber = versionID.substring(8);

                    if (context.getKey().endsWith("-version" + versionNumber)
                        || context.getKey().endsWith("_version" + versionNumber + "/")) {
                        // The version currently being processed [context.key]
                        // and the current application version [version] are the
                        // same, save details
                        ApplicationInformation applicationInformation = applications.get(applicationName);
                        if (applicationInformation == null) {
                            applicationInformation = new ApplicationInformation();
                            applications.put(applicationName, applicationInformation);
                        }
                        applicationInformation.versions.put(versionNumber, new VersionInformation(context.getValue(),
                            j2eeApplication));
                        applicationInformation.managers.add(versionedEjbOrWar.getValue());
                    }
                }
            }
        }

        // Process non-versioned EARs. This will also clean up
        // nonVersionedEjbsAndWars to make it have only the standalone archives
        for (ObjectName ear : (Set<ObjectName>) this.mbscnx.queryNames(new ObjectName("*:j2eeType=J2EEApplication,*"), null)) {
            String applicationName = ear.getKeyProperty("name");
            if (!applicationName.contains("-version") && !applications.containsKey(applicationName)) {
                applications.put(applicationName, null);

                for (String module : (String[]) this.mbscnx.getAttribute(ear, "modules")) {
                    nonVersionedEjbsAndWars.remove(new ObjectName(module));
                }
            }
        }

        // Process non-versioned standalone archives
        for (ObjectName nonVersionedEjbOrWar : nonVersionedEjbsAndWars) {
            String applicationName = nonVersionedEjbOrWar.getKeyProperty("name");

            // Filter out bizarre characters from applicationName
            int doubleSlash = applicationName.indexOf("//");
            if (doubleSlash >= 0) {
                int nextSlash = applicationName.indexOf('/', doubleSlash + 2);
                if (nextSlash >= 0) {
                    // URL like //localhost/applicationName
                    applicationName = applicationName.substring(nextSlash + 1);
                }
            }

            if (applicationName.length() == 0) {
                applicationName = "[ ROOT ]";
            }

            if (!applications.containsKey(applicationName)) {
                applications.put(applicationName, null);
            }
        }

        return applications;
    }

    @Override
    @SuppressWarnings("unchecked")
    public List<ApplicationBean> listOfApplications() throws Exception {
        this.checkJMXConnection();
        this.appendToLog("Getting list of applications");
        List<ApplicationBean> applications = new ArrayList<ApplicationBean>();

        this.appendToLog("\tGetting deployed applications");
        for (Map.Entry<String, ApplicationInformation> application : this.getApplicationsList().entrySet()) {
            if (application.getValue() == null) {
                ApplicationBean applicationBean = new ApplicationBean(application.getKey());
                applicationBean.setState(ServerAction.STATE_DEPLOYED);
                applications.add(applicationBean);
            } else {
                for (Map.Entry<String, VersionInformation> versionInformation : application.getValue().versions.entrySet()) {
                    ApplicationBean applicationBean = new ApplicationBean(application.getKey(), versionInformation.getKey());
                    applicationBean.setState(ServerAction.STATE_DEPLOYED);
                    applicationBean.setPolicy(versionInformation.getValue().policy);
                    applications.add(applicationBean);
                }
            }
        }

        this.appendToLog("\tGetting present yet not deployed applications");
        try {
            for (String application : (List<String>) this.mbscnx.getAttribute(this.j2eeServer, "deployableEars")) {
                ApplicationBean applicationBean = new ApplicationBean(new File(application).getName());
                applicationBean.setState(ServerAction.STATE_PRESENT);
                applications.add(applicationBean);
            }
        } catch (AttributeNotFoundException ignored) {
            // EAR service not present? Ignore
        }

        try {
            for (String application : (List<String>) this.mbscnx.getAttribute(this.j2eeServer, "deployableJars")) {
                ApplicationBean applicationBean = new ApplicationBean(new File(application).getName());
                applicationBean.setState(ServerAction.STATE_PRESENT);
                applications.add(applicationBean);
            }
        } catch (AttributeNotFoundException ignored) {
            // EJB service not present? Ignore
        }

        try {
            for (String application : (List<String>) this.mbscnx.getAttribute(this.j2eeServer, "deployableWars")) {
                ApplicationBean applicationBean = new ApplicationBean(new File(application).getName());
                applicationBean.setState(ServerAction.STATE_PRESENT);
                applications.add(applicationBean);
            }
        } catch (AttributeNotFoundException ignored) {
            // WAR service not present? Ignore
        }

        this.appendToLog("Got the list of applications");
        return applications;
    }

    @Override
    @SuppressWarnings("unchecked")
    public String upload(final ApplicationBean application) throws Exception {
        this.checkJMXConnection();
        this.appendToLog("Uploading ApplicationBean '" + application.toString() + "'");

        FileInputStream inputStream = new FileInputStream(application.getFile());
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        byte[] buf = new byte[JonasServerAction.BUFFER_SIZE];
        // Read bytes
        int len;
        while ((len = inputStream.read(buf)) > 0) {
            baos.write(buf, 0, len);
        }
        byte[] bytesOfFile = baos.toByteArray();

        Object[] opParams = new Object[] {bytesOfFile, application.getFile().getName(), false};
        String[] opSignature = new String[] {"[B", "java.lang.String", "boolean"};

        String sendFile = (String) this.mbscnx.invoke(this.j2eeServer, "sendFile", opParams, opSignature);
        URL url = new URL("file:" + sendFile);
        this.appendToLog("Call to J2EEServer.sendFile successful, ApplicationBean saved as '" + url.toString() + "'");
        String result = url.getFile();

        // Get if deployment manager is in development mode
        ObjectName deploymentMonitor = ((Set<ObjectName>) this.mbscnx.queryNames(new ObjectName(
            "*:type=service,name=depmonitor"), null)).iterator().next();
        boolean developmentMode = ((Boolean) this.mbscnx.getAttribute(deploymentMonitor, "developmentMode")).booleanValue();

        if (developmentMode) {
            this.autoDeploy.add(result);
            this.appendToLog("Development mode is active. This file is therefore expected to be auto-deployed by the server, "
                + "the call to the JonasServerAction.deploy method will just wait for the auto-deployer...");
        }

        return result;
    }

    @Override
    public void deploy(final String appName) throws Exception {
        this.checkJMXConnection();
        Object[] opParams = new Object[] {appName};
        String[] opSignature = new String[] {"java.lang.String"};

        if (this.autoDeploy.contains(appName)) {
            this.appendToLog("Waiting for the JOnAS auto-deployer to deploy '" + appName + "' automatically");
            boolean isDeployed = false;
            int count = 0;
            do {
                Thread.sleep(1000);
                try {
                    isDeployed = ((Boolean) this.mbscnx.invoke(this.j2eeServer, "isDeployed", opParams, opSignature))
                        .booleanValue();
                } catch (Exception e) {
                    // Ignored
                }
                if (isDeployed) {
                    // Deployed :)
                    break;
                }
                this.appendToLog("\tWaiting for the application '" + appName + "' to get auto-deployed (" + count + "/"
                    + JonasServerAction.AUTODEPLOY_TIMEOUT + ")...");
                count++;
            } while (!isDeployed && count < JonasServerAction.AUTODEPLOY_TIMEOUT);

            this.autoDeploy.remove(appName);
            if (isDeployed) {
                this.appendToLog("Application '" + appName + "' has been deployed by the JOnAS auto-deployer");
            } else {
                throw new DeployerException("Application '" + appName
                    + "' did not get deployed by the JOnAS auto-deployer after more than "
                    + JonasServerAction.AUTODEPLOY_TIMEOUT
                    + " seconds! Try deploying manually by clicking the \"Retry server\" button...");
            }
        } else {
            this.appendToLog("Deploying application '" + appName + "'");
            this.mbscnx.invoke(this.j2eeServer, "deploy", opParams, opSignature);
            this.appendToLog("Call to J2EEServer.deploy successful for application '" + appName + "'");
        }
    }

    @Override
    public String setDefault(final String appName) throws Exception {
        this.checkJMXConnection();
        this.appendToLog("Setting application '" + appName + "' as default");

        if (this.versioningService == null) {
            throw new IllegalStateException("The versioning service is not enabled. Please enable it and "
                + "redeploy all versioned applications before setting any version as default.");
        }

        // Find the application
        String newDefaultVersion = null;
        String oldDefaultVersionPath = null;
        ApplicationInformation applicationInformation = null;
        for (ApplicationInformation ai : this.getApplicationsList().values()) {
            if (ai != null) {
                for (Map.Entry<String, VersionInformation> vi : ai.versions.entrySet()) {
                    String path = this.getPath(vi.getValue().mbean);
                    if (path.endsWith(appName)) {
                        applicationInformation = ai;
                        newDefaultVersion = vi.getKey();
                        break;
                    }
                }
                if (applicationInformation != null) {
                    for (Map.Entry<String, VersionInformation> vi : ai.versions.entrySet()) {
                        if ("Default".equals(vi.getValue().policy)) {
                            oldDefaultVersionPath = this.getPath(vi.getValue().mbean);
                            break;
                        }
                    }
                    break;
                }
            }
        }

        if (applicationInformation == null) {
            throw new IllegalArgumentException("No versioned application found for '" + appName
                + "'. Please make sure that application is versioned.");
        }

        // Set new default
        this.setPolicy(applicationInformation.managers, newDefaultVersion, "Default");
        if (oldDefaultVersionPath != null) {
            this.appendToLog("   Old default was '" + oldDefaultVersionPath + "'");
        } else {
            this.appendToLog("   There was no old default version");
        }
        return oldDefaultVersionPath;
    }

    protected String getPath(final ObjectName mbean) throws AttributeNotFoundException, InstanceNotFoundException,
        MBeanException, ReflectionException, IOException {
        String j2eeType = mbean.getKeyProperty("j2eeType");
        if ("J2EEApplication".equals(j2eeType)) {
            return this.mbscnx.getAttribute(mbean, "earUrl").toString();
        } else if ("EJBModule".equals(j2eeType)) {
            return this.mbscnx.getAttribute(mbean, "url").toString();
        } else if ("WebModule".equals(j2eeType)) {
            return this.mbscnx.getAttribute(mbean, "warUrl").toString();
        } else {
            throw new IllegalArgumentException("Unknown j2eeType: \"" + j2eeType + "\"");
        }
    }

    @SuppressWarnings("unchecked")
    protected void setPolicy(final Set<ObjectName> managers, final String version, final String policy)
        throws AttributeNotFoundException, InstanceNotFoundException, MBeanException, ReflectionException, IOException {
        this.appendToLog("   Setting version '" + version + "' as " + policy);
        String versionID_WAR = "-version" + version;
        String versionID_EJB = "version" + version + "/";
        for (ObjectName manager : managers) {
            for (String context : ((Map<String, String>) this.mbscnx.getAttribute(manager, "Contexts")).keySet()) {
                if (context.contains(versionID_WAR) || context.contains(versionID_EJB)) {
                    Object[] opParams = new Object[] {context, policy};
                    String[] opSignature = new String[] {"java.lang.String", "java.lang.String"};
                    this.mbscnx.invoke(manager, "rebindContext", opParams, opSignature);
                    break;
                }
            }
        }
    }

    @Override
    public void undeploy(final String appName) throws Exception {
        this.checkJMXConnection();
        this.appendToLog("Undeploying application '" + appName + "'");

        Object[] opParams = new Object[] {appName};
        String[] opSignature = new String[] {"java.lang.String"};
        this.mbscnx.invoke(this.j2eeServer, "undeploy", opParams, opSignature);
        this.appendToLog("Call to J2EEServer.undeploy successful for application '" + appName + "'");
    }

    @Override
    public void erase(final String appName) throws Exception {
        this.checkJMXConnection();
        this.appendToLog("Erasing application '" + appName + "'");

        Object[] opParams = new Object[] {appName};
        String[] opSignature = new String[] {"java.lang.String"};
        this.mbscnx.invoke(this.j2eeServer, "removeModuleFile", opParams, opSignature);
        this.appendToLog("Call to J2EEServer.removeModuleFile successful for application '" + appName + "'");
    }

    @Override
    @SuppressWarnings("unchecked")
    public ApplicationBean getApplicationBean(final String appName) throws Exception {
        this.checkJMXConnection();
        this.appendToLog("Getting ApplicationBean for application '" + appName + "'");

        URL url = new URL("file:" + appName);
        for (Map.Entry<String, ApplicationInformation> application : this.getApplicationsList().entrySet()) {
            if (application.getValue() != null) {
                for (Map.Entry<String, VersionInformation> version : application.getValue().versions.entrySet()) {
                    if (url.equals(new URL("file:" + this.getPath(version.getValue().mbean)))) {
                        ApplicationBean applicationBean = new ApplicationBean();
                        applicationBean.setName(application.getKey());
                        applicationBean.setPolicy(version.getValue().policy);
                        applicationBean.setState(ServerAction.STATE_DEPLOYED);
                        applicationBean.setVersion(version.getKey());
                        this.appendToLog("ApplicationBean for application '" + appName
                            + "' is a deployed versioned application, returning '" + applicationBean.toString() + "'");
                        return applicationBean;
                    }
                }
            }
        }

        for (String application : (List<String>) this.mbscnx.getAttribute(this.j2eeServer, "deployedEars")) {
            if (url.equals(new URL("file:" + application))) {
                ApplicationBean applicationBean = new ApplicationBean(new File(application).getName());
                applicationBean.setState(ServerAction.STATE_DEPLOYED);
                this.appendToLog("ApplicationBean for application '" + appName
                    + "' is a deployed non-versioned application, returning '" + applicationBean.toString() + "'");
                return applicationBean;
            }
        }

        for (String application : (List<String>) this.mbscnx.getAttribute(this.j2eeServer, "deployedJars")) {
            if (url.equals(new URL("file:" + application))) {
                ApplicationBean applicationBean = new ApplicationBean(new File(application).getName());
                applicationBean.setState(ServerAction.STATE_DEPLOYED);
                this.appendToLog("ApplicationBean for application '" + appName
                    + "' is a deployed non-versioned application, returning '" + applicationBean.toString() + "'");
                return applicationBean;
            }
        }

        for (String application : (List<String>) this.mbscnx.getAttribute(this.j2eeServer, "deployedWars")) {
            if (url.equals(new URL("file:" + application))) {
                ApplicationBean applicationBean = new ApplicationBean(new File(application).getName());
                applicationBean.setState(ServerAction.STATE_DEPLOYED);
                this.appendToLog("ApplicationBean for application '" + appName
                    + "' is a deployed non-versioned application, returning '" + applicationBean.toString() + "'");
                return applicationBean;
            }
        }

        for (String application : (List<String>) this.mbscnx.getAttribute(this.j2eeServer, "deployableEars")) {
            if (url.equals(new URL("file:" + application))) {
                ApplicationBean applicationBean = new ApplicationBean(new File(application).getName());
                applicationBean.setState(ServerAction.STATE_PRESENT);
                this.appendToLog("ApplicationBean for application '" + appName + "' is a present application, returning '"
                    + applicationBean.toString() + "'");
                return applicationBean;
            }
        }

        for (String application : (List<String>) this.mbscnx.getAttribute(this.j2eeServer, "deployableJars")) {
            if (url.equals(new URL("file:" + application))) {
                ApplicationBean applicationBean = new ApplicationBean(new File(application).getName());
                applicationBean.setState(ServerAction.STATE_PRESENT);
                this.appendToLog("ApplicationBean for application '" + appName + "' is a present application, returning '"
                    + applicationBean.toString() + "'");
                return applicationBean;
            }
        }

        for (String application : (List<String>) this.mbscnx.getAttribute(this.j2eeServer, "deployableWars")) {
            if (url.equals(new URL("file:" + application))) {
                ApplicationBean applicationBean = new ApplicationBean(new File(application).getName());
                applicationBean.setState(ServerAction.STATE_PRESENT);
                this.appendToLog("ApplicationBean for application '" + appName + "' is a present application, returning '"
                    + applicationBean.toString() + "'");
                return applicationBean;
            }
        }

        this.appendToLog("Application '" + appName + "' not found");
        return null;
    }

    @Override
    @SuppressWarnings("unchecked")
    public int getActiveSessions(final String appName) throws Exception {
        this.checkJMXConnection();

        List<String> names = new ArrayList<String>();
        // Find all names for the application (EARs may have many)
        {
            URL url = new URL("file:" + appName);

            if (names.size() == 0) {
                for (ObjectName ear : (Set<ObjectName>) this.mbscnx.queryNames(new ObjectName("*:j2eeType=J2EEApplication,*"),
                    null)) {
                    if (url.equals(this.mbscnx.getAttribute(ear, "earUrl"))) {
                        String[] modules = (String[]) this.mbscnx.getAttribute(ear, "modules");
                        for (String module : modules) {
                            ObjectName moduleON = new ObjectName(module);
                            if ("WebModule".equals(moduleON.getKeyProperty("j2eeType"))) {
                                names.add(moduleON.getKeyProperty("name"));
                            }
                        }
                    }
                }
            }

            if (names.size() == 0) {
                for (ObjectName war : (Set<ObjectName>) this.mbscnx.queryNames(new ObjectName("*:j2eeType=WebModule,*"), null)) {
                    if (url.equals(this.mbscnx.getAttribute(war, "warURL"))) {
                        names.add(war.getKeyProperty("name"));
                    }
                }
            }
        }

        if (names.size() == 0) {
            throw new IllegalArgumentException("No application for appName '" + appName + "'");
        }

        int totalActiveSessions = 0;
        for (String fullPath : names) {
            while (fullPath.startsWith("/")) {
                fullPath = fullPath.substring(1);
            }
            int slash = fullPath.indexOf('/');

            String host = fullPath.substring(0, slash);
            String path = fullPath.substring(slash);

            for (ObjectName manager : (Set<ObjectName>) this.mbscnx.queryNames(new ObjectName("*:type=Manager,path=" + path
                + ",host=" + host), null)) {
                totalActiveSessions += ((Integer) this.mbscnx.getAttribute(manager, "activeSessions")).intValue();
            }
        }
        return totalActiveSessions;
    }

    @Override
    @SuppressWarnings("unchecked")
    public int getActiveSessions() throws Exception {
        this.checkJMXConnection();

        int totalActiveSessions = 0;
        for (ObjectName manager : (Set<ObjectName>) this.mbscnx.queryNames(new ObjectName("*:type=Manager,*"), null)) {
            totalActiveSessions += ((Integer) this.mbscnx.getAttribute(manager, "activeSessions")).intValue();
        }
        return totalActiveSessions;
    }

    @Override
    public void maintain() throws Exception {
        // TODO Auto-generated method stub
    }

    @Override
    @SuppressWarnings("unchecked")
    public void start() throws Exception {
        ObjectName cdObjectName = null;
        MBeanServerConnection managerConnection = null;

        try {
            this.checkJMXConnection();
        } catch (IOException e) {
            if (this.managerConnector != null) {
                String url = this.managerConnector.getConnectorUrl();
                String username = this.managerConnector.getUsername();
                String password = this.managerConnector.getPassword();

                this.appendToLog("Connection to server's JMX server failed.");
                this.appendToLog("Trying the associated Cluster Daemon's JMX server on URL '" + url + "'");

                Map<String, Object> env = new HashMap<String, Object>();
                if (username != null && password != null) {
                    String[] creds = {username, password};
                    env.put(JMXConnector.CREDENTIALS, creds);
                }
                JMXConnector c = JMXConnectorFactory.connect(new JMXServiceURL(url), env);
                managerConnection = c.getMBeanServerConnection();
                cdObjectName = ((Set<ObjectName>) managerConnection.queryNames(new ObjectName("*:type=ClusterDaemon,*"), null))
                    .iterator().next();

                this.appendToLog("JMX server connection OK for Cluster Daemon for server '" + this.name
                    + "', ClusterDaemon is '" + cdObjectName + "'");
            } else {
                throw e;
            }
        }

        this.appendToLog("Starting server");

        if (managerConnection == null || cdObjectName == null) {
            this.mbscnx.invoke(this.j2eeServer, "start", null, null);
        } else {
            Object[] opParams = new Object[] {this.name, null, null};
            String[] opSignature = new String[] {String.class.getName(), String.class.getName(), String.class.getName()};
            managerConnection.invoke(cdObjectName, "start", opParams, opSignature);
        }

        this.appendToLog("Server started");
    }

    @Override
    public void stop() throws Exception {
        this.checkJMXConnection();
        this.appendToLog("Stopping server");

        this.mbscnx.invoke(this.j2eeServer, "stop", null, null);

        this.appendToLog("Server stopped");
    }

    @Override
    public boolean isStarted() throws Exception {
        this.appendToLog("Checking server state via JMX");

        boolean result = true;
        try {
            this.checkJMXConnection();
        } catch (IOException e) {
            result = false;
        }

        this.appendToLog("Server.started is '" + result + "'");
        return result;
    }

    @Override
    @SuppressWarnings("unchecked")
    public boolean enableOrDisableApplications(final boolean enable) throws Exception {
        this.checkJMXConnection();
        this.appendToLog("Enabling or disabling all applications on this server");

        boolean failed = false;
        for (ObjectName war : (Set<ObjectName>) this.mbscnx.queryNames(new ObjectName("*:j2eeType=WebModule,*"), null)) {
            if (war.getKeyProperty("virtualContext") == null) {
                String path = (String) this.mbscnx.getAttribute(war, "path");
                if (path == null || path.length() < 1) {
                    path = "/";
                }
                ObjectName filter;
                try {
                    filter = ((Set<ObjectName>) this.mbscnx.queryNames(new ObjectName("*:type=J2EEFilter,path=" + path), null))
                        .iterator().next();
                } catch (NoSuchElementException e) {
                    this.appendToLog("\tApplication on path '" + path
                        + "' doesn't have the OnlyAllowUsersWithSessionFilter filter! Please "
                        + "import the filter JAR from the jadort-samples package and set it in the application's descriptor");
                    failed = true;
                    continue;
                }

                this.mbscnx.setAttribute(filter, new Attribute("active", Boolean.valueOf(!enable)));
                this.appendToLog("\tFilter for application on path '" + path + "' has been set as " + (enable ? "in" : "")
                    + "active, application is therefore " + (enable ? "enabled" : "disabled"));
            }
        }
        if (failed) {
            this.appendToLog("At least one application on this server couldn't be " + (enable ? "enabled" : "disabled"));
            return false;
        } else {
            this.appendToLog("All applications on this server are now " + (enable ? "enabled" : "disabled"));
            return true;
        }
    }
}
