/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008 Bull S.A.S.
 * Copyright (C) 2008 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ModProxyBalancer.java 2854 2008-12-03 16:09:32Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.service.action.modProxyBalancer;

import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.codec.binary.Base64;

/**
 * The mod_proxy_balancer manager.
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
public class ModProxyBalancer {

    private String login;

    private String password;

    private String url;

    protected List<Balancer> balancers;

    protected String nonce;

    /**
     * Constructor.
     * 
     * @param url : mod_proxy_balancer manager url
     * @param login : login of mod_proxy_balancer manager web application
     * @param password : password of mod_proxy_balancer manager web application
     */
    public ModProxyBalancer(final String url, final String login, final String password) {
        this.url = url;
        this.login = login;
        this.password = password;
    }

    /**
     * Constructor.
     * 
     * @param url : mod_proxy_balancer manager url
     */
    public ModProxyBalancer(final String url) {
        this(url, null, null);
    }

    protected void updateBalancerInfo() throws Exception {
        String response = this.openConnection();
        this.parseAndUpdateModProxyBalancer(response);
    }

    /**
     * parse the html response and update the modProxyBalacncer Info
     * 
     * @param response : the html response to be parsed
     */
    protected void parseAndUpdateModProxyBalancer(String response) {
        this.balancers = new ArrayList<Balancer>();
        int l1 = -1;

        while ((l1 = response.indexOf("<h3>")) > -1) {
            response = response.substring(l1 + 4);
            String s = response.substring(0, response.indexOf("</h3>"));
            String balancerName = (s.trim()).substring(s.lastIndexOf("balancer://") + 11);

            int l2 = -1;
            List<Worker> workers = new ArrayList<Worker>();
            while ((l2 = response.indexOf("<td><a")) > -1) {
                response = response.substring(l2 + 6);
                int l3 = response.indexOf("</a></td>");
                s = response.substring(0, l3);

                String nonce = (s.trim()).substring(s.indexOf("&nonce=") + 6, s.indexOf("\">") - 1);
                this.nonce = nonce;
                String workerUrl = (s.trim()).substring(s.indexOf("\">") + 1);
                response = response.substring(l3 + 13);

                for (int i = 0; i < 4; i++) {
                    response = response.substring(response.indexOf("</td><td>") + 9);
                }

                String status = (response.trim()).substring(0, response.indexOf("</td><td>"));
                workers.add(new Worker(workerUrl, status.equalsIgnoreCase("OK")));

            }
            this.balancers.add(new Balancer(balancerName, workers));
        }
    }

    /**
     * open the http connection and send the request to ModProxyBalancer url
     * 
     * @return the html content of the requested page
     */
    protected String openConnection() throws Exception {
        return this.openConnection(null);
    }

    /**
     * open the http connection and send the request to ModProxyBalancer url
     * 
     * @param param : parameters that will be passed to the url
     * @return the html content of the requested page
     * @throws Exception
     */
    protected String openConnection(final String param) throws Exception {
        HttpURLConnection con = null;
        String response = null;

        if (param != null) {
            con = (HttpURLConnection) new URL(this.url + "?" + param).openConnection();
        } else {
            con = (HttpURLConnection) new URL(this.url).openConnection();
        }
        con.setRequestMethod("GET");

        if (this.login != null) {
            Base64 enc = new Base64();
            String authData = this.login + ":" + this.password;
            byte[] base64AuthData = enc.encode(authData.getBytes());
            con.setRequestProperty("Authorization", "Basic " + new String(base64AuthData, "UTF-8"));
        }

        int responseCode = con.getResponseCode();

        if (responseCode == HttpURLConnection.HTTP_OK) {
            InputStream is = con.getInputStream();
            byte[] b = new byte[is.available()];
            int nbOctetsLus = is.read(b);
            response = new String(b, 0, nbOctetsLus);
        } else {
            throw new Exception("Connection error: " + con.getResponseMessage());
        }

        return response;
    }

    public String getLogin() {
        return this.login;
    }

    public void setLogin(final String login) {
        this.login = login;
    }

    public String getPassword() {
        return this.password;
    }

    public void setPassword(final String password) {
        this.password = password;
    }

    public String getUrl() {
        return this.url;
    }

    public void setUrl(final String url) {
        this.url = url;
    }

    public List<Balancer> getBalancers() throws Exception {
        this.updateBalancerInfo();
        return this.balancers;
    }

    /**
     * Activate a worker
     * 
     * @param balancerName : the name of the target balancer
     * @param workerUrl : the url of the target worker that will be activated
     * @throws Exception
     */
    public void activateWorker(final String balancerName, final String workerUrl) throws Exception {
        this.updateBalancerInfo();
        String param = "dw=Enable&nonce=" + this.nonce + "&w=" + URLEncoder.encode(workerUrl, "UTF-8") + "&b=" + balancerName;
        String response = this.openConnection(param);
        this.parseAndUpdateModProxyBalancer(response);
    }

    /**
     * Deactivate a worker
     * 
     * @param balancerName : the name of the target balancer
     * @param workerUrl : the url of the target worker that will be deactivated
     * @throws Exception
     */
    public void deactivateWorker(final String balancerName, final String workerUrl) throws Exception {
        this.updateBalancerInfo();
        String param = "dw=Disable&w=" + URLEncoder.encode(workerUrl, "UTF-8") + "&b=" + balancerName + "&nonce=" + this.nonce;
        String response = this.openConnection(param);
        this.parseAndUpdateModProxyBalancer(response);
    }
}
