/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008 Bull S.A.S.
 * Copyright (C) 2008 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JadortServiceImpl.java 2952 2009-01-09 15:11:13Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.service.implementation;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.ejb.Local;
import javax.ejb.Remote;
import javax.ejb.Stateful;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.QueueSender;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.naming.InitialContext;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.UnmarshalException;
import javax.xml.bind.Unmarshaller;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;

import org.ow2.jasmine.deployme.generated.Configuration;
import org.ow2.jasmine.deployme.generated.Configuration.Protocols;
import org.ow2.jasmine.deployme.generated.Topology.Domains;
import org.ow2.jasmine.deployme.generated.Topology.Domains.Domain.ClusterDaemons;
import org.ow2.jasmine.deployme.generated.Topology.Domains.Domain.Servers;
import org.ow2.jasmine.jadort.api.IJadortService;
import org.ow2.jasmine.jadort.api.JadortServiceException;
import org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean;
import org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean;
import org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean;
import org.ow2.jasmine.jadort.api.entities.deployment.WorkerProgressBean;
import org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean.ActionState;
import org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean.Step;
import org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean.ServerProgressState;
import org.ow2.jasmine.jadort.api.entities.deployment.WorkerProgressBean.WorkerProgressState;
import org.ow2.jasmine.jadort.api.entities.topology.ConnectorBean;
import org.ow2.jasmine.jadort.api.entities.topology.GroupBean;
import org.ow2.jasmine.jadort.api.entities.topology.ServerBean;
import org.ow2.jasmine.jadort.api.entities.topology.TopologyBean;
import org.ow2.jasmine.jadort.api.entities.topology.WorkerBean;
import org.ow2.jasmine.jadort.api.entities.topology.ServerBean.Type;
import org.ow2.jasmine.jadort.service.action.ServerAction;
import org.ow2.jasmine.jadort.service.action.WorkerAction;
import org.ow2.jasmine.jadort.service.topology.xml.Balancer;
import org.ow2.jasmine.jadort.service.topology.xml.Server;
import org.ow2.jasmine.jadort.service.topology.xml.Target;
import org.ow2.jasmine.jadort.service.topology.xml.URLConnector;
import org.ow2.jasmine.jadort.service.topology.xml.Worker;
import org.ow2.jasmine.jadort.service.topology.xml.Topology.Group;
import org.xml.sax.SAXException;

/**
 * This class provides default implementations for the IJadortService interface.
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
@Local(value = IJadortService.class)
@Remote(value = IJadortService.class)
@Stateful(mappedName = "ow2.jasmine.jadort.ServiceSFB")
public class JadortServiceImpl implements IJadortService {

    // Constants used internally by JadortServiceImpl
    private final static int BUFFER_SIZE = 16384;

    private final static String XSD_PATH_JADORT = "XMLSchema/topology_JadortSpecific.xsd";

    private final static String XSD_PATH_DEPLOYME = "xml/topology.xsd";

    // Constants shared within the package (mostly JadortServiceJMSActions)
    final static String QUEUE_NAME = "queue/JadortServiceJMSQueue";

    final static String SERVER_PROGRESS_ID = "serverProgressId";

    final static String WORKER_PROGRESS_ID = "workerProgressId";

    final static String APPLICATION_ID = "applicationId";

    final static String ACTION_TYPE = "actionType";

    final static String GROUP_CLUSTERED = "groupClustered";

    final static String APPLICATION_NAME = "applicationName";

    final static String OPERATION_TYPE = "operationType";

    // Constants used for tests
    static boolean DUMMY = false;

    /**
     * Action type
     */
    public enum ActionType {
        UPLOAD, DEPLOY, SET_DEFAULT, UNDEPLOY, ERASE, START_SERVER, MAINTAIN_SERVER, STOP_SERVER, START_WORKER, STOP_WORKER, DISABLE_APPLICATIONS, ENABLE_APPLICATIONS
    }

    private OperationStateBean operation;

    @PersistenceContext
    protected EntityManager em;

    protected QueueConnection qc;

    protected QueueSession session;

    protected Queue queue;

    @PostConstruct
    protected void initialize() {
        try {
            InitialContext ctx = new InitialContext();
            this.queue = (Queue) ctx.lookup(JadortServiceImpl.QUEUE_NAME);
            QueueConnectionFactory factory = (QueueConnectionFactory) ctx.lookup("QCF");
            this.qc = factory.createQueueConnection();
            this.session = this.qc.createQueueSession(false, Session.AUTO_ACKNOWLEDGE);
        } catch (Exception e) {
            IllegalStateException exception = new IllegalStateException("Failed initializing a JadortService stateful bean: "
                + e.getMessage(), e);
            System.out.println(exception.getMessage());
            e.printStackTrace();
            throw exception;
        }
    }

    @PreDestroy
    protected void terminate() {
        try {
            this.session.close();
            this.qc.close();
        } catch (Exception e) {
            IllegalStateException exception = new IllegalStateException("Failed terminating a JadortService stateful bean: "
                + e.getMessage(), e);
            System.out.println(exception.getMessage());
            e.printStackTrace();
            throw exception;
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException if the getOperationsList() method can not
     *         be called in the current step. (this method can be called only in
     *         SELECT_OPERATION step)
     */
    @SuppressWarnings("unchecked")
    public List<OperationStateBean> getOperationsList() throws JadortServiceException {
        synchronized (this) {
            this.checkStep(Step.SELECT_OPERATION);

            List<OperationStateBean> result = new ArrayList<OperationStateBean>();

            // Create a dummy list which contains only the needed fields
            for (OperationStateBean operation : (List<OperationStateBean>) this.em.createQuery(
                "select o from JaDOrT_OperationStateBean o order by o.date asc").getResultList()) {
                OperationStateBean temp = new OperationStateBean();
                temp.setAimedServerProgressState(operation.getAimedServerProgressState());
                temp.setAimedServerProgressProgress(operation.getAimedServerProgressProgress());
                temp.setCurrentStep(operation.getCurrentStep());
                temp.setDate(operation.getDate());
                temp.setId(operation.getId());
                temp.setName(operation.getName());
                result.add(temp);
            }
            return result;
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException If fetching fails.
     */
    public OperationStateBean getCurrentOperation() throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();

            if (this.operation == null) {
                return null;
            } else {
                // Return a copy since operation has been loaded by lazy
                // initialization. Remember to copy all list attributes as well!
                OperationStateBean bean = new OperationStateBean();
                bean.setAimedServerProgressState(this.operation.getAimedServerProgressState());
                bean.setAimedServerProgressProgress(this.operation.getAimedServerProgressProgress());
                bean.setAimedWorkerProgressState(this.operation.getAimedWorkerProgressState());
                bean.setAimedWorkerProgressProgress(this.operation.getAimedWorkerProgressProgress());
                bean.setApplication(this.operation.getApplication());
                bean.setCurrentStep(this.operation.getCurrentStep());
                bean.setDate(this.operation.getDate());
                bean.setServerProgressList(this.fetchServerProgressList());
                bean.setWorkerProgressList(this.fetchWorkerProgressList());
                bean.setAllServerProgressList(this.fetchMigratedServerProgressList());
                bean.setAllWorkerProgressList(this.fetchMigratedWorkerProgressList());
                bean.setId(this.operation.getId());
                bean.setName(this.operation.getName());
                bean.setSelectedGroup(this.fetchSelectedGroup());
                bean.setSelectedServers(this.fetchSelectedServers());
                if (this.operation.getTopology() != null) {
                    TopologyBean topology = new TopologyBean();
                    topology.setGroups(this.fetchGroups());
                    topology.setId(this.operation.getTopology().getId());
                    bean.setTopology(topology);
                }
                return bean;
            }
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException <ul>
     *         <li>if the loadTopology method is called outside the
     *         INITIALIZE_TOPOLOGY step,
     *         <li>if an error is occurred when parsing the .xml file
     *         </ul>
     */
    public void loadTopology(final File xmlTopoFile) throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();
            this.checkStep(Step.INITIALIZE_TOPOLOGY);

            try {
                this.loadJadortSpecificTopology(xmlTopoFile);
            } catch (UnmarshalException e1) {
                try {
                    this.loadDeployMETopology(xmlTopoFile);
                } catch (UnmarshalException e2) {
                    throw new JadortServiceException(xmlTopoFile.getName()
                        + " is not a valid topology file.\n\nHere are some potentially helpful error messages:\n\t["
                        + JadortServiceImpl.XSD_PATH_JADORT + "] " + e1.getMessage() + "\n\t["
                        + JadortServiceImpl.XSD_PATH_DEPLOYME + "] " + e2.getMessage(), null);
                }
            }

            this.mergeOperation();
        }
    }

    /**
     * @see JadortServiceImpl#loadSchemaAndFile(String, Class, File)
     * @param xmlTopoFile JaDOrT-specific topology XML file to load.
     */
    protected void loadJadortSpecificTopology(final File xmlTopoFile) throws UnmarshalException, JadortServiceException {
        org.ow2.jasmine.jadort.service.topology.xml.Topology xmlTopology = this.loadSchemaAndFile(
            JadortServiceImpl.XSD_PATH_JADORT, org.ow2.jasmine.jadort.service.topology.xml.Topology.class, xmlTopoFile);

        TopologyBean topologyBean = new TopologyBean();
        this.operation.setTopology(topologyBean);

        Map<String, Worker> workerXMLs = new HashMap<String, Worker>();
        for (Worker workerXML : xmlTopology.getWorker()) {
            workerXMLs.put(workerXML.getName(), workerXML);
        }

        List<GroupBean> groups = new ArrayList<GroupBean>();
        for (Group groupXML : xmlTopology.getGroup()) {
            GroupBean groupBean = new GroupBean();
            groupBean.setName(groupXML.getName());
            groupBean.setClustered(groupXML.isClustered());

            List<ServerBean> servers = new ArrayList<ServerBean>();
            List<WorkerBean> workers = new ArrayList<WorkerBean>();
            for (Server serverXML : groupXML.getServer()) {
                ServerBean serverBean = new ServerBean();
                serverBean.setName(serverXML.getName());
                if (serverXML.getCapacity() != null) {
                    serverBean.setCapacity(serverXML.getCapacity());
                }
                serverBean.setType(ServerBean.Type.valueOf(serverXML.getType().name()));

                serverBean.setServerConnector(this.createJMXConnectorBean(serverXML.getURLConnector()));
                if (serverXML.getManager() != null) {
                    serverBean.setManagerConnector(this.createJMXConnectorBean(serverXML.getManager().getURLConnector()));
                }

                Target targetXML = serverXML.getTarget();
                ServerBean target = null;
                if (targetXML != null) {
                    target = new ServerBean();
                    target.setName(targetXML.getName());
                    target.setType(ServerBean.Type.valueOf(targetXML.getType().name()));

                    target.setServerConnector(this.createJMXConnectorBean(targetXML.getURLConnector()));
                    if (targetXML.getManager() != null) {
                        target.setManagerConnector(this.createJMXConnectorBean(targetXML.getManager().getURLConnector()));
                    }

                    serverBean.setTarget(target);
                }

                List<Balancer> balancerXMLs = serverXML.getBalancer();
                if (balancerXMLs != null) {
                    for (Balancer balancerXML : balancerXMLs) {
                        Worker workerXML = workerXMLs.get(balancerXML.getWorker());
                        if (workerXML == null) {
                            throw new JadortServiceException("There is no worker named \"" + balancerXML.getWorker()
                                + "\", please check your topology file.", null);
                        }
                        WorkerBean workerBean = new WorkerBean();
                        workerBean.setName(balancerXML.getServer());
                        workerBean.setWorkerName(workerXML.getName());
                        workerBean.setType(WorkerBean.Type.valueOf(workerXML.getType().name()));
                        workerBean.setServerName(balancerXML.getServer());
                        workerBean.setConnector(this.createJMXConnectorBean(workerXML.getURLConnector()));
                        workerBean.setServer(serverBean);
                        workers.add(workerBean);
                    }
                }

                servers.add(serverBean);
            }
            groupBean.setServers(servers);
            groupBean.setWorkers(workers);

            groups.add(groupBean);
        }
        topologyBean.setGroups(groups);
    }

    /**
     * @see JadortServiceImpl#loadSchemaAndFile(String, Class, File)
     * @param xmlTopoFile JASMINe Deploy ME topology XML file to load.
     */
    protected void loadDeployMETopology(final File xmlTopoFile) throws UnmarshalException, JadortServiceException {
        org.ow2.jasmine.deployme.generated.Topology xmlTopology = this.loadSchemaAndFile(JadortServiceImpl.XSD_PATH_DEPLOYME,
            org.ow2.jasmine.deployme.generated.Topology.class, xmlTopoFile);

        TopologyBean topologyBean = new TopologyBean();
        this.operation.setTopology(topologyBean);

        List<GroupBean> groups = new ArrayList<GroupBean>();
        for (Domains.Domain domain : xmlTopology.getDomains().getDomain()) {
            Configuration conf = domain.getConfiguration();
            String defaultHost = null;
            int defaultPort = 0;
            Protocols protocols = null;

            if (conf != null) {
                if (conf.getGlobalJonas() != null) {
                    defaultHost = conf.getGlobalJonas().getHost();
                }
                protocols = conf.getProtocols();
            }
            if (defaultHost == null) {
                defaultHost = "localhost";
            }
            String defaultProtocol = null;
            if (protocols != null) {
                defaultProtocol = protocols.getProtocolsList();
                BigInteger jrmpPort = protocols.getJrmpPort();
                if (jrmpPort != null) {
                    defaultPort = jrmpPort.intValue();
                }
            }
            if (defaultProtocol != null) {
                // Take the first protocol of the list
                defaultProtocol = defaultProtocol.split(",")[0];
            } else {
                defaultProtocol = "jrmp";
            }

            GroupBean groupBean = new GroupBean();
            groupBean.setName(domain.getName());

            boolean clustered = false;
            try {
                if (conf.getActiveServices().getWeb().isHttpReplicationActivated()) {
                    clustered = true;
                } else {
                    clustered = false;
                }
            } catch (NullPointerException ignored) {
                // If NPE, then there's no HttpReplicationConf
            }

            List<ServerBean> servers = new ArrayList<ServerBean>();
            for (Servers.Server server : domain.getServers().getServer()) {
                conf = server.getConfiguration();
                String host = null;
                if (conf != null) {
                    if (conf.getGlobalJonas() != null) {
                        host = conf.getGlobalJonas().getHost();
                    }
                    protocols = conf.getProtocols();
                }
                if (host == null) {
                    host = defaultHost;
                }
                String name = server.getName();
                int port = defaultPort;
                String protocol = null;
                if (protocols != null) {
                    protocol = protocols.getProtocolsList();
                    BigInteger jrmpPort = protocols.getJrmpPort();
                    if (jrmpPort != null) {
                        port = jrmpPort.intValue();
                    }
                }
                if (protocol != null) {
                    // Take the first protocol of the list
                    protocol = protocol.split(",")[0];
                } else {
                    protocol = defaultProtocol;
                }
                String transport;
                if (protocol.equals("irmi")) {
                    transport = "rmi";
                    if (port == 0) {
                        port = 1098;
                    }
                } else if (protocol.equals("jrmp")) {
                    transport = "rmi";
                    if (port == 0) {
                        port = 1099;
                    }
                } else if (protocol.equals("iiop")) {
                    transport = "iiop";
                    if (port == 0) {
                        port = 2001;
                    }
                } else {
                    throw new JadortServiceException("Unknown protocol \"" + protocol + "\" for server \"" + name + "\".", null);
                }

                String jmxURL = "service:jmx:" + transport + "://" + host + "/jndi/" + transport + "://" + host + ":" + port
                    + "/" + protocol + "connector_" + name;

                ConnectorBean connector = new ConnectorBean();
                connector.setConnectorUrl(jmxURL);
                ServerBean serverBean = new ServerBean();
                serverBean.setName(name);
                serverBean.setServerConnector(connector);
                if (JadortServiceImpl.DUMMY) {
                    serverBean.setType(Type.DUMMY);
                } else {
                    serverBean.setType(Type.JONAS);
                }
                servers.add(serverBean);

                try {
                    if (conf.getActiveServices().getWeb().isHttpReplicationActivated()) {
                        clustered = true;
                    } else {
                        clustered = false;
                    }
                } catch (NullPointerException ignored) {
                    // If NPE, then there's no HttpReplicationConf
                }
            }
            if (domain.getClusterDaemons() != null) {
                for (ClusterDaemons.ClusterDaemon cd : domain.getClusterDaemons().getClusterDaemon()) {
                    int nb = 1;
                    String prefix = cd.getServerNamePrefix();
                    for (String serverName : cd.getServers().getServerName()) {
                        boolean found = false;
                        for (ServerBean s : servers) {
                            if (s.getName().equals(serverName)) {
                                s.setName(prefix + nb);
                                s.getServerConnector().setConnectorUrl(
                                    s.getServerConnector().getConnectorUrl().replace("connector_" + serverName,
                                        "connector_" + s.getName()));

                                nb++;
                                String protocol = cd.getClusterDaemonProtocol();
                                if (protocol != null) {
                                    // Take the first protocol of the list
                                    protocol = protocol.split(",")[0];
                                } else {
                                    protocol = defaultProtocol;
                                }
                                int port = defaultPort;
                                Integer portInteger = cd.getClusterDaemonPort();
                                if (portInteger != null) {
                                    port = portInteger.intValue();
                                }
                                String transport;
                                if (protocol.equals("irmi")) {
                                    transport = "rmi";
                                    if (port == 0) {
                                        port = 1098;
                                    }
                                } else if (protocol.equals("jrmp")) {
                                    transport = "rmi";
                                    if (port == 0) {
                                        port = 1099;
                                    }
                                } else if (protocol.equals("iiop")) {
                                    transport = "iiop";
                                    if (port == 0) {
                                        port = 2001;
                                    }
                                } else {
                                    throw new JadortServiceException("Unknown protocol \"" + protocol
                                        + "\" for cluster daemon \"" + cd.getName() + "\".", null);
                                }

                                String serverURL = s.getServerConnector().getConnectorUrl();
                                int hostStart = serverURL.indexOf("://") + 3;
                                String host = serverURL.substring(hostStart, serverURL.indexOf('/', hostStart));
                                String jmxURL = "service:jmx:" + transport + "://" + host + "/jndi/" + transport + "://" + host
                                    + ":" + port + "/" + protocol + "connector_" + cd.getName();
                                ConnectorBean manager = new ConnectorBean();
                                manager.setConnectorUrl(jmxURL);
                                s.setManagerConnector(manager);

                                found = true;
                                break;
                            }
                        }

                        if (!found) {
                            throw new JadortServiceException("Cannot find server \"" + serverName + "\" for cluster daemon \""
                                + cd.getName() + "\".", null);
                        }
                    }
                }
            }
            groupBean.setServers(servers);
            groupBean.setClustered(clustered);

            groups.add(groupBean);
        }
        topologyBean.setGroups(groups);
    }

    /**
     * Verifies an XML file against an XSD and instantiates it using a given
     * class.
     * 
     * @param xsdPath XSD file path.
     * @param rootClass Root class used for instantiating JAXB.
     * @param xmlFile XML to load.
     * @return XML loaded using JAXB and the rootClass.
     * @throws JadortServiceException If loading fails.
     * @throws UnmarshalException If parsing fails (file doesn't correspond to
     *         XSD).
     */
    @SuppressWarnings("unchecked")
    protected <T> T loadSchemaAndFile(final String xsdPath, final Class<T> rootClass, final File xmlFile)
        throws JadortServiceException, UnmarshalException {
        try {
            JAXBContext jc = JAXBContext.newInstance(rootClass.getPackage().getName());
            Unmarshaller unMarshaller = jc.createUnmarshaller();

            URL xsdURL = this.getClass().getClassLoader().getResource(xsdPath);
            SchemaFactory schemaFactory = SchemaFactory.newInstance("http://www.w3.org/2001/XMLSchema");
            Schema schema = schemaFactory.newSchema(xsdURL);
            unMarshaller.setSchema(schema);

            return (T) unMarshaller.unmarshal(xmlFile);
        } catch (JAXBException e) {
            if (e instanceof UnmarshalException) {
                UnmarshalException ue = (UnmarshalException) e;
                if (ue.getLinkedException() != null) {
                    throw new UnmarshalException(ue.getLinkedException().getMessage(), e);
                } else {
                    throw ue;
                }
            } else {
                throw new JadortServiceException("Error creating the topology parser: " + e.getMessage(), e);
            }
        } catch (SAXException e) {
            throw new JadortServiceException("Error creating the topology parser: " + e.getMessage(), e);
        }
    }

    /**
     * @param urlConnector URLConnector for which to create a JMXConnector bean.
     * @return JMXConnector bean for the given urlConnector.
     */
    protected ConnectorBean createJMXConnectorBean(final URLConnector urlConnector) {
        synchronized (this) {
            ConnectorBean jmxConnectorBean = new ConnectorBean();

            jmxConnectorBean.setConnectorUrl(urlConnector.getURL());
            if (urlConnector.getUsername() != null && urlConnector.getPassword() != null) {
                jmxConnectorBean.setUsername(urlConnector.getUsername());
                jmxConnectorBean.setPassword(urlConnector.getPassword());
            }

            return jmxConnectorBean;
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException if the getGroups() method can not be
     *         called in the current step. (this method can be called only in
     *         SELECT_GROUP step)
     */
    public List<GroupBean> getGroups() throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();
            this.checkStep(Step.SELECT_GROUP, Step.SELECT_APPLICATION);
            return this.fetchGroups();
        }
    }

    /**
     * Returns the list of groups obtained from the loaded topology in the
     * current operation.(returns a copy since
     * operation.getTopology().getGroups() does lazy)
     * 
     * @return The list of groups
     */
    private List<GroupBean> fetchGroups() {
        if (this.operation.getTopology() != null && this.operation.getTopology().getGroups() != null) {
            // Return a copy since operation.getTopology().getGroups() does lazy
            // initialization. Remember to copy all list attributes as well!
            List<GroupBean> result = new ArrayList<GroupBean>(this.operation.getTopology().getGroups().size());
            for (GroupBean groupBean : this.operation.getTopology().getGroups()) {
                GroupBean groupBeanCopy = new GroupBean();
                groupBeanCopy.setApplications(new ArrayList<ApplicationBean>(groupBean.getApplications()));
                groupBeanCopy.setId(groupBean.getId());
                groupBeanCopy.setWorkers(new ArrayList<WorkerBean>(groupBean.getWorkers()));
                groupBeanCopy.setName(groupBean.getName());
                groupBeanCopy.setServers(new ArrayList<ServerBean>(groupBean.getServers()));
                groupBeanCopy.setClustered(groupBean.getClustered());
                result.add(groupBeanCopy);
            }
            return result;
        } else {
            return null;
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException if the selectedGroup method can not be
     *         called in the current step. (this method can be called only in
     *         SELECT_GROUP step)
     */
    public void selectGroup(final GroupBean selectedGroup, final OperationType operationType) throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();
            this.checkStep(Step.SELECT_GROUP);

            for (GroupBean group : this.operation.getTopology().getGroups()) {
                if (group.getId().equals(selectedGroup.getId())) {
                    if (operationType.equals(IJadortService.OperationType.MIGRATE)) {
                        // Copy the oldServerProgressList array to avoid
                        // LazyInitializationException after doing em.remove
                        List<ServerProgressBean> oldServerProgressList = null;
                        if (this.operation.getServerProgressList() != null) {
                            oldServerProgressList = new ArrayList<ServerProgressBean>(this.operation.getServerProgressList());
                        }

                        this.operation.setSelectedGroup(group);
                        this.operation.setType(operationType);

                        List<ServerProgressBean> serverProgressList = new ArrayList<ServerProgressBean>();
                        for (ServerBean server : group.getServers()) {
                            ServerProgressBean serverProgressBean = new ServerProgressBean(server);
                            serverProgressList.add(serverProgressBean);
                        }
                        this.operation.setServerProgressList(serverProgressList);

                        this.mergeOperation();

                        if (oldServerProgressList != null) {
                            for (ServerProgressBean serverProgressBean : oldServerProgressList) {
                                this.em.remove(serverProgressBean);
                            }
                            this.em.flush();
                        }
                    } else if (operationType.equals(IJadortService.OperationType.MAINTAIN)) {
                        // Copy the oldServerProgressList and
                        // oldWorkerProgressList array to avoid
                        // LazyInitializationException after doing em.remove
                        Set<ServerProgressBean> oldServerProgressList = new HashSet<ServerProgressBean>();
                        Set<WorkerProgressBean> oldWorkerProgressList = new HashSet<WorkerProgressBean>();
                        if (this.operation.getServerProgressList() != null) {
                            oldServerProgressList.addAll(this.operation.getServerProgressList());
                        }
                        if (this.operation.getAllServerProgressList() != null) {
                            oldServerProgressList.addAll(this.operation.getAllServerProgressList());
                        }
                        if (this.operation.getWorkerProgressList() != null) {
                            oldWorkerProgressList.addAll(this.operation.getWorkerProgressList());
                        }
                        if (this.operation.getAllWorkerProgressList() != null) {
                            oldWorkerProgressList.addAll(this.operation.getAllWorkerProgressList());
                        }

                        this.operation.setServerProgressList(null);
                        this.operation.setWorkerProgressList(null);
                        this.operation.setAllServerProgressList(null);
                        this.operation.setAllWorkerProgressList(null);
                        this.operation.setSelectedGroup(group);
                        this.operation.setType(operationType);

                        this.mergeOperation();

                        for (ServerProgressBean serverProgressBean : oldServerProgressList) {
                            this.em.remove(serverProgressBean);
                        }
                        for (WorkerProgressBean workerProgressBean : oldWorkerProgressList) {
                            this.em.remove(workerProgressBean);
                        }
                        this.em.flush();
                    } else {
                        throw new IllegalStateException("Unknown OperationType: " + operationType);
                    }

                    return;
                }
            }
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException if the getActiveSessions() method can not
     *         be called in the current step (this method can be called only in
     *         EXECUTING_MIGRATION_PART1, EXECUTING_MIGRATION_PART2,
     *         EXECUTING_MAINTENANCE_CLUSTER and
     *         EXECUTING_MAINTENANCE_NO_CLUSTER) or contacting a server has
     *         failed.
     */
    public int getActiveSessions() throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();
            this.checkStep(Step.EXECUTING_MIGRATION_PART1, Step.EXECUTING_MIGRATION_PART2, Step.EXECUTING_MAINTENANCE_CLUSTER,
                Step.EXECUTING_MAINTENANCE_NO_CLUSTER);
            this.refreshActiveSessions();
            int activeSessions = 0;
            for (ServerProgressBean serverProgressBean : this.operation.getServerProgressList()) {
                activeSessions += serverProgressBean.getServer().getActiveSessions();
            }
            return activeSessions;
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException if the getServerProgressList() method can
     *         not be called in the current step. (this method can be called
     *         only in SELECT_SERVERS, EXECUTING_MIGRATION_PART1,
     *         EXECUTING_MIGRATION_PART2, EXECUTING_MAINTENANCE_CLUSTER,
     *         EXECUTING_MAINTENANCE_NO_CLUSTER and FINISHED steps)
     */
    public List<ServerProgressBean> getServerProgressList() throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();
            this.refreshActiveSessions();
            this.checkStep(Step.SELECT_SERVERS, Step.EXECUTING_MIGRATION_PART1, Step.EXECUTING_MIGRATION_PART2,
                Step.EXECUTING_MAINTENANCE_CLUSTER, Step.EXECUTING_MAINTENANCE_NO_CLUSTER, Step.FINISHED);
            return this.fetchServerProgressList();
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException if the getWorkerProgressList() method can
     *         not be called in the current step. (this method can be called
     *         only in SELECT_SERVERS, EXECUTING_MIGRATION_PART1,
     *         EXECUTING_MIGRATION_PART2, EXECUTING_MAINTENANCE_CLUSTER,
     *         EXECUTING_MAINTENANCE_NO_CLUSTER and FINISHED steps)
     */
    public List<WorkerProgressBean> getWorkerProgressList() throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();
            this.checkStep(Step.SELECT_SERVERS, Step.EXECUTING_MIGRATION_PART1, Step.EXECUTING_MIGRATION_PART2,
                Step.EXECUTING_MAINTENANCE_CLUSTER, Step.EXECUTING_MAINTENANCE_NO_CLUSTER, Step.FINISHED);
            return this.fetchWorkerProgressList();
        }
    }

    /**
     * Refreshes the number of active sessions in the current operation:
     * <ul>
     * <li>If migrating, gets the number of active sessions on the old version.
     * If no old version, sets the number of active sessions to 0.</li>
     * <li>If maintaining, gets the number of active sessions on all
     * applications on the server.</li>
     * </ul>
     * Note that this doesn't merge with the current operation in DB.
     */
    private void refreshActiveSessions() throws JadortServiceException {
        try {
            OperationType type = this.operation.getType();
            if (type == null) {
                // Nothing to do yet...
            } else if (type.equals(OperationType.MIGRATE)) {
                if (this.operation.getServerProgressList() != null) {
                    for (ServerProgressBean serverProgressBean : this.operation.getServerProgressList()) {
                        ServerAction serverAction = ServerAction.getServerAction(serverProgressBean.getServer());
                        String oldApplication = serverProgressBean.getOldApplication();
                        if (oldApplication != null) {
                            try {
                                serverProgressBean.getServer()
                                    .setActiveSessions(serverAction.getActiveSessions(oldApplication));
                            } catch (Exception ignored) {
                                serverProgressBean.getServer().setActiveSessions(0);
                            }
                        } else {
                            serverProgressBean.getServer().setActiveSessions(0);
                        }
                    }
                }
            } else if (type.equals(OperationType.MAINTAIN)) {
                for (ServerBean server : this.operation.getSelectedGroup().getServers()) {
                    ServerAction serverAction = ServerAction.getServerAction(server);
                    try {
                        server.setActiveSessions(serverAction.getActiveSessions());
                    } catch (Exception ignored) {
                        server.setActiveSessions(0);
                    }
                }
            } else {
                throw new IllegalStateException("Unknown operation type: " + type);
            }
        } catch (Exception e) {
            throw new JadortServiceException("Cannot get the number of active sessions: " + e.getMessage(), e);
        }
    }

    /**
     * Refreshes the {@link ServerBean#getMaintained()} of all servers in the
     * current operation. Note that this doesn't merge with the current
     * operation in DB.
     */
    private void refreshMaintainStatus() throws JadortServiceException {
        for (ServerBean server : this.operation.getSelectedGroup().getServers()) {
            boolean maintained = false;
            if (this.operation.getAllServerProgressList() != null) {
                for (ServerProgressBean serverProgress : this.operation.getAllServerProgressList()) {
                    if (serverProgress.getServer().equals(server)) {
                        maintained = serverProgress.getProgress() == 100;
                        break;
                    }
                }
            }
            server.setMaintained(maintained);
        }
    }

    /**
     * Returns a copy of the WorkerProgressBean objects in the current
     * operation.
     * 
     * @throws JadortServiceException If refreshing the active sessions fails.
     * @return The list of ServerProgressBean, null if none present.
     */
    private List<ServerProgressBean> fetchServerProgressList() throws JadortServiceException {
        if (this.operation.getServerProgressList() != null) {
            // Return a copy since operation.getServerProgressList() does lazy
            // initialization
            return new ArrayList<ServerProgressBean>(this.operation.getServerProgressList());
        } else {
            return null;
        }
    }

    /**
     * Returns a copy of the set (i.e. elements are unique) of
     * ServerProgressBean in the current operation.
     * 
     * @throws JadortServiceException If refreshing the active sessions fails.
     * @return The list of ServerProgressBean, null if none present.
     */
    private Set<ServerProgressBean> fetchMigratedServerProgressList() throws JadortServiceException {
        if (this.operation.getAllServerProgressList() != null) {
            // Return a copy since operation.getAllServerProgressList() does
            // lazy initialization
            return new HashSet<ServerProgressBean>(this.operation.getAllServerProgressList());
        } else {
            return null;
        }
    }

    /**
     * Returns a copy of the list of server serverProgressList in the current
     * operation.
     * 
     * @return The list of serverProgressList, null if none present.
     */
    private List<ServerBean> fetchServers() {
        if (this.operation.getSelectedGroup().getServers() != null) {
            // Return a copy since operation.getSelectedGroup().getServers()
            // does lazy initialization
            return new ArrayList<ServerBean>(this.operation.getSelectedGroup().getServers());
        } else {
            return null;
        }
    }

    /**
     * Returns a copy of the list WorkerProgressBean objects in the current
     * operation.
     * 
     * @return The list of serverProgressList
     */
    private List<WorkerProgressBean> fetchWorkerProgressList() throws JadortServiceException {
        if (this.operation.getWorkerProgressList() != null) {
            // Return a copy since operation.getWorkerProgressList() does lazy
            // initialization
            return new ArrayList<WorkerProgressBean>(this.operation.getWorkerProgressList());
        } else {
            return null;
        }
    }

    /**
     * Returns a copy of the set (i.e. elements are unique) of
     * WorkerProgressBean in the current operation.
     * 
     * @return The list of WorkerProgressBean, null if none present.
     */
    private Set<WorkerProgressBean> fetchMigratedWorkerProgressList() {
        if (this.operation.getAllWorkerProgressList() != null) {
            // Return a copy since operation.getServerProgressList() does lazy
            // initialization
            return new HashSet<WorkerProgressBean>(this.operation.getAllWorkerProgressList());
        } else {
            return null;
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException if the getSelectedGroup() method can not
     *         be called in the current step. (this method can be called only in
     *         SELECT_APPLICATION and SELECT_SERVERS steps)
     */
    public GroupBean getSelectedGroup() throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();
            this.checkStep(Step.SELECT_APPLICATION, Step.SELECT_SERVERS);
            return this.fetchSelectedGroup();
        }
    }

    /**
     * Returns a copy of the selected Group in the current operation (Return a
     * copy since GroupBean has been initialized lazily)
     * 
     * @return the selected group
     */
    private GroupBean fetchSelectedGroup() {
        if (this.operation.getSelectedGroup() != null) {
            // Return a copy since GroupBean has been initialized lazily
            GroupBean bean = new GroupBean();
            bean.setApplications(new ArrayList<ApplicationBean>(this.operation.getSelectedGroup().getApplications()));
            bean.setId(this.operation.getSelectedGroup().getId());
            bean.setWorkers(new ArrayList<WorkerBean>(this.operation.getSelectedGroup().getWorkers()));
            bean.setName(this.operation.getSelectedGroup().getName());
            bean.setServers(new ArrayList<ServerBean>(this.operation.getSelectedGroup().getServers()));
            bean.setClustered(this.operation.getSelectedGroup().getClustered());
            return bean;
        } else {
            return null;
        }
    }

    /**
     * Returns a copy of the selected Group in the current operation (Return a
     * copy since GroupBean has been initialized lazily)
     * 
     * @throws JadortServiceException If refreshing the active sessions fails.
     * @return the selected group
     */
    private List<ServerBean> fetchSelectedServers() throws JadortServiceException {
        if (this.operation.getSelectedServers() != null) {
            // Return a copy since operation.getServerProgressList() does lazy
            // initialization
            return new ArrayList<ServerBean>(this.operation.getSelectedServers());
        } else {
            return null;
        }
    }

    /**
     * Returns the serverProgress object of a server
     * 
     * @param server - the server that the method will return his serverProgress
     * @return the serverProgress of the server, null if none found
     */
    private ServerProgressBean getServerProgress(final ServerBean server) {
        for (ServerProgressBean serverProgressBean : this.operation.getServerProgressList()) {
            if (serverProgressBean.getServer().equals(server)) {
                return serverProgressBean;
            }
        }

        return null;
    }

    /**
     * Returns the workerProgress object of a worker
     * 
     * @param worker - the server that the method will return his workerProgress
     * @return the workerProgress of the worker, null if none found
     */
    private WorkerProgressBean getWorkerProgress(final WorkerBean worker) {
        for (WorkerProgressBean workerProgressBean : this.operation.getWorkerProgressList()) {
            if (workerProgressBean.getWorker().getId().equals(worker.getId())) {
                return workerProgressBean;
            }
        }
        return null;
    }

    /**
     * This function allows to execute the next sub step in the migration
     * execution step
     */
    private void nextExecuteMigration() {
        if (this.operation.getType().equals(OperationType.MAINTAIN)) {
            if (this.operation.getSelectedGroup().getClustered()) {
                for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                    serverProgress.setServerProgressState(this.operation.getAimedServerProgressState());
                    serverProgress.setActionState(ActionState.RUNNING);
                }

                for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                    workerProgress.setWorkerProgressState(this.operation.getAimedWorkerProgressState());
                    workerProgress.setActionState(ActionState.RUNNING);
                }

                ServerProgressState aimedServerProgressState = this.operation.getAimedServerProgressState();
                WorkerProgressState aimedWorkerProgressState = this.operation.getAimedWorkerProgressState();

                if (aimedWorkerProgressState.equals(WorkerProgressState.INITIAL)) {
                    this.operation.setAimedWorkerProgressState(WorkerProgressState.STOP_OK);
                    this.operation.setAimedServerProgressProgress(20);
                    this.operation.setAimedWorkerProgressProgress(20);
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        serverProgress.setActionState(ActionState.WAITING);
                        serverProgress.setProgress(20);
                    }
                    this.mergeOperation();
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        this.reachAimedWorkerProgressState(workerProgress);
                    }
                } else if (aimedServerProgressState.equals(ServerProgressState.INITIAL)) {
                    this.operation.setAimedServerProgressState(ServerProgressState.STOP_OK);
                    this.operation.setAimedServerProgressProgress(40);
                    this.operation.setAimedWorkerProgressProgress(40);
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        workerProgress.setActionState(ActionState.WAITING);
                        workerProgress.setProgress(40);
                    }
                    this.mergeOperation();
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        this.reachAimedServerProgressState(serverProgress);
                    }
                } else if (aimedServerProgressState.equals(ServerProgressState.STOP_OK)) {
                    this.operation.setAimedServerProgressState(ServerProgressState.MAINTAIN_OK);
                    this.operation.setAimedServerProgressProgress(60);
                    this.operation.setAimedWorkerProgressProgress(60);
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        workerProgress.setActionState(ActionState.WAITING);
                        workerProgress.setProgress(60);
                    }
                    this.mergeOperation();
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        this.reachAimedServerProgressState(serverProgress);
                    }
                } else if (aimedServerProgressState.equals(ServerProgressState.MAINTAIN_OK)) {
                    this.operation.setAimedServerProgressState(ServerProgressState.START_OK);
                    this.operation.setAimedServerProgressProgress(80);
                    this.operation.setAimedWorkerProgressProgress(80);
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        workerProgress.setActionState(ActionState.WAITING);
                        workerProgress.setProgress(80);
                    }
                    this.mergeOperation();
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        this.reachAimedServerProgressState(serverProgress);
                    }
                } else if (aimedWorkerProgressState.equals(WorkerProgressState.STOP_OK)) {
                    this.operation.setAimedWorkerProgressState(WorkerProgressState.START_OK);
                    this.operation.setAimedServerProgressProgress(100);
                    this.operation.setAimedWorkerProgressProgress(100);
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        serverProgress.setActionState(ActionState.WAITING);
                        serverProgress.setProgress(100);
                    }
                    this.mergeOperation();
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        this.reachAimedWorkerProgressState(workerProgress);
                    }
                } else {
                    throw new IllegalStateException("Unknown WorkerProgressState: " + aimedWorkerProgressState);
                }
            } else {
                for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                    serverProgress.setServerProgressState(this.operation.getAimedServerProgressState());
                    serverProgress.setActionState(ActionState.RUNNING);
                }

                for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                    workerProgress.setWorkerProgressState(this.operation.getAimedWorkerProgressState());
                    workerProgress.setActionState(ActionState.RUNNING);
                }

                ServerProgressState aimedServerProgressState = this.operation.getAimedServerProgressState();
                WorkerProgressState aimedWorkerProgressState = this.operation.getAimedWorkerProgressState();

                if (aimedServerProgressState.equals(ServerProgressState.INITIAL)) {
                    this.operation.setAimedServerProgressState(ServerProgressState.DISABLE_APPLICATIONS_OK);
                    this.operation.setAimedServerProgressProgress(16);
                    this.operation.setAimedWorkerProgressProgress(16);
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        workerProgress.setActionState(ActionState.WAITING);
                        workerProgress.setProgress(16);
                    }
                    this.mergeOperation();
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        this.reachAimedServerProgressState(serverProgress);
                    }
                } else if (aimedWorkerProgressState.equals(WorkerProgressState.INITIAL)) {
                    this.operation.setAimedWorkerProgressState(WorkerProgressState.STOP_OK);
                    this.operation.setAimedServerProgressProgress(33);
                    this.operation.setAimedWorkerProgressProgress(33);
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        serverProgress.setActionState(ActionState.WAITING);
                        serverProgress.setProgress(33);
                    }
                    this.mergeOperation();
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        this.reachAimedWorkerProgressState(workerProgress);
                    }
                } else if (aimedServerProgressState.equals(ServerProgressState.DISABLE_APPLICATIONS_OK)) {
                    this.operation.setAimedServerProgressState(ServerProgressState.STOP_OK);
                    this.operation.setAimedServerProgressProgress(50);
                    this.operation.setAimedWorkerProgressProgress(50);
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        workerProgress.setActionState(ActionState.WAITING);
                        workerProgress.setProgress(50);
                    }
                    this.mergeOperation();
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        this.reachAimedServerProgressState(serverProgress);
                    }
                } else if (aimedServerProgressState.equals(ServerProgressState.STOP_OK)) {
                    this.operation.setAimedServerProgressState(ServerProgressState.MAINTAIN_OK);
                    this.operation.setAimedServerProgressProgress(66);
                    this.operation.setAimedWorkerProgressProgress(66);
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        workerProgress.setActionState(ActionState.WAITING);
                        workerProgress.setProgress(66);
                    }
                    this.mergeOperation();
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        this.reachAimedServerProgressState(serverProgress);
                    }
                } else if (aimedServerProgressState.equals(ServerProgressState.MAINTAIN_OK)) {
                    this.operation.setAimedServerProgressState(ServerProgressState.START_OK);
                    this.operation.setAimedServerProgressProgress(83);
                    this.operation.setAimedWorkerProgressProgress(83);
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        workerProgress.setActionState(ActionState.WAITING);
                        workerProgress.setProgress(83);
                    }
                    this.mergeOperation();
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        this.reachAimedServerProgressState(serverProgress);
                    }
                } else if (aimedWorkerProgressState.equals(WorkerProgressState.STOP_OK)) {
                    this.operation.setAimedWorkerProgressState(WorkerProgressState.START_OK);
                    this.operation.setAimedServerProgressProgress(100);
                    this.operation.setAimedWorkerProgressProgress(100);
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        serverProgress.setActionState(ActionState.WAITING);
                        serverProgress.setProgress(100);
                    }
                    this.mergeOperation();
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        this.reachAimedWorkerProgressState(workerProgress);
                    }
                } else {
                    throw new IllegalStateException("Unknown WorkerProgressState: " + aimedWorkerProgressState);
                }
            }
        } else {
            for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                serverProgress.setServerProgressState(this.operation.getAimedServerProgressState());
                serverProgress.setActionState(ActionState.RUNNING);
            }

            ServerProgressState aimedServerProgressState = this.operation.getAimedServerProgressState();
            if (aimedServerProgressState.equals(ServerProgressState.INITIAL)) {
                this.operation.setAimedServerProgressState(ServerProgressState.UPLOAD_OK);
                this.operation.setAimedServerProgressProgress(33);
            } else if (aimedServerProgressState.equals(ServerProgressState.UPLOAD_OK)) {
                this.operation.setAimedServerProgressState(ServerProgressState.DEPLOY_OK);
                this.operation.setAimedServerProgressProgress(66);
            } else if (aimedServerProgressState.equals(ServerProgressState.DEPLOY_OK)) {
                this.operation.setAimedServerProgressState(ServerProgressState.SET_DEFAULT_OK);
                this.operation.setAimedServerProgressProgress(100);
            } else if (aimedServerProgressState.equals(ServerProgressState.SET_DEFAULT_OK)) {
                this.operation.setAimedServerProgressState(ServerProgressState.UNDEPLOY_OK);
                this.operation.setAimedServerProgressProgress(50);
            } else if (aimedServerProgressState.equals(ServerProgressState.UNDEPLOY_OK)) {
                this.operation.setAimedServerProgressState(ServerProgressState.ERASE_OK);
                this.operation.setAimedServerProgressProgress(100);
            } else if (aimedServerProgressState.equals(ServerProgressState.UNDEPLOY_OK)) {
                this.operation.setAimedServerProgressState(ServerProgressState.ERASE_OK);
                this.operation.setAimedServerProgressProgress(100);
            } else {
                throw new IllegalStateException("Unknown ServerProgressState: " + aimedServerProgressState);
            }
            this.mergeOperation();

            for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                this.reachAimedServerProgressState(serverProgress);
            }
        }
    }

    /**
     * this function allows to execute the previous sub step in the migration
     * execution step
     */
    private void previousExecuteMigration() {
        if (this.operation.getType().equals(OperationType.MAINTAIN)) {
            if (this.operation.getSelectedGroup().getClustered()) {
                for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                    serverProgress.setServerProgressState(this.operation.getAimedServerProgressState());
                    serverProgress.setActionState(ActionState.RUNNING);
                }

                for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                    workerProgress.setWorkerProgressState(this.operation.getAimedWorkerProgressState());
                    workerProgress.setActionState(ActionState.RUNNING);
                }

                ServerProgressState aimedServerProgressState = this.operation.getAimedServerProgressState();
                WorkerProgressState aimedWorkerProgressState = this.operation.getAimedWorkerProgressState();

                if (aimedWorkerProgressState.equals(WorkerProgressState.START_OK)) {
                    this.operation.setAimedWorkerProgressState(WorkerProgressState.STOP_OK);
                    this.operation.setAimedServerProgressProgress(80);
                    this.operation.setAimedWorkerProgressProgress(80);
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        serverProgress.setActionState(ActionState.WAITING);
                        serverProgress.setProgress(80);
                    }
                    this.mergeOperation();
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        this.reachAimedWorkerProgressState(workerProgress);
                    }
                } else if (aimedServerProgressState.equals(ServerProgressState.START_OK)) {
                    this.operation.setAimedServerProgressState(ServerProgressState.MAINTAIN_OK);
                    this.operation.setAimedServerProgressProgress(60);
                    this.operation.setAimedWorkerProgressProgress(60);
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        workerProgress.setActionState(ActionState.WAITING);
                        workerProgress.setProgress(60);
                    }
                    this.mergeOperation();
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        this.reachAimedServerProgressState(serverProgress);
                    }
                } else if (aimedServerProgressState.equals(ServerProgressState.MAINTAIN_OK)) {
                    this.operation.setAimedServerProgressState(ServerProgressState.STOP_OK);
                    this.operation.setAimedServerProgressProgress(40);
                    this.operation.setAimedWorkerProgressProgress(40);
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        workerProgress.setActionState(ActionState.WAITING);
                        workerProgress.setProgress(40);
                    }
                    this.mergeOperation();
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        this.reachAimedServerProgressState(serverProgress);
                    }
                } else if (aimedServerProgressState.equals(ServerProgressState.STOP_OK)) {
                    this.operation.setAimedServerProgressState(ServerProgressState.INITIAL);
                    this.operation.setAimedServerProgressProgress(20);
                    this.operation.setAimedWorkerProgressProgress(20);
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        workerProgress.setActionState(ActionState.WAITING);
                        workerProgress.setProgress(20);
                    }
                    this.mergeOperation();
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        this.reachAimedServerProgressState(serverProgress);
                    }
                } else if (aimedWorkerProgressState.equals(WorkerProgressState.STOP_OK)) {
                    this.operation.setAimedWorkerProgressState(WorkerProgressState.INITIAL);
                    this.operation.setAimedServerProgressProgress(0);
                    this.operation.setAimedWorkerProgressProgress(0);
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        serverProgress.setActionState(ActionState.WAITING);
                        serverProgress.setProgress(0);
                    }
                    this.mergeOperation();
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        this.reachAimedWorkerProgressState(workerProgress);
                    }
                } else {
                    throw new IllegalStateException("Unknown WorkerProgressState: " + aimedWorkerProgressState);
                }
            } else {
                for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                    serverProgress.setServerProgressState(this.operation.getAimedServerProgressState());
                    serverProgress.setActionState(ActionState.RUNNING);
                }

                for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                    workerProgress.setWorkerProgressState(this.operation.getAimedWorkerProgressState());
                    workerProgress.setActionState(ActionState.RUNNING);
                }

                ServerProgressState aimedServerProgressState = this.operation.getAimedServerProgressState();
                WorkerProgressState aimedWorkerProgressState = this.operation.getAimedWorkerProgressState();

                if (aimedWorkerProgressState.equals(WorkerProgressState.START_OK)) {
                    this.operation.setAimedWorkerProgressState(WorkerProgressState.STOP_OK);
                    this.operation.setAimedServerProgressProgress(83);
                    this.operation.setAimedWorkerProgressProgress(83);
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        serverProgress.setActionState(ActionState.WAITING);
                        serverProgress.setProgress(83);
                    }
                    this.mergeOperation();
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        this.reachAimedWorkerProgressState(workerProgress);
                    }
                } else if (aimedServerProgressState.equals(ServerProgressState.START_OK)) {
                    this.operation.setAimedServerProgressState(ServerProgressState.MAINTAIN_OK);
                    this.operation.setAimedServerProgressProgress(66);
                    this.operation.setAimedWorkerProgressProgress(66);
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        workerProgress.setActionState(ActionState.WAITING);
                        workerProgress.setProgress(66);
                    }
                    this.mergeOperation();
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        this.reachAimedServerProgressState(serverProgress);
                    }
                } else if (aimedServerProgressState.equals(ServerProgressState.MAINTAIN_OK)) {
                    this.operation.setAimedServerProgressState(ServerProgressState.STOP_OK);
                    this.operation.setAimedServerProgressProgress(50);
                    this.operation.setAimedWorkerProgressProgress(50);
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        workerProgress.setActionState(ActionState.WAITING);
                        workerProgress.setProgress(50);
                    }
                    this.mergeOperation();
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        this.reachAimedServerProgressState(serverProgress);
                    }
                } else if (aimedServerProgressState.equals(ServerProgressState.STOP_OK)) {
                    this.operation.setAimedServerProgressState(ServerProgressState.DISABLE_APPLICATIONS_OK);
                    this.operation.setAimedServerProgressProgress(33);
                    this.operation.setAimedWorkerProgressProgress(33);
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        workerProgress.setActionState(ActionState.WAITING);
                        workerProgress.setProgress(33);
                    }
                    this.mergeOperation();
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        this.reachAimedServerProgressState(serverProgress);
                    }
                } else if (aimedWorkerProgressState.equals(WorkerProgressState.STOP_OK)) {
                    this.operation.setAimedWorkerProgressState(WorkerProgressState.INITIAL);
                    this.operation.setAimedServerProgressProgress(16);
                    this.operation.setAimedWorkerProgressProgress(16);
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        serverProgress.setActionState(ActionState.WAITING);
                        serverProgress.setProgress(16);
                    }
                    this.mergeOperation();
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        this.reachAimedWorkerProgressState(workerProgress);
                    }
                } else if (aimedServerProgressState.equals(ServerProgressState.DISABLE_APPLICATIONS_OK)) {
                    this.operation.setAimedServerProgressState(ServerProgressState.INITIAL);
                    this.operation.setAimedServerProgressProgress(0);
                    this.operation.setAimedWorkerProgressProgress(0);
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        workerProgress.setActionState(ActionState.WAITING);
                        workerProgress.setProgress(0);
                    }
                    this.mergeOperation();
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        this.reachAimedServerProgressState(serverProgress);
                    }
                } else {
                    throw new IllegalStateException("Unknown WorkerProgressState: " + aimedWorkerProgressState);
                }

            }
        } else {
            for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                serverProgress.setServerProgressState(this.operation.getAimedServerProgressState());
                serverProgress.setActionState(ActionState.RUNNING);
            }

            ServerProgressState aimedServerProgressState = this.operation.getAimedServerProgressState();
            if (aimedServerProgressState.equals(ServerProgressState.INITIAL)) {
                // Nothing to do
            } else if (aimedServerProgressState.equals(ServerProgressState.UPLOAD_OK)) {
                this.operation.setAimedServerProgressState(ServerProgressState.INITIAL);
                this.operation.setAimedServerProgressProgress(0);
            } else if (aimedServerProgressState.equals(ServerProgressState.DEPLOY_OK)) {
                this.operation.setAimedServerProgressState(ServerProgressState.UPLOAD_OK);
                this.operation.setAimedServerProgressProgress(33);
            } else if (aimedServerProgressState.equals(ServerProgressState.SET_DEFAULT_OK)) {
                this.operation.setAimedServerProgressState(ServerProgressState.DEPLOY_OK);
                this.operation.setAimedServerProgressProgress(66);
            } else if (aimedServerProgressState.equals(ServerProgressState.UNDEPLOY_OK)) {
                this.operation.setAimedServerProgressState(ServerProgressState.SET_DEFAULT_OK);
                this.operation.setAimedServerProgressProgress(100);
            } else {
                throw new IllegalStateException("Unknown ServerProgressState: " + aimedServerProgressState);
            }

            for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                this.reachAimedServerProgressState(serverProgress);
            }
            this.mergeOperation();
        }
    }

    /**
     * this function allows the serverProgress to executes the proper task to
     * attend the Aimed ServerProgressState this function executes the
     * correspondent task to go from the current serverProgress state to the
     * Aimed ServerProgressList State.
     * 
     * @param serverProgress - the serverProgress that will execute the task.
     */
    private void reachAimedServerProgressState(final ServerProgressBean serverProgress) {
        ServerProgressState currentServerProgressState = serverProgress.getServerProgressState();
        ServerProgressState aimedServerProgressState = this.operation.getAimedServerProgressState();

        try {
            OperationType type = this.operation.getType();
            if (type.equals(OperationType.MAINTAIN)) {
                if (this.operation.getSelectedGroup().getClustered()) {
                    if (currentServerProgressState.equals(ServerProgressState.INITIAL)) {
                        if (aimedServerProgressState.equals(ServerProgressState.STOP_OK)) {
                            // next
                            this.stopServer(serverProgress);
                            return;
                        }
                        // no previous
                    } else if (currentServerProgressState.equals(ServerProgressState.STOP_OK)) {
                        if (aimedServerProgressState.equals(ServerProgressState.MAINTAIN_OK)) {
                            // next
                            this.maintainServer(serverProgress);
                            return;
                        } else if (aimedServerProgressState.equals(ServerProgressState.INITIAL)) {
                            // previous
                            this.startServer(serverProgress);
                            return;
                        }
                    } else if (currentServerProgressState.equals(ServerProgressState.MAINTAIN_OK)) {
                        if (aimedServerProgressState.equals(ServerProgressState.START_OK)) {
                            // next
                            this.startServer(serverProgress);
                            return;
                        } else if (aimedServerProgressState.equals(ServerProgressState.STOP_OK)) {
                            // previous
                            this.maintainServer(serverProgress);
                            return;
                        }
                    } else if (currentServerProgressState.equals(ServerProgressState.START_OK)) {
                        if (aimedServerProgressState.equals(ServerProgressState.MAINTAIN_OK)) {
                            // previous
                            this.stopServer(serverProgress);
                            return;
                        }
                    } else {
                        throw new IllegalStateException("Unknown ServerProgressState: " + aimedServerProgressState);
                    }
                } else {
                    if (currentServerProgressState.equals(ServerProgressState.INITIAL)) {
                        if (aimedServerProgressState.equals(ServerProgressState.DISABLE_APPLICATIONS_OK)) {
                            // next
                            this.disableApplications(serverProgress);
                            return;
                        }
                        // no previous
                    } else if (currentServerProgressState.equals(ServerProgressState.DISABLE_APPLICATIONS_OK)) {
                        if (aimedServerProgressState.equals(ServerProgressState.STOP_OK)) {
                            // next
                            this.stopServer(serverProgress);
                            return;
                        } else if (aimedServerProgressState.equals(ServerProgressState.INITIAL)) {
                            // previous
                            this.enableApplications(serverProgress);
                            return;
                        }
                    } else if (currentServerProgressState.equals(ServerProgressState.STOP_OK)) {
                        if (aimedServerProgressState.equals(ServerProgressState.MAINTAIN_OK)) {
                            // next
                            this.maintainServer(serverProgress);
                            return;
                        } else if (aimedServerProgressState.equals(ServerProgressState.DISABLE_APPLICATIONS_OK)) {
                            // previous
                            this.startServer(serverProgress);
                            return;
                        }
                    } else if (currentServerProgressState.equals(ServerProgressState.MAINTAIN_OK)) {
                        if (aimedServerProgressState.equals(ServerProgressState.START_OK)) {
                            // next
                            this.startServer(serverProgress);
                            return;
                        } else if (aimedServerProgressState.equals(ServerProgressState.STOP_OK)) {
                            // previous
                            this.maintainServer(serverProgress);
                            return;
                        }
                    } else if (currentServerProgressState.equals(ServerProgressState.START_OK)) {
                        if (aimedServerProgressState.equals(ServerProgressState.MAINTAIN_OK)) {
                            // previous
                            this.stopServer(serverProgress);
                            return;
                        }
                    } else {
                        throw new IllegalStateException("Unknown ServerProgressState: " + aimedServerProgressState);
                    }
                }
            } else if (type.equals(OperationType.MIGRATE)) {
                if (currentServerProgressState.equals(ServerProgressState.INITIAL)) {
                    if (aimedServerProgressState.equals(ServerProgressState.UPLOAD_OK)) {
                        // next
                        this.upload(serverProgress, this.operation.getApplication());
                        return;
                    }
                    // no previous
                } else if (currentServerProgressState.equals(ServerProgressState.UPLOAD_OK)) {
                    if (aimedServerProgressState.equals(ServerProgressState.DEPLOY_OK)) {
                        // next
                        this.deploy(serverProgress, serverProgress.getNewApplication());
                        return;
                    } else if (aimedServerProgressState.equals(ServerProgressState.INITIAL)) {
                        // previous
                        this.erase(serverProgress, serverProgress.getNewApplication());
                        return;
                    }
                } else if (currentServerProgressState.equals(ServerProgressState.DEPLOY_OK)) {
                    if (aimedServerProgressState.equals(ServerProgressState.SET_DEFAULT_OK)) {
                        // next
                        this.setAsDefault(serverProgress, serverProgress.getNewApplication());
                        return;
                    } else if (aimedServerProgressState.equals(ServerProgressState.UPLOAD_OK)) {
                        // previous
                        this.undeploy(serverProgress, serverProgress.getNewApplication());
                        return;
                    }
                } else if (currentServerProgressState.equals(ServerProgressState.SET_DEFAULT_OK)) {
                    if (aimedServerProgressState.equals(ServerProgressState.UNDEPLOY_OK)) {
                        // next
                        if (!serverProgress.hasOldVersion()) {
                            throw new Exception("There's no old version or the old and new versions are the same");
                        } else {
                            this.undeploy(serverProgress, serverProgress.getOldApplication());
                        }
                        return;
                    } else if (aimedServerProgressState.equals(ServerProgressState.DEPLOY_OK)) {
                        // previous
                        if (serverProgress.getOldApplication() == null) {
                            serverProgress.appendToLog("There's no old version, ignoring step.\n"
                                + "This implies that the new [and only] version will keep on being the default one.");
                            serverProgress.setActionState(ActionState.WAITING);
                            serverProgress.setServerProgressState(aimedServerProgressState);
                            serverProgress.setProgress(this.operation.getAimedServerProgressProgress());
                        } else {
                            this.setAsDefault(serverProgress, serverProgress.getOldApplication());
                        }
                        return;
                    }
                } else if (currentServerProgressState.equals(ServerProgressState.UNDEPLOY_OK)) {
                    if (aimedServerProgressState.equals(ServerProgressState.ERASE_OK)) {
                        // next
                        if (!serverProgress.hasOldVersion()) {
                            throw new Exception("There's no old version or the old and new versions are the same");
                        } else {
                            this.erase(serverProgress, serverProgress.getOldApplication());
                        }
                        return;
                    } else if (aimedServerProgressState.equals(ServerProgressState.SET_DEFAULT_OK)) {
                        // previous
                        this.deploy(serverProgress, serverProgress.getOldApplication());
                        return;
                    }
                } else {
                    throw new IllegalStateException("Unknown ServerProgressState: " + aimedServerProgressState);
                }
            } else {
                throw new IllegalStateException("Unknown OperationType: " + type);
            }
        } catch (Exception e) {
            serverProgress.appendToLog("Failed executing server action: " + e.getClass().getCanonicalName() + ", "
                + e.getMessage());
            serverProgress.setActionState(ActionState.FINISHED_ERROR);
            return;
        }

        serverProgress.appendToLog("Don't know what to do: operation.getAimedServerProgressState() is "
            + aimedServerProgressState + ", serverProgress.getServerProgressState() is " + currentServerProgressState);
        serverProgress.setActionState(ActionState.FINISHED_ERROR);

        return;
    }

    /**
     * this function allows the workerProgress to executes the proper task to
     * attend the Aimed WorkerProgress this function executes the correspondent
     * task to go from the current serverProgress state to the Aimed
     * WorkerProgress State.
     * 
     * @param workerProgress - the workerProgress that will execute the task.
     */
    private void reachAimedWorkerProgressState(final WorkerProgressBean workerProgress) {
        WorkerProgressState currentWorkerProgressState = workerProgress.getWorkerProgressState();
        WorkerProgressState aimedWorkerProgressState = this.operation.getAimedWorkerProgressState();

        try {
            OperationType type = this.operation.getType();
            if (type.equals(OperationType.MAINTAIN)) {
                if (currentWorkerProgressState.equals(WorkerProgressState.INITIAL)) {
                    if (aimedWorkerProgressState.equals(WorkerProgressState.STOP_OK)) {
                        // next
                        this.stopWorker(workerProgress);
                        return;
                    }
                    // no previous
                } else if (currentWorkerProgressState.equals(WorkerProgressState.STOP_OK)) {
                    if (aimedWorkerProgressState.equals(WorkerProgressState.START_OK)) {
                        // next
                        this.startWorker(workerProgress);
                        return;
                    } else if (aimedWorkerProgressState.equals(WorkerProgressState.INITIAL)) {
                        // previous
                        this.startWorker(workerProgress);
                        return;
                    }
                } else if (currentWorkerProgressState.equals(WorkerProgressState.START_OK)) {
                    if (aimedWorkerProgressState.equals(WorkerProgressState.STOP_OK)) {
                        // previous
                        this.stopWorker(workerProgress);
                        return;
                    }
                } else {
                    throw new IllegalStateException("Unknown WorkerProgressState: " + currentWorkerProgressState);
                }
            } else {
                throw new IllegalStateException("Unknown OperationType: " + type);
            }
        } catch (Exception e) {
            workerProgress.appendToLog("Failed executing server action: " + e.getClass().getCanonicalName() + ", "
                + e.getMessage());
            workerProgress.setActionState(ActionState.FINISHED_ERROR);
            return;
        }

        workerProgress.appendToLog("Don't know what to do: operation.getAimedWorkerProgressState() is "
            + aimedWorkerProgressState + ", workerProgress.getAimedWorkerProgressState() is " + currentWorkerProgressState);
        workerProgress.setActionState(ActionState.FINISHED_ERROR);

        return;
    }

    /**
     * this function allows to upload an application on a server
     * 
     * @param serverProgress - the serverProgress server on which the
     *        application will be uploaded.
     * @param application - the application to be uploaded.
     * @throws JMSException
     */
    private void upload(final ServerProgressBean serverProgress, final ApplicationBean application) throws JMSException {
        Message message = this.session.createMessage();
        message.setStringProperty(JadortServiceImpl.OPERATION_TYPE, OperationType.MIGRATE.toString());
        message.setStringProperty(JadortServiceImpl.ACTION_TYPE, ActionType.UPLOAD.toString());
        message.setIntProperty(JadortServiceImpl.SERVER_PROGRESS_ID, serverProgress.getId());
        message.setIntProperty(JadortServiceImpl.APPLICATION_ID, application.getId());
        QueueSender sender = this.session.createSender(this.queue);
        sender.send(message);
    }

    /**
     * this function allow to deploy an application on the server
     * 
     * @param serverProgress - the serverProgress server on which the
     *        application will be deployed
     * @param applicationName - the application name to be deployed
     */
    private void deploy(final ServerProgressBean serverProgress, final String applicationName) throws JMSException,
        JadortServiceException {
        if (applicationName == null) {
            serverProgress.setServerProgressState(this.operation.getAimedServerProgressState());
            serverProgress.setProgress(this.operation.getAimedServerProgressProgress());
            serverProgress.setActionState(ActionState.FINISHED_ERROR);
            serverProgress.appendToLog("The application to deploy is null !");
        } else {
            Message message = this.session.createMessage();
            message.setStringProperty(JadortServiceImpl.OPERATION_TYPE, OperationType.MIGRATE.toString());
            message.setStringProperty(JadortServiceImpl.ACTION_TYPE, ActionType.DEPLOY.toString());
            message.setIntProperty(JadortServiceImpl.SERVER_PROGRESS_ID, serverProgress.getId());
            message.setStringProperty(JadortServiceImpl.APPLICATION_NAME, applicationName);
            QueueSender sender = this.session.createSender(this.queue);
            sender.send(message);
        }
    }

    /**
     * This function allows to activate an application on the the server. it
     * makes the application policy to default.
     * 
     * @param serverProgress - the serverProgress server on which the new
     *        application will be activated.
     * @param applicationName - the application name
     */
    private void setAsDefault(final ServerProgressBean serverProgress, final String applicationName) throws JMSException,
        JadortServiceException {
        if (applicationName == null) {
            serverProgress.setServerProgressState(this.operation.getAimedServerProgressState());
            serverProgress.setProgress(this.operation.getAimedServerProgressProgress());
            serverProgress.setActionState(ActionState.FINISHED_ERROR);
            serverProgress.appendToLog("The application to set as default is null !");
        } else {
            Message message = this.session.createMessage();
            message.setStringProperty(JadortServiceImpl.OPERATION_TYPE, OperationType.MIGRATE.toString());
            message.setStringProperty(JadortServiceImpl.ACTION_TYPE, ActionType.SET_DEFAULT.toString());
            message.setIntProperty(JadortServiceImpl.SERVER_PROGRESS_ID, serverProgress.getId());
            message.setStringProperty(JadortServiceImpl.APPLICATION_NAME, applicationName);
            QueueSender sender = this.session.createSender(this.queue);
            sender.send(message);
        }
    }

    /**
     * this function allows to undeploy an application on the server
     * 
     * @param serverProgress - the serverProgress server on which the
     *        application will be undeployed
     * @param applicationName - the application name to undeploy
     */
    private void undeploy(final ServerProgressBean serverProgress, final String applicationName) throws JMSException,
        JadortServiceException {
        if (applicationName == null) {
            serverProgress.setServerProgressState(this.operation.getAimedServerProgressState());
            serverProgress.setProgress(this.operation.getAimedServerProgressProgress());
            serverProgress.setActionState(ActionState.FINISHED_ERROR);
            serverProgress.appendToLog("The application to undeploy is null !");
        } else {
            Message message = this.session.createMessage();
            message.setStringProperty(JadortServiceImpl.OPERATION_TYPE, OperationType.MIGRATE.toString());
            message.setStringProperty(JadortServiceImpl.ACTION_TYPE, ActionType.UNDEPLOY.toString());
            message.setIntProperty(JadortServiceImpl.SERVER_PROGRESS_ID, serverProgress.getId());
            message.setStringProperty(JadortServiceImpl.APPLICATION_NAME, applicationName);
            QueueSender sender = this.session.createSender(this.queue);
            sender.send(message);
        }
    }

    /**
     * this function allow to erase an application on the server
     * 
     * @param serverProgress - the serverProgress server on which the new
     *        application will be erased
     * @param applicationName - the application name to erase
     */
    private void erase(final ServerProgressBean serverProgress, final String applicationName) throws JMSException,
        JadortServiceException {
        if (applicationName == null) {
            serverProgress.setServerProgressState(this.operation.getAimedServerProgressState());
            serverProgress.setProgress(this.operation.getAimedServerProgressProgress());
            serverProgress.setActionState(ActionState.FINISHED_ERROR);
            serverProgress.appendToLog("The application to erase is null !");
        } else {
            Message message = this.session.createMessage();
            message.setStringProperty(JadortServiceImpl.OPERATION_TYPE, OperationType.MIGRATE.toString());
            message.setStringProperty(JadortServiceImpl.ACTION_TYPE, ActionType.ERASE.toString());
            message.setIntProperty(JadortServiceImpl.SERVER_PROGRESS_ID, serverProgress.getId());
            message.setStringProperty(JadortServiceImpl.APPLICATION_NAME, applicationName);
            QueueSender sender = this.session.createSender(this.queue);
            sender.send(message);
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException <ul>
     *         <li>if the restartServermethod can not be called in the current
     *         step.
     *         <li>if the server state is not in error state. (this method can
     *         be called only in Step.EXECUTING_MIGRATION_PART1,
     *         EXECUTING_MIGRATION_PART2, EXECUTING_MAINTENANCE_CLUSTER and
     *         EXECUTING_MAINTENANCE_NO_CLUSTER steps)
     *         </ul>
     */
    public void restartServer(final ServerBean server) throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();
            this.checkStep(Step.EXECUTING_MIGRATION_PART1, Step.EXECUTING_MIGRATION_PART2, Step.EXECUTING_MAINTENANCE_CLUSTER,
                Step.EXECUTING_MAINTENANCE_NO_CLUSTER);

            ServerProgressBean serverProgress = this.getServerProgress(server);
            if (!serverProgress.getActionState().equals(ActionState.FINISHED_ERROR)) {
                throw new JadortServiceException("Server state for server '" + server.getName() + "' is "
                    + serverProgress.getActionState() + ", it should be " + ActionState.FINISHED_ERROR, null);
            }
            serverProgress.setActionState(ActionState.RUNNING);

            this.mergeOperation();

            this.reachAimedServerProgressState(this.getServerProgress(server));
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException <ul>
     *         <li>if the ignoreServer can not be called in the current step.
     *         <li>if the server state is not in error state. (this method can
     *         be called only in Step.EXECUTING_MIGRATION_PART1,
     *         EXECUTING_MIGRATION_PART2 EXECUTING_MAINTENANCE_CLUSTER and
     *         EXECUTING_MAINTENANCE_NO_CLUSTER steps)
     *         </ul>
     */
    public void ignoreServer(final ServerBean server) throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();
            this.checkStep(Step.EXECUTING_MIGRATION_PART1, Step.EXECUTING_MIGRATION_PART2, Step.EXECUTING_MAINTENANCE_CLUSTER,
                Step.EXECUTING_MAINTENANCE_NO_CLUSTER);

            ServerProgressBean serverProgress = this.getServerProgress(server);
            if (!serverProgress.getActionState().equals(ActionState.FINISHED_ERROR)) {
                throw new JadortServiceException("Server state for server '" + server.getName() + "' is "
                    + serverProgress.getActionState() + ", it should be " + ActionState.FINISHED_ERROR, null);
            }

            serverProgress.appendToLog("Error on the server has been ignored by the user");

            serverProgress.setActionState(ActionState.WAITING);
            serverProgress.setServerProgressState(this.operation.getAimedServerProgressState());

            this.mergeOperation();
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException <ul>
     *         <li>if the abortServer can not be called in the current step.
     *         <li>if the server state is not in running state. (this method can
     *         be called only in Step.EXECUTING_MIGRATION_PART1,
     *         Step.EXECUTING_MIGRATION_PART2, EXECUTING_MAINTENANCE_CLUSTER,
     *         Step.EXECUTING_MAINTENANCE_NO_CLUSTER steps)
     *         </ul>
     */
    public void abortServer(final ServerBean server) throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();
            this.checkStep(Step.EXECUTING_MIGRATION_PART1, Step.EXECUTING_MIGRATION_PART2, Step.EXECUTING_MAINTENANCE_CLUSTER,
                Step.EXECUTING_MAINTENANCE_NO_CLUSTER);

            ServerProgressBean serverProgress = this.getServerProgress(server);
            if (!serverProgress.getActionState().equals(ActionState.RUNNING)) {
                throw new JadortServiceException("Server state for server '" + server.getName() + "' is "
                    + serverProgress.getActionState() + ", it should be " + ActionState.RUNNING, null);
            }

            serverProgress.appendToLog("Action has been aborted by the user");

            serverProgress.setActionState(ActionState.FINISHED_ERROR);
            serverProgress.setProgress(this.operation.getAimedServerProgressProgress());

            this.mergeOperation();
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException if the checkServer can not be called in
     *         the current step. (this method can be called only in
     *         EXECUTING_MIGRATION_PART1, EXECUTING_MIGRATION_PART2,
     *         EXECUTING_MAINTENANCE_CLUSTER and
     *         EXECUTING_MAINTENANCE_NO_CLUSTER steps)
     */
    public boolean checkServer(final ServerBean server) throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();
            this.checkStep(Step.EXECUTING_MIGRATION_PART1, Step.EXECUTING_MIGRATION_PART2, Step.EXECUTING_MAINTENANCE_CLUSTER,
                Step.EXECUTING_MAINTENANCE_NO_CLUSTER);

            try {
                boolean result = true;
                ServerProgressBean serverProgress = this.getServerProgress(server);
                ServerAction serverAction = ServerAction.getServerAction(serverProgress.getServer());
                List<ApplicationBean> applications = serverAction.listOfApplications();

                ServerProgressState serverProgressState = serverProgress.getServerProgressState();
                if (serverProgressState.equals(ServerProgressState.INITIAL)) {
                    OperationType type = this.operation.getType();
                    if (type.equals(OperationType.MAINTAIN)) {
                        result = serverAction.isStarted();
                    } else if (type.equals(OperationType.MIGRATE)) {
                        for (ApplicationBean application : applications) {
                            if (this.isSameApplication(this.operation.getApplication(), application)) {
                                if (!this.checkApplication(serverProgress, application, null, null)) {
                                    result = false;
                                    break;
                                }
                            }
                            if (this.isSameApplication(serverAction.getApplicationBean(serverProgress.getNewApplication()),
                                application)) {
                                if (!this.checkApplication(serverProgress, application, null, null)) {
                                    result = false;
                                    break;
                                }
                            }
                            if (this.isSameApplication(serverAction.getApplicationBean(serverProgress.getOldApplication()),
                                application)) {
                                if (!this.checkApplication(serverProgress, application, ServerAction.STATE_DEPLOYED, "Default")) {
                                    result = false;
                                    break;
                                }
                            }
                        }
                    } else {
                        throw new IllegalStateException("Unknown OperationType: " + type);
                    }
                } else if (serverProgressState.equals(ServerProgressState.UPLOAD_OK)) {
                    for (ApplicationBean application : applications) {
                        if (this.isSameApplication(this.operation.getApplication(), application)) {
                            if (!this.checkApplication(serverProgress, application, ServerAction.STATE_PRESENT, null)) {
                                result = false;
                                break;
                            }
                        }
                        if (this.isSameApplication(serverAction.getApplicationBean(serverProgress.getNewApplication()),
                            application)) {
                            if (!this.checkApplication(serverProgress, application, ServerAction.STATE_PRESENT, null)) {
                                result = false;
                                break;
                            }
                        }
                        if (this.isSameApplication(serverAction.getApplicationBean(serverProgress.getOldApplication()),
                            application)) {
                            if (!this.checkApplication(serverProgress, application, ServerAction.STATE_DEPLOYED, "Default")) {
                                result = false;
                                break;
                            }
                        }
                    }
                } else if (serverProgressState.equals(ServerProgressState.DEPLOY_OK)) {
                    for (ApplicationBean application : applications) {
                        if (this.isSameApplication(this.operation.getApplication(), application)) {
                            if (!this.checkApplication(serverProgress, application, ServerAction.STATE_DEPLOYED, "Reserved")) {
                                result = false;
                                break;
                            }
                        }
                        if (this.isSameApplication(serverAction.getApplicationBean(serverProgress.getNewApplication()),
                            application)) {
                            if (!this.checkApplication(serverProgress, application, ServerAction.STATE_DEPLOYED, "Reserved")) {
                                result = false;
                                break;
                            }
                        }
                        if (this.isSameApplication(serverAction.getApplicationBean(serverProgress.getOldApplication()),
                            application)) {
                            if (!this.checkApplication(serverProgress, application, ServerAction.STATE_DEPLOYED, "Default")) {
                                result = false;
                                break;
                            }
                        }
                    }
                } else if (serverProgressState.equals(ServerProgressState.SET_DEFAULT_OK)) {
                    for (ApplicationBean application : applications) {
                        if (this.isSameApplication(this.operation.getApplication(), application)) {
                            if (!this.checkApplication(serverProgress, application, ServerAction.STATE_DEPLOYED, "Default")) {
                                result = false;
                                break;
                            }
                        }
                        if (this.isSameApplication(serverAction.getApplicationBean(serverProgress.getNewApplication()),
                            application)) {
                            if (!this.checkApplication(serverProgress, application, ServerAction.STATE_DEPLOYED, "Default")) {
                                result = false;
                                break;
                            }
                        }
                        if (this.isSameApplication(serverAction.getApplicationBean(serverProgress.getOldApplication()),
                            application)) {
                            if (!this.checkApplication(serverProgress, application, ServerAction.STATE_DEPLOYED, "Disabled")) {
                                result = false;
                                break;
                            }
                        }
                    }
                } else if (serverProgressState.equals(ServerProgressState.UNDEPLOY_OK)) {
                    for (ApplicationBean application : applications) {
                        if (this.isSameApplication(this.operation.getApplication(), application)) {
                            if (!this.checkApplication(serverProgress, application, ServerAction.STATE_DEPLOYED, "Default")) {
                                result = false;
                                break;
                            }
                        }
                        if (this.isSameApplication(serverAction.getApplicationBean(serverProgress.getNewApplication()),
                            application)) {
                            if (!this.checkApplication(serverProgress, application, ServerAction.STATE_DEPLOYED, "Default")) {
                                result = false;
                                break;
                            }
                        }
                        if (this.isSameApplication(serverAction.getApplicationBean(serverProgress.getOldApplication()),
                            application)) {
                            if (!this.checkApplication(serverProgress, application, ServerAction.STATE_PRESENT, null)) {
                                result = false;
                                break;
                            }
                        }
                    }
                } else if (serverProgressState.equals(ServerProgressState.ERASE_OK)) {
                    for (ApplicationBean application : applications) {
                        if (this.isSameApplication(this.operation.getApplication(), application)) {
                            if (!this.checkApplication(serverProgress, application, ServerAction.STATE_DEPLOYED, "Default")) {
                                result = false;
                                break;
                            }
                        }
                        if (this.isSameApplication(serverAction.getApplicationBean(serverProgress.getNewApplication()),
                            application)) {
                            if (!this.checkApplication(serverProgress, application, ServerAction.STATE_DEPLOYED, "Default")) {
                                result = false;
                                break;
                            }
                        }
                        if (this.isSameApplication(serverAction.getApplicationBean(serverProgress.getOldApplication()),
                            application)) {
                            if (!this.checkApplication(serverProgress, application, null, null)) {
                                result = false;
                                break;
                            }
                        }
                    }
                } else if (serverProgressState.equals(ServerProgressState.DISABLE_APPLICATIONS_OK)
                    || serverProgressState.equals(ServerProgressState.START_OK)) {
                    result = serverAction.isStarted();
                } else if (serverProgressState.equals(ServerProgressState.STOP_OK)
                    || serverProgressState.equals(ServerProgressState.MAINTAIN_OK)) {
                    result = !serverAction.isStarted();
                } else {
                    throw new IllegalStateException("Unknown ServerProgressState: " + serverProgressState);
                }

                this.mergeOperation();

                return result;
            } catch (Exception e) {
                throw new JadortServiceException("Cannot check server '" + server + "': " + e.getMessage(), e);
            }
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException if the checkWorker can not be called in
     *         the current step. (this method can be called only in
     *         EXECUTING_MAINTENANCE_CLUSTER and
     *         EXECUTING_MAINTENANCE_NO_CLUSTER steps)
     */
    public boolean checkWorker(final WorkerBean worker) throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();
            this.checkStep(Step.EXECUTING_MAINTENANCE_CLUSTER, Step.EXECUTING_MAINTENANCE_NO_CLUSTER);

            try {
                boolean result = true;
                WorkerProgressBean workerProgress = this.getWorkerProgress(worker);
                WorkerAction workerAction = WorkerAction.getWorkerAction(workerProgress.getWorker());
                WorkerAction.State state = workerAction.getState();

                WorkerProgressState workerProgressState = workerProgress.getWorkerProgressState();
                if (workerProgressState.equals(WorkerProgressState.INITIAL)
                    || workerProgressState.equals(WorkerProgressState.START_OK)) {
                    result = (state == WorkerAction.State.ACTIVE);
                } else if (workerProgressState.equals(WorkerProgressState.STOP_OK)) {
                    result = (state == WorkerAction.State.STOPPED);
                } else {
                    throw new IllegalStateException("Unknown WorkerProgressState: " + workerProgressState);
                }

                this.mergeOperation();

                return result;
            } catch (Exception e) {
                throw new JadortServiceException("Cannot check worker '" + worker + "': " + e.getMessage(), e);
            }
        }
    }

    /**
     * Compares two application, tests if a two applications are equal). The
     * comparison is based on the name,and version of each application. the
     * result is a true the two applications are the same. the result is false
     * if the two applications are different.
     * 
     * @param a1 - the first application to be compared.
     * @param a2 - the second application to be compared.
     * @return True the two applications are the same. False if the two
     *         applications are different.
     */
    protected boolean isSameApplication(final ApplicationBean a1, final ApplicationBean a2) {
        if (a1 == null || a2 == null || a1.getName() == null) {
            return false;
        }
        return a1.getName().equals(a2.getName())
            && (a1.getVersion() == a2.getVersion() || a1.getVersion() != null && a1.getVersion().equals(a2.getVersion()));
    }

    /**
     * Checks if an application is already deployed on the server, or if has a
     * bad policy, or if has a bad state
     * 
     * @param serverProgress - serverProgress of the server on which the
     *        application will be checked.
     * @param application - the application to be checked
     * @param state - the state that the application should has.
     * @param policy - the policy that the application should has.
     * @return true if no problem is found while checking the application false
     *         if a problem is found while checking the application
     */
    protected boolean checkApplication(final ServerProgressBean serverProgress, final ApplicationBean application,
        final String state, final String policy) {
        if (state == null) {
            if (application != null) {
                serverProgress.appendToLog("Application " + application
                    + " exists (it should not exist), checkServer will return false");
                return false;
            } else {
                return true;
            }
        } else {
            if (state.equals(application.getState())) {
                if (policy == null) {
                    if (application.getPolicy() != null) {
                        serverProgress.appendToLog("Application " + application + " has policy " + application.getPolicy()
                            + " (it should not have any), checkServer will return false");
                        return false;
                    } else {
                        return true;
                    }
                } else {
                    if (!policy.equals(application.getPolicy())) {
                        serverProgress.appendToLog("Application " + application + " has policy " + application.getPolicy()
                            + " instead of " + policy + ", checkServer will return false");
                        return false;
                    } else {
                        return true;
                    }
                }
            } else {
                serverProgress.appendToLog("Application " + application + " has state " + application.getState()
                    + " instead of " + state + ", checkServer will return false");
                return false;
            }
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException <ul>
     *         <li>if the createApplication can not be called in the current
     *         step. (this method can be called only in SELECT_APPLICATION step)
     *         </ul>
     */
    public void createApplication(final URL url) throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();
            this.checkStep(Step.SELECT_APPLICATION);

            ApplicationBean applicationBean;
            File file;
            if (url.getProtocol().equals("file")) {
                try {
                    file = new File(url.toURI());
                } catch (URISyntaxException e) {
                    throw new JadortServiceException("Cannot create file for URL '" + url + "': " + e.getMessage(), e);
                }
            } else {
                try {
                    URLConnection connection = url.openConnection();
                    InputStream reader = connection.getInputStream();
                    String name = url.getPath();
                    name = name.substring(name.lastIndexOf('/'));
                    file = new File(System.getProperty("java.io.tmpdir") + File.separatorChar + "jadort.temp."
                        + System.currentTimeMillis());
                    file.mkdirs();
                    file = new File(file, name);
                    FileOutputStream writer = new FileOutputStream(file);
                    int read;
                    byte[] buffer = new byte[JadortServiceImpl.BUFFER_SIZE];
                    while ((read = reader.read(buffer)) > 0) {
                        writer.write(buffer, 0, read);
                    }
                    writer.close();
                    reader.close();
                } catch (IOException e) {
                    throw new JadortServiceException("Cannot read from URL: '" + url + "'", null);
                }
            }
            try {
                applicationBean = new ApplicationBean(file);
            } catch (Exception e) {
                throw new JadortServiceException("Cannot create application for URL '" + url + "': " + e.getMessage(), e);
            }

            this.operation.setApplication(applicationBean);

            this.mergeOperation();
        }
    }

    /**
     * {@inheritDoc}
     */
    public ApplicationBean getApplication() throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();

            if (this.operation == null) {
                return null;
            } else {
                return this.operation.getApplication();
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    public Step getCurrentStep() {
        synchronized (this) {
            this.refreshOperation();

            if (this.operation == null) {
                return Step.SELECT_OPERATION;
            } else {
                return this.operation.getCurrentStep();
            }
        }
    }

    /**
     * Gets the list of applications.
     * 
     * @throws JadortServiceException If getting the list of applications fails.
     */
    private void getApplications() throws JadortServiceException {
        for (GroupBean groupBean : this.operation.getTopology().getGroups()) {
            List<ApplicationBean> applications = new ArrayList<ApplicationBean>();
            // Assume all servers in a group have the same applications
            ServerBean serverBean = groupBean.getServers().iterator().next();
            try {
                for (ApplicationBean applicationBean : ServerAction.getServerAction(serverBean).listOfApplications()) {
                    applications.add(applicationBean);
                }
            } catch (Exception e) {
                String message = e.getClass().getName() + ": " + e.getMessage();
                throw new JadortServiceException("Failed getting the applications on the servers: " + message, e);
            }
            groupBean.setApplications(applications);
        }
    }

    /**
     * @return true if any serverProgress has an old version, false otherwise.
     */
    private boolean hasOldVersion() {
        for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
            if (serverProgress.hasOldVersion()) {
                return true;
            }
        }
        return false;
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException if it is not allowed to go to the next
     *         step.
     */
    public void next() throws JadortServiceException {
        synchronized (this) {
            if (!this.canGoToNextStep()) {
                throw new JadortServiceException("You are not allowed to go to the next step!", null);
            }

            // We don't need to call refreshOperation() as canGoToNextStep() has
            // already done it
            if (this.getCurrentStep().equals(OperationStateBean.Step.EXECUTING_MAINTENANCE_CLUSTER)
                || this.getCurrentStep().equals(OperationStateBean.Step.EXECUTING_MAINTENANCE_NO_CLUSTER)) {
                if (!this.operation.getAimedWorkerProgressState().equals(WorkerProgressBean.WorkerProgressState.START_OK)) {
                    this.nextExecuteMigration();
                } else {
                    // Else, prepare for Step.FINISHED
                    for (ServerProgressBean serverProgressBean : this.operation.getServerProgressList()) {
                        serverProgressBean.setActionState(ActionState.FINISHED_OK);
                    }
                    for (WorkerProgressBean workerProgressBean : this.operation.getWorkerProgressList()) {
                        workerProgressBean.setActionState(ActionState.FINISHED_OK);
                    }
                    this.goToNextStep();
                }
            } else if (this.getCurrentStep().equals(OperationStateBean.Step.EXECUTING_MIGRATION_PART1)) {
                if (this.operation.getAimedServerProgressState().equals(ServerProgressState.SET_DEFAULT_OK)) {
                    if (this.hasOldVersion()) {
                        // If there's an old version present,
                        // prepare for Step.EXECUTING_MIGRATION_PART2
                        for (ServerProgressBean serverProgressServer : this.operation.getServerProgressList()) {
                            serverProgressServer.setProgress(0);
                        }
                    } else {
                        // Else, prepare for Step.FINISHED
                        for (ServerProgressBean serverProgressServer : this.operation.getServerProgressList()) {
                            serverProgressServer.setActionState(ActionState.FINISHED_OK);
                        }
                        this.operation.setCurrentStep(Step.EXECUTING_MIGRATION_PART2);
                    }
                    this.goToNextStep();
                } else {
                    this.nextExecuteMigration();
                }
            } else if (this.getCurrentStep().equals(OperationStateBean.Step.EXECUTING_MIGRATION_PART2)) {
                if (this.operation.getAimedServerProgressState().equals(ServerProgressState.ERASE_OK)) {
                    for (ServerProgressBean serverProgressServer : this.operation.getServerProgressList()) {
                        serverProgressServer.setActionState(ActionState.FINISHED_OK);
                    }
                    this.goToNextStep();
                } else {
                    this.nextExecuteMigration();
                }
            } else {
                this.goToNextStep();
            }
        }
    }

    /**
     * go the next step
     * 
     * @throws JadortServiceException If getting the list of applications fails.
     */
    private void goToNextStep() throws JadortServiceException {
        Step currentStep = this.getCurrentStep();

        if (currentStep.equals(Step.SELECT_OPERATION)) {
            this.operation.setCurrentStep(Step.INITIALIZE_TOPOLOGY);
        } else if (currentStep.equals(Step.INITIALIZE_TOPOLOGY)) {
            this.getApplications();
            this.operation.setCurrentStep(Step.SELECT_GROUP);
        } else if (currentStep.equals(Step.SELECT_GROUP)) {
            IJadortService.OperationType operationType = this.getOperationType();
            if (operationType.equals(IJadortService.OperationType.MIGRATE)) {
                this.getApplications();
                this.operation.setCurrentStep(Step.SELECT_APPLICATION);
            } else if (operationType.equals(IJadortService.OperationType.MAINTAIN)) {
                this.operation.setCurrentStep(Step.SELECT_SERVERS);
            } else {
                throw new IllegalStateException("Unknown OperationType: " + operationType);
            }
        } else if (currentStep.equals(Step.SELECT_SERVERS)) {
            if (this.operation.getSelectedGroup().getClustered()) {
                this.operation.setCurrentStep(Step.EXECUTING_MAINTENANCE_CLUSTER);
            } else {
                this.operation.setCurrentStep(Step.EXECUTING_MAINTENANCE_NO_CLUSTER);
            }
        } else if (currentStep.equals(Step.EXECUTING_MAINTENANCE_CLUSTER)
            || currentStep.equals(Step.EXECUTING_MAINTENANCE_NO_CLUSTER)) {
            if (this.allServersHaveBeenMaintained()) {
                List<ServerProgressBean> serverProgressList = new ArrayList<ServerProgressBean>();

                for (ServerProgressBean serverProgress : this.operation.getAllServerProgressList()) {
                    serverProgressList.add(serverProgress);
                }
                this.operation.setServerProgressList(serverProgressList);

                List<WorkerProgressBean> workerProgressList = new ArrayList<WorkerProgressBean>();
                for (WorkerProgressBean workerProgress : this.operation.getAllWorkerProgressList()) {
                    workerProgressList.add(workerProgress);
                }

                this.operation.setWorkerProgressList(workerProgressList);
                this.operation.setCurrentStep(Step.FINISHED);
            } else {
                this.operation.setAimedWorkerProgressState(WorkerProgressState.INITIAL);
                this.operation.setAimedWorkerProgressProgress(0);
                this.operation.setAimedServerProgressState(ServerProgressState.INITIAL);
                this.operation.setAimedServerProgressProgress(0);
                this.operation.setCurrentStep(Step.SELECT_SERVERS);
                this.operation.setSelectedServers(null);
            }
        } else if (currentStep.equals(Step.SELECT_APPLICATION)) {
            this.operation.setCurrentStep(Step.EXECUTING_MIGRATION_PART1);
        } else if (currentStep.equals(Step.EXECUTING_MIGRATION_PART1)) {
            this.operation.setCurrentStep(Step.EXECUTING_MIGRATION_PART2);
        } else if (currentStep.equals(Step.EXECUTING_MIGRATION_PART2)) {
            this.operation.setCurrentStep(Step.FINISHED);
        } else if (currentStep.equals(Step.FINISHED)) {
            throw new IllegalStateException("No next for Step: " + currentStep);
        } else {
            throw new IllegalStateException("Unknown Step: " + currentStep);
        }
        this.mergeOperation();
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException if it is not allowed to go to the previous
     *         step.
     */
    public void previous() throws JadortServiceException {
        synchronized (this) {
            if (!this.canGoToPreviousStep()) {
                throw new JadortServiceException("You are not allowed to go to the previous step!", null);
            }

            // We don't need to call refreshOperation() as canGoToPreviousStep()
            // has already done it
            Step currentStep = this.getCurrentStep();

            if (currentStep.equals(OperationStateBean.Step.SELECT_GROUP)) {
                TopologyBean oldTopology = this.operation.getTopology();
                this.operation.setTopology(null);
                this.goToPreviousStep();

                this.em.remove(oldTopology);
                this.em.flush();
            } else if (currentStep.equals(OperationStateBean.Step.SELECT_APPLICATION)) {
                // Don't remove the oldSelectedGroup, as the GroupBeans are
                // created at topology initialization and no copy is done when
                // selecting group).
                //
                // Remove the old progress lists as they're useless now.
                // Remember to copy them in order to avoid exceptions
                List<ServerProgressBean> oldServerProgressList = new ArrayList<ServerProgressBean>(this.operation
                    .getServerProgressList());
                this.operation.setServerProgressList(null);
                this.operation.setSelectedGroup(null);
                ApplicationBean oldApplication = this.operation.getApplication();
                this.operation.setApplication(null);
                this.goToPreviousStep();

                for (ServerProgressBean serverProgressBean : oldServerProgressList) {
                    this.em.remove(serverProgressBean);
                }
                this.em.remove(oldApplication);
                this.em.flush();
            } else if (currentStep.equals(OperationStateBean.Step.SELECT_SERVERS)) {
                if (this.anyMaintainedServerPresent()) {
                    this.operation.setAimedServerProgressProgress(100);
                    this.operation.setAimedServerProgressState(ServerProgressState.START_OK);
                    this.operation.setAimedWorkerProgressProgress(100);
                    this.operation.setAimedWorkerProgressState(WorkerProgressState.START_OK);

                    List<ServerProgressBean> serverProgressList = new ArrayList<ServerProgressBean>();
                    for (ServerProgressBean serverProgress : this.operation.getAllServerProgressList()) {
                        if (serverProgress.getProgress() == 100) {
                            serverProgress.setActionState(ActionState.WAITING);
                            serverProgressList.add(serverProgress);
                        }
                    }
                    this.operation.setServerProgressList(serverProgressList);

                    List<WorkerProgressBean> workerProgressList = new ArrayList<WorkerProgressBean>();
                    for (WorkerProgressBean workerProgress : this.operation.getAllWorkerProgressList()) {
                        if (workerProgress.getProgress() == 100) {
                            workerProgress.setActionState(ActionState.WAITING);
                            workerProgressList.add(workerProgress);
                        }
                    }
                    this.operation.setWorkerProgressList(workerProgressList);
                    if (this.operation.getSelectedGroup().getClustered()) {
                        this.operation.setCurrentStep(Step.EXECUTING_MAINTENANCE_CLUSTER);
                    } else {
                        this.operation.setCurrentStep(Step.EXECUTING_MAINTENANCE_NO_CLUSTER);
                    }
                    this.mergeOperation();
                } else {
                    // Don't remove the oldSelectedGroup, as the GroupBeans are
                    // created at topology initialization and no copy is done
                    // when selecting group).
                    //
                    // Remove the old progress lists as they're useless now.
                    // Remember to copy them in order to avoid exceptions
                    Set<ServerProgressBean> oldServerProgressList = new HashSet<ServerProgressBean>();
                    if (this.operation.getServerProgressList() != null) {
                        oldServerProgressList.addAll(this.operation.getServerProgressList());
                    }
                    if (this.operation.getAllServerProgressList() != null) {
                        oldServerProgressList.addAll(this.operation.getAllServerProgressList());
                    }
                    Set<WorkerProgressBean> oldWorkerProgressList = new HashSet<WorkerProgressBean>();
                    if (this.operation.getWorkerProgressList() != null) {
                        oldWorkerProgressList.addAll(this.operation.getWorkerProgressList());
                    }
                    if (this.operation.getAllWorkerProgressList() != null) {
                        oldWorkerProgressList.addAll(this.operation.getAllWorkerProgressList());
                    }
                    this.operation.setServerProgressList(null);
                    this.operation.setWorkerProgressList(null);
                    this.operation.setAllServerProgressList(null);
                    this.operation.setAllWorkerProgressList(null);
                    this.operation.setSelectedGroup(null);
                    this.goToPreviousStep();

                    for (WorkerProgressBean workerProgressBean : oldWorkerProgressList) {
                        this.em.remove(workerProgressBean);
                    }
                    for (ServerProgressBean serverProgressBean : oldServerProgressList) {
                        this.em.remove(serverProgressBean);
                    }
                    this.em.flush();
                }
            } else if (currentStep.equals(OperationStateBean.Step.EXECUTING_MAINTENANCE_CLUSTER)) {
                if (this.operation.getAimedWorkerProgressState().equals(WorkerProgressBean.WorkerProgressState.INITIAL)) {
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        serverProgress.setActionState(ActionState.WAITING);
                    }
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        workerProgress.setActionState(ActionState.WAITING);
                    }
                    this.goToPreviousStep();
                } else {
                    this.previousExecuteMigration();
                }
            } else if (currentStep.equals(OperationStateBean.Step.EXECUTING_MAINTENANCE_NO_CLUSTER)) {
                if (this.operation.getAimedServerProgressState().equals(ServerProgressBean.ServerProgressState.INITIAL)) {
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        serverProgress.setActionState(ActionState.WAITING);
                    }
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        workerProgress.setActionState(ActionState.WAITING);
                    }
                    this.goToPreviousStep();
                } else {
                    this.previousExecuteMigration();
                }
            } else if (currentStep.equals(OperationStateBean.Step.EXECUTING_MIGRATION_PART1)) {
                if (this.operation.getAimedServerProgressState().equals(ServerProgressState.INITIAL)) {
                    for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                        serverProgress.setActionState(ActionState.WAITING);
                    }
                    this.goToPreviousStep();
                } else {
                    this.previousExecuteMigration();
                }
            } else if (currentStep.equals(OperationStateBean.Step.EXECUTING_MIGRATION_PART2)) {
                if (this.operation.getAimedServerProgressState().equals(ServerProgressState.SET_DEFAULT_OK)) {
                    for (ServerProgressBean serverProgressServer : this.operation.getServerProgressList()) {
                        serverProgressServer.setProgress(100);
                    }
                    this.goToPreviousStep();
                } else {
                    this.previousExecuteMigration();
                }
            } else if (currentStep.equals(OperationStateBean.Step.FINISHED)) {
                for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                    serverProgress.setActionState(ActionState.WAITING);
                }
                if (this.operation.getType().equals(OperationType.MAINTAIN)) {
                    for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                        workerProgress.setActionState(ActionState.WAITING);
                    }
                }
                this.goToPreviousStep();
            } else {
                throw new IllegalStateException("Unknown Step: " + currentStep);
            }
        }
    }

    /**
     * go the previous step
     * 
     * @throws JadortServiceException If getting the list of applications fails.
     */
    private void goToPreviousStep() throws JadortServiceException {
        Step currentStep = this.getCurrentStep();

        if (currentStep.equals(Step.SELECT_OPERATION) || currentStep.equals(Step.INITIALIZE_TOPOLOGY)) {
            throw new IllegalStateException("No previous for Step: " + currentStep);
        } else if (currentStep.equals(Step.SELECT_GROUP)) {
            this.operation.setCurrentStep(Step.INITIALIZE_TOPOLOGY);
        } else if (currentStep.equals(Step.SELECT_APPLICATION)) {
            this.getApplications();
            this.operation.setCurrentStep(Step.SELECT_GROUP);
        } else if (currentStep.equals(Step.SELECT_SERVERS)) {
            this.getApplications();
            this.operation.setCurrentStep(Step.SELECT_GROUP);
        } else if (currentStep.equals(Step.EXECUTING_MAINTENANCE_CLUSTER)
            || currentStep.equals(Step.EXECUTING_MAINTENANCE_NO_CLUSTER)) {
            this.operation.setCurrentStep(Step.SELECT_SERVERS);
            this.operation.setSelectedServers(null);
        } else if (currentStep.equals(Step.EXECUTING_MIGRATION_PART1)) {
            this.getApplications();
            this.operation.setCurrentStep(Step.SELECT_APPLICATION);
        } else if (currentStep.equals(Step.EXECUTING_MIGRATION_PART2)) {
            this.operation.setCurrentStep(Step.EXECUTING_MIGRATION_PART1);
        } else if (currentStep.equals(Step.FINISHED)) {
            if (this.operation.getType().equals(OperationType.MAINTAIN)) {
                if (this.operation.getSelectedGroup().getClustered()) {
                    this.operation.setCurrentStep(Step.EXECUTING_MAINTENANCE_CLUSTER);
                } else {
                    this.operation.setCurrentStep(Step.EXECUTING_MAINTENANCE_NO_CLUSTER);
                }
            } else {
                if (this.hasOldVersion()) {
                    this.operation.setCurrentStep(Step.EXECUTING_MIGRATION_PART2);
                } else {
                    this.operation.setCurrentStep(Step.EXECUTING_MIGRATION_PART1);
                }
            }
        } else {
            throw new IllegalStateException("Unknown Step: " + currentStep);
        }

        this.mergeOperation();
    }

    /**
     * {@inheritDoc}
     */
    public boolean canGoToNextStep() {
        synchronized (this) {
            this.refreshOperation();

            Step currentStep = this.getCurrentStep();
            if (currentStep.equals(Step.SELECT_OPERATION)) {
                return this.operation != null;
            } else if (currentStep.equals(Step.INITIALIZE_TOPOLOGY)) {
                return this.operation.getTopology() != null;
            } else if (currentStep.equals(Step.SELECT_GROUP)) {
                return this.operation.getSelectedGroup() != null;
            } else if (currentStep.equals(Step.SELECT_APPLICATION)) {
                return this.operation.getApplication() != null;
            } else if (currentStep.equals(Step.EXECUTING_MIGRATION_PART1) || currentStep.equals(Step.EXECUTING_MIGRATION_PART2)) {
                for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                    if (!serverProgress.getActionState().equals(ActionState.WAITING)) {
                        return false;
                    }
                }
                return true;
            } else if (currentStep.equals(Step.SELECT_SERVERS)) {
                return this.operation.getSelectedServers() != null && this.operation.getSelectedServers().size() > 0;
            } else if (currentStep.equals(Step.EXECUTING_MAINTENANCE_CLUSTER)
                || currentStep.equals(Step.EXECUTING_MAINTENANCE_NO_CLUSTER)) {
                for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                    if (!serverProgress.getActionState().equals(ActionState.WAITING)) {
                        return false;
                    }
                }
                for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                    if (!workerProgress.getActionState().equals(ActionState.WAITING)) {
                        return false;
                    }
                }
                return true;
            } else if (currentStep.equals(Step.FINISHED)) {
                return false;
            } else {
                throw new IllegalStateException("Unknown Step: " + currentStep);
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    public boolean canGoToPreviousStep() {
        synchronized (this) {
            this.refreshOperation();

            Step currentStep = this.getCurrentStep();
            if (currentStep.equals(Step.SELECT_OPERATION) || currentStep.equals(Step.INITIALIZE_TOPOLOGY)) {
                return false;
            } else if (currentStep.equals(Step.SELECT_GROUP) || currentStep.equals(Step.SELECT_APPLICATION)) {
                return true;
            } else if (currentStep.equals(Step.EXECUTING_MIGRATION_PART1) || currentStep.equals(Step.EXECUTING_MIGRATION_PART2)) {
                if (this.operation.getAimedServerProgressState().equals(ServerProgressState.ERASE_OK)) {
                    return false;
                }
                int progress = -1;
                for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                    if (!serverProgress.getActionState().equals(ActionState.WAITING)
                        && !serverProgress.getActionState().equals(ActionState.FINISHED_ERROR)) {
                        return false;
                    }
                    if (progress == -1) {
                        progress = serverProgress.getProgress();
                        continue;
                    }
                    if (serverProgress.getProgress() != progress) {
                        return false;
                    }
                }
                return true;
            } else if (currentStep.equals(Step.SELECT_SERVERS)) {
                return true;
            } else if (currentStep.equals(Step.EXECUTING_MAINTENANCE_CLUSTER)
                || currentStep.equals(Step.EXECUTING_MAINTENANCE_NO_CLUSTER)) {
                for (ServerProgressBean serverProgress : this.operation.getServerProgressList()) {
                    if (!serverProgress.getActionState().equals(ActionState.WAITING)
                        && !serverProgress.getActionState().equals(ActionState.FINISHED_ERROR)) {
                        return false;
                    }
                }
                for (WorkerProgressBean workerProgress : this.operation.getWorkerProgressList()) {
                    if (!workerProgress.getActionState().equals(ActionState.WAITING)
                        && !workerProgress.getActionState().equals(ActionState.FINISHED_ERROR)) {
                        return false;
                    }
                }
                return true;
            } else if (currentStep.equals(Step.FINISHED)) {
                return true;
            } else {
                throw new IllegalStateException("Unknown Step: " + currentStep);
            }
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException if the selectOperation can not be called
     *         in the current step. (this method can be called only in
     *         SELECT_OPERATION step)
     */
    public void selectOperation(final OperationStateBean selectedOperation) throws JadortServiceException {
        synchronized (this) {
            this.checkStep(Step.SELECT_OPERATION);

            this.operation = this.em.find(OperationStateBean.class, selectedOperation.getId());
            if (this.operation == null) {
                throw new JadortServiceException("Cannot select operation: OperationStateBean with id "
                    + selectedOperation.getId() + " not found", null);
            }
            if (this.getCurrentStep().equals(Step.SELECT_GROUP)) {
                this.getApplications();
            }
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException <ul>
     *         <li>if the selectOperation can not be called in the current step.
     *         (this method can be called only in SELECT_OPERATION step)
     *         </ul>
     */
    public void createNewOperation(final String newOperationName) throws JadortServiceException {
        synchronized (this) {
            this.checkStep(Step.SELECT_OPERATION);
            String filteredOperationName = newOperationName.trim();
            if (filteredOperationName.length() < 1) {
                throw new JadortServiceException("Operation name cannot be empty!", null);
            }

            this.operation = new OperationStateBean();
            this.operation.setName(filteredOperationName);
            this.operation.setDate(new Date());
            this.em.persist(this.operation);
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException <ul>
     *         <li>if the selectOperation can not be called in the current step.
     *         (this method can be called only in SELECT_OPERATION step)
     *         <li>if the operation to delete is still in progress in a critical
     *         phase and therefore cannot be deleter
     *         </ul>
     */
    public void deleteOperation(final OperationStateBean selectedOperation) throws JadortServiceException {
        synchronized (this) {
            this.checkStep(Step.SELECT_OPERATION);

            OperationStateBean toRemove = this.em.find(OperationStateBean.class, selectedOperation.getId());

            if (toRemove.getIsDeletable()) {
                this.em.remove(toRemove);
                this.em.flush();

                if (this.operation != null && selectedOperation.getId() == this.operation.getId()) {
                    this.operation = null;
                }
            } else {
                throw new JadortServiceException("Operation could not be deleted because it currently is in the "
                    + toRemove.getCurrentStep() + " step.", null);
            }
        }
    }

    /**
     * This method checks if an method(or command) can be called in the current.
     * 
     * @param steps :the list of steps where the calling method(the methods
     *        where checkstep is called) can be called. throws
     *        JadortServiceException
     *        <ul>
     *        <li>if it is not allowed to call the method in the current step.
     *        </ul>
     */
    private void checkStep(final Step... steps) throws JadortServiceException {
        String possibleSteps = null;
        Step currentStep = this.getCurrentStep();

        for (Step step : steps) {
            if (currentStep.equals(step)) {
                return;
            } else {
                if (possibleSteps == null) {
                    possibleSteps = step.toString();
                } else {
                    possibleSteps += " or " + step.toString();
                }
            }
        }

        throw new JadortServiceException("That method should only be called when step is " + possibleSteps
            + ". Currently, step is " + currentStep + ".", null);
    }

    /**
     * This method refreshes {@link JadortServiceImpl#operation} and should be
     * called when entering any method that uses datum that can be modified in
     * another thread than the stateful bean's. This is the case when taking any
     * serverProgress action: the {@link OperationStateBean} gets updated by the
     * {@link JadortServiceJMSActions} handler in an asynchronous way.
     */
    private void refreshOperation() {
        if (this.operation != null) {
            this.operation = this.em.find(OperationStateBean.class, this.operation.getId());
        }
    }

    /**
     * This method merges {@link JadortServiceImpl#operation} onto the DB.
     */
    private void mergeOperation() {
        if (this.operation != null) {
            this.operation = this.em.merge(this.operation);
            this.em.flush();
        }
    }

    /**
     * {@inheritDoc}
     */
    public OperationType getOperationType() {
        synchronized (this) {
            this.refreshOperation();
            return this.operation.getType();
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException if the selectServers method can not be
     *         called in the current step (this method can be called only in
     *         SELECT_SERVERS step)
     */
    public void selectServers(final List<ServerBean> selectedServers) throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();
            this.checkStep(Step.SELECT_SERVERS);
            List<ServerBean> selectedServersList = new ArrayList<ServerBean>();
            List<ServerProgressBean> serverProgressList = new ArrayList<ServerProgressBean>();

            for (ServerBean selectedServer : selectedServers) {
                for (ServerBean server : this.operation.getSelectedGroup().getServers()) {
                    if (server.equals(selectedServer)) {
                        if (!selectedServersList.contains(server)) {
                            selectedServersList.add(server);
                            boolean hasProgress = false;
                            if (this.operation.getAllServerProgressList() != null) {
                                for (ServerProgressBean oldServerProgress : this.operation.getAllServerProgressList()) {
                                    if (oldServerProgress.getServer().equals(server)) {
                                        hasProgress = true;
                                        serverProgressList.add(oldServerProgress);
                                        break;
                                    }
                                }
                            }
                            if (!hasProgress) {
                                serverProgressList.add(new ServerProgressBean(server));
                            }
                        }
                        break;
                    }
                }
            }
            if (selectedServersList.size() == 0 || serverProgressList.size() == 0) {
                throw new JadortServiceException("You must select at least one server!", null);
            }
            for (ServerProgressBean serverProgress : serverProgressList) {
                if (serverProgress.getProgress() != 0) {
                    throw new JadortServiceException("The server '" + serverProgress.getServer().getName()
                        + "' has already been maintained!", null);
                }
            }
            this.operation.setSelectedServers(selectedServersList);
            this.operation.setServerProgressList(serverProgressList);

            Set<WorkerBean> workersList = new HashSet<WorkerBean>();
            List<WorkerProgressBean> workerProgressList = new ArrayList<WorkerProgressBean>();
            for (WorkerBean worker : this.operation.getSelectedGroup().getWorkers()) {
                for (ServerBean server : selectedServersList) {
                    if (worker.getServer().equals(server)) {
                        workersList.add(worker);
                    }
                }
            }
            for (WorkerBean worker : workersList) {
                boolean hasProgress = false;
                if (this.operation.getAllWorkerProgressList() != null) {
                    for (WorkerProgressBean oldWorkerProgress : this.operation.getAllWorkerProgressList()) {
                        if (oldWorkerProgress.getWorker().equals(worker)) {
                            hasProgress = true;
                            workerProgressList.add(oldWorkerProgress);
                            break;
                        }
                    }
                }
                if (!hasProgress) {
                    workerProgressList.add(new WorkerProgressBean(worker));
                }
            }
            this.operation.setWorkerProgressList(workerProgressList);

            this.mergeOperation();

            this.operation.addAllServerProgressList(serverProgressList);
            this.operation.addAllWorkerProgressList(workerProgressList);

            this.mergeOperation();
        }
    }

    /**
     * this function allow to stop a server
     * 
     * @param serverProgress - the serverProgress of the server that will be
     *        stopped
     */
    private void stopServer(final ServerProgressBean serverProgress) throws JMSException, JadortServiceException {
        Message message = this.session.createMessage();
        message.setStringProperty(JadortServiceImpl.OPERATION_TYPE, OperationType.MAINTAIN.toString());
        message.setStringProperty(JadortServiceImpl.ACTION_TYPE, ActionType.STOP_SERVER.toString());
        message.setIntProperty(JadortServiceImpl.SERVER_PROGRESS_ID, serverProgress.getId());
        message.setBooleanProperty(JadortServiceImpl.GROUP_CLUSTERED, this.operation.getSelectedGroup().getClustered());
        QueueSender sender = this.session.createSender(this.queue);
        sender.send(message);
    }

    /**
     * this function allow to disable applications on a server
     * 
     * @param serverProgress - the serverProgress of the server on which the
     *        applications will be disabled
     */
    private void disableApplications(final ServerProgressBean serverProgress) throws JMSException, JadortServiceException {
        Message message = this.session.createMessage();
        message.setStringProperty(JadortServiceImpl.OPERATION_TYPE, OperationType.MAINTAIN.toString());
        message.setStringProperty(JadortServiceImpl.ACTION_TYPE, ActionType.DISABLE_APPLICATIONS.toString());
        message.setIntProperty(JadortServiceImpl.SERVER_PROGRESS_ID, serverProgress.getId());
        message.setBooleanProperty(JadortServiceImpl.GROUP_CLUSTERED, this.operation.getSelectedGroup().getClustered());
        QueueSender sender = this.session.createSender(this.queue);
        sender.send(message);
    }

    /**
     * this function allow to enable applications on a server
     * 
     * @param serverProgress - the serverProgress of the server on which the
     *        applications will be enabled
     */
    private void enableApplications(final ServerProgressBean serverProgress) throws JMSException, JadortServiceException {
        Message message = this.session.createMessage();
        message.setStringProperty(JadortServiceImpl.OPERATION_TYPE, OperationType.MAINTAIN.toString());
        message.setStringProperty(JadortServiceImpl.ACTION_TYPE, ActionType.ENABLE_APPLICATIONS.toString());
        message.setIntProperty(JadortServiceImpl.SERVER_PROGRESS_ID, serverProgress.getId());
        message.setBooleanProperty(JadortServiceImpl.GROUP_CLUSTERED, this.operation.getSelectedGroup().getClustered());
        QueueSender sender = this.session.createSender(this.queue);
        sender.send(message);
    }

    /**
     * this function allow to stop a worker
     * 
     * @param workerProgress - the workerProgress of the worker that will be
     *        stopped
     */
    private void stopWorker(final WorkerProgressBean workerProgress) throws JMSException, JadortServiceException {
        Message message = this.session.createMessage();
        message.setStringProperty(JadortServiceImpl.OPERATION_TYPE, OperationType.MAINTAIN.toString());
        message.setStringProperty(JadortServiceImpl.ACTION_TYPE, ActionType.STOP_WORKER.toString());
        message.setIntProperty(JadortServiceImpl.WORKER_PROGRESS_ID, workerProgress.getId());
        message.setBooleanProperty(JadortServiceImpl.GROUP_CLUSTERED, this.operation.getSelectedGroup().getClustered());
        QueueSender sender = this.session.createSender(this.queue);
        sender.send(message);
    }

    /**
     * this function allow to start a worker
     * 
     * @param workerProgress - the workerProgress of the worker that will be
     *        started
     */
    private void startWorker(final WorkerProgressBean workerProgress) throws JMSException, JadortServiceException {
        Message message = this.session.createMessage();
        message.setStringProperty(JadortServiceImpl.OPERATION_TYPE, OperationType.MAINTAIN.toString());
        message.setStringProperty(JadortServiceImpl.ACTION_TYPE, ActionType.START_WORKER.toString());
        message.setIntProperty(JadortServiceImpl.WORKER_PROGRESS_ID, workerProgress.getId());
        message.setBooleanProperty(JadortServiceImpl.GROUP_CLUSTERED, this.operation.getSelectedGroup().getClustered());
        QueueSender sender = this.session.createSender(this.queue);
        sender.send(message);
    }

    /**
     * this function allow to start a server
     * 
     * @param serverProgress - the serverProgress of the server that will be
     *        started
     */
    private void startServer(final ServerProgressBean serverProgress) throws JMSException, JadortServiceException {
        Message message = this.session.createMessage();
        message.setStringProperty(JadortServiceImpl.OPERATION_TYPE, OperationType.MAINTAIN.toString());
        message.setStringProperty(JadortServiceImpl.ACTION_TYPE, ActionType.START_SERVER.toString());
        message.setIntProperty(JadortServiceImpl.SERVER_PROGRESS_ID, serverProgress.getId());
        message.setBooleanProperty(JadortServiceImpl.GROUP_CLUSTERED, this.operation.getSelectedGroup().getClustered());
        QueueSender sender = this.session.createSender(this.queue);
        sender.send(message);
    }

    /**
     * this function allow to maintain a server
     * 
     * @param serverProgress - the serverProgress of the server that will be
     *        maintained
     */
    private void maintainServer(final ServerProgressBean serverProgress) throws JMSException, JadortServiceException {
        Message message = this.session.createMessage();
        message.setStringProperty(JadortServiceImpl.OPERATION_TYPE, OperationType.MAINTAIN.toString());
        message.setStringProperty(JadortServiceImpl.ACTION_TYPE, ActionType.MAINTAIN_SERVER.toString());
        message.setIntProperty(JadortServiceImpl.SERVER_PROGRESS_ID, serverProgress.getId());
        message.setBooleanProperty(JadortServiceImpl.GROUP_CLUSTERED, this.operation.getSelectedGroup().getClustered());
        QueueSender sender = this.session.createSender(this.queue);
        sender.send(message);
    }

    private boolean allServersHaveBeenMaintained() {
        for (ServerBean server : this.operation.getSelectedGroup().getServers()) {
            boolean serverFound = false;
            for (ServerProgressBean serverProgress : this.operation.getAllServerProgressList()) {
                if (serverProgress.getServer().equals(server)) {
                    serverFound = true;
                    if (serverProgress.getProgress() != 100) {
                        return false;
                    }
                }
            }
            if (!serverFound) {
                return false;
            }
        }
        return true;
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException If fetching fails or if the getServers()
     *         method can not be called in the current step (this method can be
     *         called only in the SELECT_SERVERS step).
     */
    public List<ServerBean> getServers() throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();
            this.refreshActiveSessions();
            this.refreshMaintainStatus();
            this.checkStep(Step.SELECT_SERVERS);
            return this.fetchServers();
        }
    }

    /**
     * @return true if any server has been maintained, false otherwise
     */
    private boolean anyMaintainedServerPresent() {
        if (this.operation.getAllServerProgressList() != null) {
            for (ServerProgressBean serverProgress : this.operation.getAllServerProgressList()) {
                if (serverProgress.getProgress() == 100) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException <ul>
     *         <li>if the restartWorkermethod can not be called in the current
     *         step.
     *         <li>if the worker state is not in error state. (this method can
     *         be called only in EXECUTING_MAINTENANCE step)
     *         </ul>
     */
    public void restartWorker(final WorkerBean worker) throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();
            this.checkStep(Step.EXECUTING_MAINTENANCE_CLUSTER, Step.EXECUTING_MAINTENANCE_NO_CLUSTER);

            WorkerProgressBean workerProgress = this.getWorkerProgress(worker);
            if (!workerProgress.getActionState().equals(ActionState.FINISHED_ERROR)) {
                throw new JadortServiceException("Worker state for worker '" + worker.getName() + "' is "
                    + workerProgress.getActionState() + ", it should be " + ActionState.FINISHED_ERROR, null);
            }
            workerProgress.setActionState(ActionState.RUNNING);

            this.mergeOperation();

            this.reachAimedWorkerProgressState(this.getWorkerProgress(worker));
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException <ul>
     *         <li>if the ignoreWorker can not be called in the current step.
     *         <li>if the worker state is not in error state. (this method can
     *         be called only in EXECUTING_MAINTENANCE_CLUSTER and
     *         EXECUTING_MAINTENANCE_NO_CLUSTER step)
     *         </ul>
     */
    public void ignoreWorker(final WorkerBean worker) throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();
            this.checkStep(Step.EXECUTING_MAINTENANCE_CLUSTER, Step.EXECUTING_MAINTENANCE_NO_CLUSTER);

            WorkerProgressBean workerProgress = this.getWorkerProgress(worker);
            if (!workerProgress.getActionState().equals(ActionState.FINISHED_ERROR)) {
                throw new JadortServiceException("Worker state for worker '" + worker.getName() + "' is "
                    + workerProgress.getActionState() + ", it should be " + ActionState.FINISHED_ERROR, null);
            }

            workerProgress.appendToLog("Error on the worker has been ignored by the user");

            workerProgress.setActionState(ActionState.WAITING);
            workerProgress.setWorkerProgressState(this.operation.getAimedWorkerProgressState());

            this.mergeOperation();
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException <ul>
     *         <li>if the abortWorker can not be called in the current step.
     *         <li>if the worker state is not in running state. (this method can
     *         be called only in EXECUTING_MAINTENANCE_CLUSTER and
     *         EXECUTING_MAINTENANCE_NO_CLUSTER step)
     *         </ul>
     */
    public void abortWorker(final WorkerBean worker) throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();
            this.checkStep(Step.EXECUTING_MAINTENANCE_CLUSTER, Step.EXECUTING_MAINTENANCE_NO_CLUSTER);

            WorkerProgressBean workerProgress = this.getWorkerProgress(worker);
            if (!workerProgress.getActionState().equals(ActionState.RUNNING)) {
                throw new JadortServiceException("Worker state for worker '" + worker.getName() + "' is "
                    + workerProgress.getActionState() + ", it should be " + ActionState.RUNNING, null);
            }

            workerProgress.appendToLog("Action has been aborted by the user");

            workerProgress.setActionState(ActionState.FINISHED_ERROR);
            workerProgress.setProgress(this.operation.getAimedWorkerProgressProgress());

            this.mergeOperation();
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @throws JadortServiceException If fetching fails.
     */
    public List<ServerBean> getSelectedServers() throws JadortServiceException {
        synchronized (this) {
            this.refreshOperation();
            return this.fetchSelectedServers();
        }
    }
}
