/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Copyright (C) 2008-2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: GlassFishServerAction.java 3038 2009-01-28 11:19:11Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.service.action;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;

import javax.management.Attribute;
import javax.management.MBeanServerConnection;
import javax.management.ObjectName;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXServiceURL;

import org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean;
import org.ow2.jasmine.jadort.api.entities.topology.ServerBean;

/**
 * Action for the WebSphere server.
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
public class GlassFishServerAction extends ServerAction {
    private String name;

    private String url;

    private String username;

    private String password;

    private ObjectName j2eeServer;

    private MBeanServerConnection mbscnx = null;

    public GlassFishServerAction(final ServerBean server) {
        this.name = server.getName();
        this.url = server.getServerConnector().getConnectorUrl();
        this.username = server.getServerConnector().getUsername();
        this.password = server.getServerConnector().getPassword();
        this.appendToLog("Created GlassFishAction for server '" + this.name + "'");
    }

    /**
     * Checks connectivity with the JMX URL.
     */
    @SuppressWarnings("unchecked")
    public void checkJMXConnection() throws Exception {
        if (this.mbscnx != null) {
            try {
                this.mbscnx.getMBeanCount();
            } catch (IOException e) {
                this.mbscnx = null;
                this.appendToLog("Connection dropped, reconnecting to JMX server on URL '" + this.url + "'");
            }
        }

        if (this.mbscnx == null) {
            this.appendToLog("Trying to connect to JMX server on URL '" + this.url + "'");

            ClassLoader oldCtxCL = Thread.currentThread().getContextClassLoader();
            try {
                Map<String, Object> env = new HashMap<String, Object>();
                if (this.username != null && this.password != null) {
                    String[] creds = {this.username, this.password};
                    env.put(JMXConnector.CREDENTIALS, creds);
                }
                JMXConnector c = JMXConnectorFactory.connect(new JMXServiceURL(this.url), env);
                this.mbscnx = c.getMBeanServerConnection();
                this.j2eeServer = ((Set<ObjectName>) this.mbscnx.queryNames(new ObjectName("*:j2eeType=J2EEServer,*"), null))
                    .iterator().next();
            } finally {
                Thread.currentThread().setContextClassLoader(oldCtxCL);
            }

            this
                .appendToLog("JMX server connection OK for server '" + this.name + "', J2EEServer is '" + this.j2eeServer + "'");
        }
    }

    @Override
    public boolean canDeployApplications() throws Exception {
        return false;
    }

    @Override
    public void deploy(final String appName) throws Exception {
        throw new UnsupportedOperationException("Operation not implemented yet !");
    }

    @SuppressWarnings( {"unchecked"})
    @Override
    public boolean enableOrDisableApplications(final boolean enable) throws Exception {
        this.checkJMXConnection();
        this.appendToLog("Enabling or disabling all applications on this server");

        boolean failed = false;

        for (ObjectName app : (Set<ObjectName>) this.mbscnx.queryNames(new ObjectName("*:type=Manager,*"), null)) {
            String host = app.getKeyProperty("host");
            if ("__asadmin".equals(host)) {
                continue;
            }
            String path = app.getKeyProperty("path");
            if (path == null || path.length() < 1) {
                path = "/";
            }
            ObjectName filter;
            try {
                filter = ((Set<ObjectName>) this.mbscnx.queryNames(new ObjectName("*:type=J2EEFilter,path=" + path), null))
                    .iterator().next();
            } catch (NoSuchElementException e) {
                this.appendToLog("\tApplication on path '" + path
                    + "' doesn't have the OnlyAllowUsersWithSessionFilter filter! Please "
                    + "import the filter JAR from the jadort-samples package and set it in the application's descriptor");
                failed = true;
                continue;
            }

            this.mbscnx.setAttribute(filter, new Attribute("active", Boolean.valueOf(!enable)));
            this.appendToLog("\tFilter for application on path '" + path + "' has been set as " + (enable ? "in" : "")
                + "active, application is therefore " + (enable ? "enabled" : "disabled"));
        }
        if (failed) {
            this.appendToLog("At least one application on this server couldn't be " + (enable ? "enabled" : "disabled"));
            return false;
        } else {
            this.appendToLog("All applications on this server are now " + (enable ? "enabled" : "disabled"));
            return true;
        }
    }

    @Override
    public void erase(final String appName) throws Exception {
        throw new UnsupportedOperationException("Operation not implemented yet !");
    }

    @Override
    public int getActiveSessions(final String appName) throws Exception {
        throw new UnsupportedOperationException("Operation not implemented yet !");
    }

    @Override
    @SuppressWarnings("unchecked")
    public int getActiveSessions() throws Exception {
        this.checkJMXConnection();

        int totalActiveSessions = 0;
        Set<ObjectName> managers = this.mbscnx.queryNames(new ObjectName("*:type=Manager,*"), null);
        for (ObjectName manager : managers) {
            totalActiveSessions += ((Integer) this.mbscnx.getAttribute(manager, "activeSessions")).intValue();
        }
        return totalActiveSessions;
    }

    @Override
    public ApplicationBean getApplicationBean(final String appName) throws Exception {
        throw new UnsupportedOperationException("Operation not implemented yet !");
    }

    @Override
    public boolean isStarted() throws Exception {
        this.appendToLog("Checking server state via JMX");

        boolean result = true;
        try {
            this.checkJMXConnection();
        } catch (IOException e) {
            result = false;
        }

        this.appendToLog("Server.started is '" + result + "'");
        return result;
    }

    @Override
    @SuppressWarnings("unchecked")
    public List<ApplicationBean> listOfApplications() throws Exception {
        this.checkJMXConnection();
        this.appendToLog("Getting list of applications");
        List<ApplicationBean> applications = new ArrayList<ApplicationBean>();
        Set<ObjectName> managers = this.mbscnx.queryNames(new ObjectName("*:type=Manager,*"), null);
        for (ObjectName manager : managers) {
            String host = manager.getKeyProperty("host");
            if ("__asadmin".equals(host)) {
                continue;
            }
            String name = manager.getKeyProperty("path");
            if (name != null) {
                name = name.substring(name.indexOf("/") + 1);
                if (name.length() == 0) {
                    name = "[ ROOT ]";
                }
                ApplicationBean applicationBean = new ApplicationBean(name, null);
                applicationBean.setState(ServerAction.STATE_DEPLOYED);
                applications.add(applicationBean);
            }
        }
        this.appendToLog("Got the list of applications");
        return applications;
    }

    @Override
    public void maintain() throws Exception {
        // TODO Auto-generated method stub
    }

    @Override
    public String setDefault(final String appName) throws Exception {
        throw new UnsupportedOperationException("Operation not implemented yet !");
    }

    @Override
    public void start() throws Exception {
        this.checkJMXConnection();
        this.appendToLog("Starting server");

        this.mbscnx.invoke(this.j2eeServer, "start", null, null);

        this.appendToLog("Server is started");
    }

    @Override
    public void stop() throws Exception {
        this.checkJMXConnection();
        this.appendToLog("Stopping server");

        Exception stopException = null;
        try {
            this.mbscnx.invoke(this.j2eeServer, "stop", null, null);
        } catch (Exception e) {
            stopException = e;
        }
        if (this.isStarted()) {
            if (stopException == null) {
                throw new Exception("Server did not stop, but did not throw any exception either");
            } else {
                throw stopException;
            }
        }

        this.appendToLog("Server is stopped");
    }

    @Override
    public void undeploy(final String appName) throws Exception {
        throw new UnsupportedOperationException("Operation not implemented yet !");
    }

    @Override
    public String upload(final ApplicationBean application) throws Exception {
        throw new UnsupportedOperationException("Operation not implemented yet !");
    }
}
