/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Copyright (C) 2008-2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JadortServiceJMSActions.java 3035 2009-01-28 10:24:18Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.service.implementation;

import java.io.ByteArrayOutputStream;
import java.io.OutputStream;
import java.io.PrintStream;

import javax.ejb.ActivationConfigProperty;
import javax.ejb.MessageDriven;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

import org.ow2.jasmine.jadort.api.IJadortService.OperationType;
import org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean;
import org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean;
import org.ow2.jasmine.jadort.api.entities.deployment.WorkerProgressBean;
import org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean.ActionState;
import org.ow2.jasmine.jadort.service.action.ServerAction;
import org.ow2.jasmine.jadort.service.action.WorkerAction;
import org.ow2.jasmine.jadort.service.implementation.JadortServiceImpl.ActionType;

/**
 * JadortServiceJMSActions class is the Message Driven Bean that allows
 * asynchronous execution of tasks in the maintenance / migration execution
 * step.
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
@MessageDriven(activationConfig = {
    @ActivationConfigProperty(propertyName = "destinationType", propertyValue = "javax.jms.Queue"),
    @ActivationConfigProperty(propertyName = "destination", propertyValue = JadortServiceImpl.QUEUE_NAME)})
public class JadortServiceJMSActions implements MessageListener {

    @PersistenceContext
    protected EntityManager em;

    public void onMessage(final Message message) {
        try {
            OperationType operationType = OperationType.valueOf(message.getStringProperty(JadortServiceImpl.OPERATION_TYPE));
            ActionType actionType = ActionType.valueOf(message.getStringProperty(JadortServiceImpl.ACTION_TYPE));

            if (operationType.equals(OperationType.MIGRATE)) {
                this.processMigration(message, actionType);
            } else if (operationType.equals(OperationType.MAINTAIN)) {
                boolean clustered = message.getBooleanProperty(JadortServiceImpl.GROUP_CLUSTERED);
                this.processMaintain(message, actionType, clustered);
            } else {
                throw new IllegalArgumentException("Unknown operation type: " + operationType);
            }
        } catch (Exception e) {
            System.out.println("Failed processing JMS message '" + message + "': " + e.getMessage());
            e.printStackTrace();
        }
    }

    /**
     * Execute the migration actions.
     * 
     * @param message
     * @param actionType
     * @throws JMSException
     */
    private void processMigration(final Message message, final ActionType actionType) throws JMSException {
        // Read common message parameters
        Integer serverProgressId = message.getIntProperty(JadortServiceImpl.SERVER_PROGRESS_ID);
        ServerProgressBean serverProgress = this.em.find(ServerProgressBean.class, serverProgressId);

        if (actionType == ActionType.UPLOAD) {
            Integer applicationId = message.getIntProperty(JadortServiceImpl.APPLICATION_ID);
            ApplicationBean application = this.em.find(ApplicationBean.class, applicationId);

            this.upload(serverProgress, application);
        } else if (actionType == ActionType.DEPLOY) {
            String deployApplicationName = message.getStringProperty(JadortServiceImpl.APPLICATION_NAME);

            this.deploy(serverProgress, deployApplicationName);
        } else if (actionType == ActionType.SET_DEFAULT) {
            String setDefaultApplicationName = message.getStringProperty(JadortServiceImpl.APPLICATION_NAME);

            this.setAsDefault(serverProgress, setDefaultApplicationName);
        } else if (actionType == ActionType.UNDEPLOY) {
            String undeployApplicationName = message.getStringProperty(JadortServiceImpl.APPLICATION_NAME);

            this.undeploy(serverProgress, undeployApplicationName);
        } else if (actionType == ActionType.ERASE) {
            String eraseApplicationName = message.getStringProperty(JadortServiceImpl.APPLICATION_NAME);

            this.erase(serverProgress, eraseApplicationName);
        } else {
            serverProgress.setActionState(ActionState.FINISHED_ERROR);
            serverProgress.appendToLog("Unknown ActionType: " + actionType);
        }

        this.mergeObject(serverProgress);
    }

    /**
     * Execute the maintain actions.
     * 
     * @param message
     * @param actionType
     * @param clustered
     * @throws JMSException
     */
    private void processMaintain(final Message message, final ActionType actionType, final boolean clustered)
        throws JMSException {
        if (actionType == ActionType.START_WORKER || actionType == ActionType.STOP_WORKER) {
            Integer workerProgressId = message.getIntProperty(JadortServiceImpl.WORKER_PROGRESS_ID);
            WorkerProgressBean workerProgress = this.em.find(WorkerProgressBean.class, workerProgressId);

            if (actionType == ActionType.START_WORKER) {
                this.startWorker(workerProgress, clustered);
            } else if (actionType == ActionType.STOP_WORKER) {
                this.stopWorker(workerProgress, clustered);
            }
            this.mergeObject(workerProgress);
        } else {
            Integer serverProgressId = message.getIntProperty(JadortServiceImpl.SERVER_PROGRESS_ID);
            ServerProgressBean serverProgress = this.em.find(ServerProgressBean.class, serverProgressId);
            if (actionType == ActionType.START_SERVER) {
                this.start(serverProgress, clustered);
            } else if (actionType == ActionType.MAINTAIN_SERVER) {
                this.maintain(serverProgress, clustered);
            } else if (actionType == ActionType.STOP_SERVER) {
                this.stop(serverProgress, clustered);
            } else if (actionType == ActionType.DISABLE_APPLICATIONS) {
                this.enableOrDisableApplications(serverProgress, false);
            } else if (actionType == ActionType.ENABLE_APPLICATIONS) {
                this.enableOrDisableApplications(serverProgress, true);
            } else {
                serverProgress.setActionState(ActionState.FINISHED_ERROR);
                serverProgress.appendToLog("Unknown ActionType: " + actionType);
            }
            this.mergeObject(serverProgress);
        }

    }

    /**
     * This method merges an object onto the DB.
     */
    private <T> void mergeObject(final T serverProgress) {
        this.em.merge(serverProgress);
        this.em.flush();
    }

    /**
     * this function allows to upload an application on a server
     * 
     * @param serverProgress - the serverProgress server on which the
     *        application will be uploaded.
     * @param application - the application to be uploaded.
     * @throws JMSException
     */
    private void upload(final ServerProgressBean serverProgress, final ApplicationBean application) {
        OutputStream stackTrace = null;
        ServerAction serverAction = null;

        String newApplication = null;
        try {
            serverAction = ServerAction.getServerAction(serverProgress.getServer());
            newApplication = serverAction.upload(application);
        } catch (Exception e) {
            stackTrace = new ByteArrayOutputStream();
            e.printStackTrace(new PrintStream(stackTrace));
        }

        if (newApplication != null) {
            serverProgress.setNewApplication(newApplication);
        }
        if (serverAction != null) {
            serverProgress.appendToLog(serverAction.flushLog());
        }
        if (stackTrace != null) {
            serverProgress.appendToLog(stackTrace.toString());
        }

        serverProgress.setProgress(33);

        if (stackTrace == null) {
            serverProgress.setActionState(ActionState.WAITING);
            serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.UPLOAD_OK);
        } else {
            serverProgress.setActionState(ActionState.FINISHED_ERROR);
        }
    }

    /**
     * this function allow to deploy an application on the server
     * 
     * @param serverProgress : the serverProgress server on which the
     *        application will be deployed
     * @param applicationName : the application name to be deployed
     */
    private void deploy(final ServerProgressBean serverProgress, final String applicationName) throws JMSException {
        OutputStream stackTrace = null;
        ServerAction serverAction = null;

        try {
            serverAction = ServerAction.getServerAction(serverProgress.getServer());
            serverAction.deploy(applicationName);
        } catch (Exception e) {
            stackTrace = new ByteArrayOutputStream();
            e.printStackTrace(new PrintStream(stackTrace));
        }

        if (serverAction != null) {
            serverProgress.appendToLog(serverAction.flushLog());
        }
        if (stackTrace != null) {
            serverProgress.appendToLog(stackTrace.toString());
        }

        // Check for both == and .equals to avoid
        // NullPointerExceptions
        if (applicationName == serverProgress.getOldApplication() || applicationName.equals(serverProgress.getOldApplication())) {
            serverProgress.setProgress(0);
            if (stackTrace == null) {
                serverProgress.setActionState(ActionState.WAITING);
                serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.SET_DEFAULT_OK);
            } else {
                serverProgress.setActionState(ActionState.FINISHED_ERROR);
            }
        } else {
            serverProgress.setProgress(66);
            if (stackTrace == null) {
                serverProgress.setActionState(ActionState.WAITING);
                serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.DEPLOY_OK);
            } else {
                serverProgress.setActionState(ActionState.FINISHED_ERROR);
            }
        }
    }

    /**
     * This function allows to activate an application on the the server. it
     * makes the application policy to default.
     * 
     * @param serverProgress - the serverProgress server on which the new
     *        application will be activated.
     * @param applicationName : the application name
     */
    private void setAsDefault(final ServerProgressBean serverProgress, final String applicationName) {
        OutputStream stackTrace = null;
        ServerAction serverAction = null;

        String previousDefault = null;
        try {
            serverAction = ServerAction.getServerAction(serverProgress.getServer());
            previousDefault = serverAction.setDefault(applicationName);
        } catch (Exception e) {
            stackTrace = new ByteArrayOutputStream();
            e.printStackTrace(new PrintStream(stackTrace));
        }

        if (previousDefault != null && !previousDefault.endsWith(applicationName) && serverProgress.getOldApplication() == null) {
            serverProgress.setOldApplication(previousDefault);
        }

        if (serverAction != null) {
            serverProgress.appendToLog(serverAction.flushLog());
        }
        if (stackTrace != null) {
            serverProgress.appendToLog(stackTrace.toString());
        }

        // Check for both == and .equals to avoid
        // NullPointerExceptions
        if (applicationName == serverProgress.getOldApplication() || applicationName.equals(serverProgress.getOldApplication())) {
            serverProgress.setProgress(66);
            if (stackTrace == null) {
                serverProgress.setActionState(ActionState.WAITING);
                serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.DEPLOY_OK);
            } else {
                serverProgress.setActionState(ActionState.FINISHED_ERROR);
            }
        } else {
            serverProgress.setProgress(100);
            if (stackTrace == null) {
                serverProgress.setActionState(ActionState.WAITING);
                serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.SET_DEFAULT_OK);
            } else {
                serverProgress.setActionState(ActionState.FINISHED_ERROR);
            }
        }
    }

    /**
     * this function allows to undeploy an application on the server
     * 
     * @param serverProgress : the serverProgress server on which the
     *        application will be undeployed
     * @param applicationName : the application name to be undeployed
     */
    private void undeploy(final ServerProgressBean serverProgress, final String applicationName) {
        OutputStream stackTrace = null;
        ServerAction serverAction = null;

        try {
            serverAction = ServerAction.getServerAction(serverProgress.getServer());
            serverAction.undeploy(applicationName);
        } catch (Exception e) {
            stackTrace = new ByteArrayOutputStream();
            e.printStackTrace(new PrintStream(stackTrace));
        }

        if (serverAction != null) {
            serverProgress.appendToLog(serverAction.flushLog());
        }
        if (stackTrace != null) {
            serverProgress.appendToLog(stackTrace.toString());
        }

        // Check for both == and .equals to avoid
        // NullPointerExceptions
        if (applicationName == serverProgress.getOldApplication() || applicationName.equals(serverProgress.getOldApplication())) {
            serverProgress.setProgress(50);
            if (stackTrace == null) {
                serverProgress.setActionState(ActionState.WAITING);
                serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.UNDEPLOY_OK);
            } else {
                serverProgress.setActionState(ActionState.FINISHED_ERROR);
            }
        } else {
            serverProgress.setProgress(33);
            if (stackTrace == null) {
                serverProgress.setActionState(ActionState.WAITING);
                serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.UPLOAD_OK);
            } else {
                serverProgress.setActionState(ActionState.FINISHED_ERROR);
            }
        }
    }

    /**
     * this function allow to erase an application on the server
     * 
     * @param serverProgress : the serverProgress server on which the new
     *        application will be erased
     * @param applicationName : the application name to be erased
     */
    private void erase(final ServerProgressBean serverProgress, final String applicationName) {
        OutputStream stackTrace = null;
        ServerAction serverAction = null;

        try {
            serverAction = ServerAction.getServerAction(serverProgress.getServer());
            serverAction.erase(applicationName);
        } catch (Exception e) {
            stackTrace = new ByteArrayOutputStream();
            e.printStackTrace(new PrintStream(stackTrace));
        }

        if (serverAction != null) {
            serverProgress.appendToLog(serverAction.flushLog());
        }
        if (stackTrace != null) {
            serverProgress.appendToLog(stackTrace.toString());
        }

        // Check for both == and .equals to avoid
        // NullPointerExceptions
        if (applicationName == serverProgress.getOldApplication() || applicationName.equals(serverProgress.getOldApplication())) {
            serverProgress.setProgress(100);
            if (stackTrace == null) {
                serverProgress.setActionState(ActionState.WAITING);
                serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.ERASE_OK);
            } else {
                serverProgress.setActionState(ActionState.FINISHED_ERROR);
            }
        } else {
            serverProgress.setProgress(0);
            if (stackTrace == null) {
                serverProgress.setActionState(ActionState.WAITING);
                serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.INITIAL);
            } else {
                serverProgress.setActionState(ActionState.FINISHED_ERROR);
            }
        }
    }

    /**
     * this function allows to start a server
     * 
     * @param serverProgress - the serverProgress server that will be started
     * @throws JMSException
     */
    private void start(final ServerProgressBean serverProgress, final boolean clustered) {
        OutputStream stackTrace = null;
        ServerAction serverAction = null;
        if (clustered) {
            if (serverProgress.getProgress() == 60) {
                try {
                    if (serverProgress.getServer().getTarget() == null) {
                        serverAction = ServerAction.getServerAction(serverProgress.getServer());
                    } else {
                        serverAction = ServerAction.getServerAction(serverProgress.getServer().getTarget());
                    }
                    serverAction.start();
                } catch (Exception e) {
                    stackTrace = new ByteArrayOutputStream();
                    e.printStackTrace(new PrintStream(stackTrace));
                }
                if (serverAction != null) {
                    serverProgress.appendToLog(serverAction.flushLog());
                }
                if (stackTrace != null) {
                    serverProgress.appendToLog(stackTrace.toString());
                }

                serverProgress.setProgress(80);
                if (stackTrace == null) {
                    serverProgress.setActionState(ActionState.WAITING);
                    serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.START_OK);
                } else {
                    serverProgress.setActionState(ActionState.FINISHED_ERROR);
                }
            } else if (serverProgress.getProgress() == 40) {
                try {
                    serverAction = ServerAction.getServerAction(serverProgress.getServer());
                    serverAction.start();
                } catch (Exception e) {
                    stackTrace = new ByteArrayOutputStream();
                    e.printStackTrace(new PrintStream(stackTrace));
                }
                if (serverAction != null) {
                    serverProgress.appendToLog(serverAction.flushLog());
                }
                if (stackTrace != null) {
                    serverProgress.appendToLog(stackTrace.toString());
                }

                serverProgress.setProgress(20);
                if (stackTrace == null) {
                    serverProgress.setActionState(ActionState.WAITING);
                    serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.INITIAL);
                } else {
                    serverProgress.setActionState(ActionState.FINISHED_ERROR);
                }
            }
        } else {
            if (serverProgress.getProgress() == 66) {
                try {
                    if (serverProgress.getServer().getTarget() == null) {
                        serverAction = ServerAction.getServerAction(serverProgress.getServer());
                    } else {
                        serverAction = ServerAction.getServerAction(serverProgress.getServer().getTarget());
                    }
                    serverAction.start();
                } catch (Exception e) {
                    stackTrace = new ByteArrayOutputStream();
                    e.printStackTrace(new PrintStream(stackTrace));
                }
                if (serverAction != null) {
                    serverProgress.appendToLog(serverAction.flushLog());
                }
                if (stackTrace != null) {
                    serverProgress.appendToLog(stackTrace.toString());
                }

                serverProgress.setProgress(83);
                if (stackTrace == null) {
                    serverProgress.setActionState(ActionState.WAITING);
                    serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.START_OK);
                } else {
                    serverProgress.setActionState(ActionState.FINISHED_ERROR);
                }
            } else if (serverProgress.getProgress() == 50) {
                try {
                    serverAction = ServerAction.getServerAction(serverProgress.getServer());
                    serverAction.start();
                } catch (Exception e) {
                    stackTrace = new ByteArrayOutputStream();
                    e.printStackTrace(new PrintStream(stackTrace));
                }
                if (serverAction != null) {
                    serverProgress.appendToLog(serverAction.flushLog());
                }
                if (stackTrace != null) {
                    serverProgress.appendToLog(stackTrace.toString());
                }

                serverProgress.setProgress(33);
                if (stackTrace == null) {
                    serverProgress.setActionState(ActionState.WAITING);
                    serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.DISABLE_APPLICATIONS_OK);
                } else {
                    serverProgress.setActionState(ActionState.FINISHED_ERROR);
                }
            }
        }
    }

    /**
     * this function allows to maintain a server
     * 
     * @param serverProgress - the serverProgress server that will be maintained
     */
    private void maintain(final ServerProgressBean serverProgress, final boolean clustered) {
        OutputStream stackTrace = null;
        ServerAction serverAction = null;

        if (clustered) {
            if (serverProgress.getProgress() == 40) {
                try {
                    serverAction = ServerAction.getServerAction(serverProgress.getServer());
                    serverAction.maintain();
                } catch (Exception e) {
                    stackTrace = new ByteArrayOutputStream();
                    e.printStackTrace(new PrintStream(stackTrace));
                }
                if (serverAction != null) {
                    serverProgress.appendToLog(serverAction.flushLog());
                }
                if (stackTrace != null) {
                    serverProgress.appendToLog(stackTrace.toString());
                }

                serverProgress.setProgress(60);

                if (stackTrace == null) {
                    serverProgress.setActionState(ActionState.WAITING);
                    serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.MAINTAIN_OK);
                } else {
                    serverProgress.setActionState(ActionState.FINISHED_ERROR);
                }
            } else if (serverProgress.getProgress() == 60) {
                try {
                    serverAction = ServerAction.getServerAction(serverProgress.getServer());
                    serverAction.maintain();
                } catch (Exception e) {
                    stackTrace = new ByteArrayOutputStream();
                    e.printStackTrace(new PrintStream(stackTrace));
                }
                if (serverAction != null) {
                    serverProgress.appendToLog(serverAction.flushLog());
                }
                if (stackTrace != null) {
                    serverProgress.appendToLog(stackTrace.toString());
                }

                serverProgress.setProgress(40);

                if (stackTrace == null) {
                    serverProgress.setActionState(ActionState.WAITING);
                    serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.STOP_OK);
                } else {
                    serverProgress.setActionState(ActionState.FINISHED_ERROR);
                }
            }
        } else {
            if (serverProgress.getProgress() == 50) {
                try {
                    serverAction = ServerAction.getServerAction(serverProgress.getServer());
                    serverAction.maintain();
                } catch (Exception e) {
                    stackTrace = new ByteArrayOutputStream();
                    e.printStackTrace(new PrintStream(stackTrace));
                }
                if (serverAction != null) {
                    serverProgress.appendToLog(serverAction.flushLog());
                }
                if (stackTrace != null) {
                    serverProgress.appendToLog(stackTrace.toString());
                }

                serverProgress.setProgress(66);

                if (stackTrace == null) {
                    serverProgress.setActionState(ActionState.WAITING);
                    serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.MAINTAIN_OK);
                } else {
                    serverProgress.setActionState(ActionState.FINISHED_ERROR);
                }
            } else if (serverProgress.getProgress() == 66) {
                try {
                    serverAction = ServerAction.getServerAction(serverProgress.getServer());
                    serverAction.maintain();
                } catch (Exception e) {
                    stackTrace = new ByteArrayOutputStream();
                    e.printStackTrace(new PrintStream(stackTrace));
                }
                if (serverAction != null) {
                    serverProgress.appendToLog(serverAction.flushLog());
                }
                if (stackTrace != null) {
                    serverProgress.appendToLog(stackTrace.toString());
                }

                serverProgress.setProgress(50);

                if (stackTrace == null) {
                    serverProgress.setActionState(ActionState.WAITING);
                    serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.STOP_OK);
                } else {
                    serverProgress.setActionState(ActionState.FINISHED_ERROR);
                }
            }
        }
    }

    /**
     * this function allows to stop a server
     * 
     * @param serverProgress - the serverProgress server that will be stopped
     */
    private void stop(final ServerProgressBean serverProgress, final boolean clustered) {
        OutputStream stackTrace = null;
        ServerAction serverAction = null;

        if (clustered) {
            if (serverProgress.getProgress() == 20) {
                try {
                    serverAction = ServerAction.getServerAction(serverProgress.getServer());
                    serverAction.stop();
                } catch (Exception e) {
                    stackTrace = new ByteArrayOutputStream();
                    e.printStackTrace(new PrintStream(stackTrace));
                }
                if (serverAction != null) {
                    serverProgress.appendToLog(serverAction.flushLog());
                }
                if (stackTrace != null) {
                    serverProgress.appendToLog(stackTrace.toString());
                }

                serverProgress.setProgress(40);

                if (stackTrace == null) {
                    serverProgress.setActionState(ActionState.WAITING);
                    serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.STOP_OK);
                } else {
                    serverProgress.setActionState(ActionState.FINISHED_ERROR);
                }
            } else if (serverProgress.getProgress() == 80) {
                try {
                    if (serverProgress.getServer().getTarget() == null) {
                        serverAction = ServerAction.getServerAction(serverProgress.getServer());
                    } else {
                        serverAction = ServerAction.getServerAction(serverProgress.getServer().getTarget());
                    }
                    serverAction.stop();
                } catch (Exception e) {
                    stackTrace = new ByteArrayOutputStream();
                    e.printStackTrace(new PrintStream(stackTrace));
                }
                if (serverAction != null) {
                    serverProgress.appendToLog(serverAction.flushLog());
                }
                if (stackTrace != null) {
                    serverProgress.appendToLog(stackTrace.toString());
                }
                serverProgress.setProgress(60);

                if (stackTrace == null) {
                    serverProgress.setActionState(ActionState.WAITING);
                    serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.MAINTAIN_OK);
                } else {
                    serverProgress.setActionState(ActionState.FINISHED_ERROR);
                }
            }
        } else {
            if (serverProgress.getProgress() == 33) {
                try {
                    serverAction = ServerAction.getServerAction(serverProgress.getServer());
                    serverAction.stop();
                } catch (Exception e) {
                    stackTrace = new ByteArrayOutputStream();
                    e.printStackTrace(new PrintStream(stackTrace));
                }
                if (serverAction != null) {
                    serverProgress.appendToLog(serverAction.flushLog());
                }
                if (stackTrace != null) {
                    serverProgress.appendToLog(stackTrace.toString());
                }

                serverProgress.setProgress(50);

                if (stackTrace == null) {
                    serverProgress.setActionState(ActionState.WAITING);
                    serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.STOP_OK);
                } else {
                    serverProgress.setActionState(ActionState.FINISHED_ERROR);
                }
            } else if (serverProgress.getProgress() == 83) {
                try {
                    if (serverProgress.getServer().getTarget() == null) {
                        serverAction = ServerAction.getServerAction(serverProgress.getServer());
                    } else {
                        serverAction = ServerAction.getServerAction(serverProgress.getServer().getTarget());
                    }
                    serverAction.stop();
                } catch (Exception e) {
                    stackTrace = new ByteArrayOutputStream();
                    e.printStackTrace(new PrintStream(stackTrace));
                }
                if (serverAction != null) {
                    serverProgress.appendToLog(serverAction.flushLog());
                }
                if (stackTrace != null) {
                    serverProgress.appendToLog(stackTrace.toString());
                }
                serverProgress.setProgress(66);

                if (stackTrace == null) {
                    serverProgress.setActionState(ActionState.WAITING);
                    serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.MAINTAIN_OK);
                } else {
                    serverProgress.setActionState(ActionState.FINISHED_ERROR);
                }
            }
        }
    }

    /**
     * Enable or disable or enable all applications in this server.
     * 
     * @param serverProgress - the serverProgress server on which the
     *        applications will be disabled
     * @param enable true to enable all applications, false to disable all
     *        applications.
     */
    private void enableOrDisableApplications(final ServerProgressBean serverProgress, final boolean enable) {
        OutputStream stackTrace = null;
        ServerAction serverAction = null;

        boolean successful = true;
        try {
            serverAction = ServerAction.getServerAction(serverProgress.getServer());
            successful = serverAction.enableOrDisableApplications(enable);
        } catch (Exception e) {
            stackTrace = new ByteArrayOutputStream();
            e.printStackTrace(new PrintStream(stackTrace));
        }
        if (serverAction != null) {
            serverProgress.appendToLog(serverAction.flushLog());
        }
        if (stackTrace != null) {
            serverProgress.appendToLog(stackTrace.toString());
        }

        if (enable) {
            serverProgress.setProgress(0);
        } else {
            serverProgress.setProgress(16);
        }

        if (stackTrace == null) {
            if (successful) {
                serverProgress.setActionState(ActionState.WAITING);
            } else {
                serverProgress.setActionState(ActionState.FINISHED_ERROR);
            }

            if (enable) {
                serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.INITIAL);
            } else {
                serverProgress.setServerProgressState(ServerProgressBean.ServerProgressState.DISABLE_APPLICATIONS_OK);
            }
        } else {
            serverProgress.setActionState(ActionState.FINISHED_ERROR);
        }
    }

    /**
     * this function allows to start a worker
     * 
     * @param workerProgress - the workerProgress of the worker that will be
     *        started
     */
    private void startWorker(final WorkerProgressBean workerProgress, final boolean clustered) {
        OutputStream stackTrace = null;
        WorkerAction workerAction = null;

        if (clustered) {
            if (workerProgress.getProgress() == 80) {
                try {
                    workerAction = WorkerAction.getWorkerAction(workerProgress.getWorker());
                    workerAction.activate();
                } catch (Exception e) {
                    stackTrace = new ByteArrayOutputStream();
                    e.printStackTrace(new PrintStream(stackTrace));
                }
                if (workerAction != null) {
                    workerProgress.appendToLog(workerAction.flushLog());
                }
                if (stackTrace != null) {
                    workerProgress.appendToLog(stackTrace.toString());
                }

                workerProgress.setProgress(100);

                if (stackTrace == null) {
                    workerProgress.setActionState(ActionState.WAITING);
                    workerProgress.setWorkerProgressState(WorkerProgressBean.WorkerProgressState.START_OK);
                } else {
                    workerProgress.setActionState(ActionState.FINISHED_ERROR);
                }
            } else if (workerProgress.getProgress() == 20) {
                try {
                    workerAction = WorkerAction.getWorkerAction(workerProgress.getWorker());
                    workerAction.activate();
                } catch (Exception e) {
                    stackTrace = new ByteArrayOutputStream();
                    e.printStackTrace(new PrintStream(stackTrace));
                }
                if (workerAction != null) {
                    workerProgress.appendToLog(workerAction.flushLog());
                }
                if (stackTrace != null) {
                    workerProgress.appendToLog(stackTrace.toString());
                }
                workerProgress.setProgress(0);

                if (stackTrace == null) {
                    workerProgress.setActionState(ActionState.WAITING);
                    workerProgress.setWorkerProgressState(WorkerProgressBean.WorkerProgressState.INITIAL);
                } else {
                    workerProgress.setActionState(ActionState.FINISHED_ERROR);
                }
            }
        } else {
            if (workerProgress.getProgress() == 83) {

                try {
                    workerAction = WorkerAction.getWorkerAction(workerProgress.getWorker());
                    workerAction.activate();
                } catch (Exception e) {
                    stackTrace = new ByteArrayOutputStream();
                    e.printStackTrace(new PrintStream(stackTrace));
                }
                if (workerAction != null) {
                    workerProgress.appendToLog(workerAction.flushLog());
                }
                if (stackTrace != null) {
                    workerProgress.appendToLog(stackTrace.toString());
                }

                workerProgress.setProgress(100);

                if (stackTrace == null) {
                    workerProgress.setActionState(ActionState.WAITING);
                    workerProgress.setWorkerProgressState(WorkerProgressBean.WorkerProgressState.START_OK);
                } else {
                    workerProgress.setActionState(ActionState.FINISHED_ERROR);
                }
            } else if (workerProgress.getProgress() == 33) {
                try {
                    workerAction = WorkerAction.getWorkerAction(workerProgress.getWorker());
                    workerAction.activate();
                } catch (Exception e) {
                    stackTrace = new ByteArrayOutputStream();
                    e.printStackTrace(new PrintStream(stackTrace));
                }
                if (workerAction != null) {
                    workerProgress.appendToLog(workerAction.flushLog());
                }
                if (stackTrace != null) {
                    workerProgress.appendToLog(stackTrace.toString());
                }
                workerProgress.setProgress(16);

                if (stackTrace == null) {
                    workerProgress.setActionState(ActionState.WAITING);
                    workerProgress.setWorkerProgressState(WorkerProgressBean.WorkerProgressState.INITIAL);
                } else {
                    workerProgress.setActionState(ActionState.FINISHED_ERROR);
                }
            }
        }
    }

    /**
     * this function allows to stop a worker
     * 
     * @param workerProgress - the workerProgress of the worker that will be
     *        stopped
     */
    private void stopWorker(final WorkerProgressBean workerProgress, final boolean clustered) {
        OutputStream stackTrace = null;
        WorkerAction workerAction = null;
        if (clustered) {
            if (workerProgress.getProgress() == 0) {
                try {
                    workerAction = WorkerAction.getWorkerAction(workerProgress.getWorker());
                    workerAction.stop();
                } catch (Exception e) {
                    stackTrace = new ByteArrayOutputStream();
                    e.printStackTrace(new PrintStream(stackTrace));
                }
                if (workerAction != null) {
                    workerProgress.appendToLog(workerAction.flushLog());
                }
                if (stackTrace != null) {
                    workerProgress.appendToLog(stackTrace.toString());
                }

                workerProgress.setProgress(20);

                if (stackTrace == null) {
                    workerProgress.setActionState(ActionState.WAITING);
                    workerProgress.setWorkerProgressState(WorkerProgressBean.WorkerProgressState.STOP_OK);
                } else {
                    workerProgress.setActionState(ActionState.FINISHED_ERROR);
                }
            } else if (workerProgress.getProgress() == 100) {
                try {
                    workerAction = WorkerAction.getWorkerAction(workerProgress.getWorker());
                    workerAction.stop();
                } catch (Exception e) {
                    stackTrace = new ByteArrayOutputStream();
                    e.printStackTrace(new PrintStream(stackTrace));
                }
                if (workerAction != null) {
                    workerProgress.appendToLog(workerAction.flushLog());
                }
                if (stackTrace != null) {
                    workerProgress.appendToLog(stackTrace.toString());
                }

                workerProgress.setProgress(80);

                if (stackTrace == null) {
                    workerProgress.setActionState(ActionState.WAITING);
                    workerProgress.setWorkerProgressState(WorkerProgressBean.WorkerProgressState.STOP_OK);
                } else {
                    workerProgress.setActionState(ActionState.FINISHED_ERROR);
                }
            }
        } else {
            if (workerProgress.getProgress() == 16) {
                try {
                    workerAction = WorkerAction.getWorkerAction(workerProgress.getWorker());
                    workerAction.stop();
                } catch (Exception e) {
                    stackTrace = new ByteArrayOutputStream();
                    e.printStackTrace(new PrintStream(stackTrace));
                }
                if (workerAction != null) {
                    workerProgress.appendToLog(workerAction.flushLog());
                }
                if (stackTrace != null) {
                    workerProgress.appendToLog(stackTrace.toString());
                }

                workerProgress.setProgress(33);

                if (stackTrace == null) {
                    workerProgress.setActionState(ActionState.WAITING);
                    workerProgress.setWorkerProgressState(WorkerProgressBean.WorkerProgressState.STOP_OK);
                } else {
                    workerProgress.setActionState(ActionState.FINISHED_ERROR);
                }
            } else if (workerProgress.getProgress() == 100) {
                try {
                    workerAction = WorkerAction.getWorkerAction(workerProgress.getWorker());
                    workerAction.stop();
                } catch (Exception e) {
                    stackTrace = new ByteArrayOutputStream();
                    e.printStackTrace(new PrintStream(stackTrace));
                }
                if (workerAction != null) {
                    workerProgress.appendToLog(workerAction.flushLog());
                }
                if (stackTrace != null) {
                    workerProgress.appendToLog(stackTrace.toString());
                }

                workerProgress.setProgress(83);

                if (stackTrace == null) {
                    workerProgress.setActionState(ActionState.WAITING);
                    workerProgress.setWorkerProgressState(WorkerProgressBean.WorkerProgressState.STOP_OK);
                } else {
                    workerProgress.setActionState(ActionState.FINISHED_ERROR);
                }
            }
        }
    }
}
