/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Copyright (C) 2008-2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ModJKWorkerAction.java 3529 2009-04-23 11:49:00Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.service.action;

import java.util.Iterator;

import org.apache.jk.status.JkBalancer;
import org.apache.jk.status.JkBalancerMember;
import org.ow2.jasmine.jadort.api.entities.topology.WorkerBean;
import org.ow2.jasmine.jadort.service.action.modJK.JkConnection;
import org.ow2.jasmine.jadort.service.action.modJK.JkConnectionException;

/**
 * Dummy worker action, only does internal calls.
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
public class ModJKWorkerAction extends WorkerAction {

    private static final String STATE_ACTIVE = "ACT";

    private static final String STATE_DISABLED = "DIS";

    private static final String STATE_STOPPED = "STP";

    private String name;

    private String balancerName;

    private String balancerMemberName;

    private JkConnection jkc;

    private JkBalancer balancer;

    private JkBalancerMember balancerMember;

    protected ModJKWorkerAction(final WorkerBean worker) {
        this.name = worker.getName();
        this.balancerName = worker.getWorkerName();
        this.balancerMemberName = worker.getServerName();
        this.jkc = new JkConnection(worker.getConnector().getConnectorUrl(), worker.getConnector().getUsername(), worker
            .getConnector().getPassword());

        this.appendToLog("Created ModJKWorkerAction for worker '" + this.name + "'");
    }

    @SuppressWarnings("unchecked")
    protected void refreshBalancerAndMember() throws JkConnectionException {
        this.balancer = null;
        this.balancerMember = null;

        Iterator balancers = this.jkc.readStatus_Online().getBalancers().iterator();
        while (balancers.hasNext()) {
            JkBalancer balancer = (JkBalancer) balancers.next();
            if (balancer.getName().equals(this.balancerName)) {
                Iterator balancerMembers = balancer.getBalancerMembers().iterator();
                while (balancerMembers.hasNext()) {
                    JkBalancerMember balancerMember = (JkBalancerMember) balancerMembers.next();
                    if (balancerMember.getName().equals(this.balancerMemberName)) {
                        this.balancer = balancer;
                        this.balancerMember = balancerMember;
                        return;
                    }
                }
                throw new IllegalArgumentException("Cannot find balancer member named \"" + this.balancerMemberName + "\"");
            }
        }

        throw new IllegalArgumentException("Cannot find balancer named \"" + this.balancerName + "\"");
    }

    @Override
    public void activate() throws Exception {
        this.appendToLog("Starting worker '" + this.name + "'");

        this.refreshBalancerAndMember();
        this.balancerMember.setActivation(ModJKWorkerAction.STATE_ACTIVE);
        this.jkc.updateBalancerMember(this.balancer, this.balancerMember);

        this.appendToLog("Worker '" + this.name + "' has been succesuffully started");
    }

    @Override
    public void disable() throws Exception {
        this.appendToLog("Disabling worker '" + this.name + "'");

        this.refreshBalancerAndMember();
        this.balancerMember.setActivation(ModJKWorkerAction.STATE_DISABLED);
        this.jkc.updateBalancerMember(this.balancer, this.balancerMember);

        this.appendToLog("Worker '" + this.name + "' has  been succesuffully disabled");
    }

    @Override
    public void stop() throws Exception {
        this.appendToLog("Stopping worker '" + this.name + "'");

        this.refreshBalancerAndMember();
        this.balancerMember.setActivation(ModJKWorkerAction.STATE_STOPPED);
        this.jkc.updateBalancerMember(this.balancer, this.balancerMember);

        this.appendToLog("Worker '" + this.name + "' has  been succesuffully stopped");
    }

    @Override
    public State getState() throws Exception {
        State state;
        this.refreshBalancerAndMember();
        String activation = this.balancerMember.getActivation();
        if (ModJKWorkerAction.STATE_ACTIVE.equals(activation)) {
            state = State.ACTIVE;
        } else if (ModJKWorkerAction.STATE_DISABLED.equals(activation)) {
            state = State.DISABLED;
        } else if (ModJKWorkerAction.STATE_STOPPED.equals(activation)) {
            state = State.STOPPED;
        } else {
            throw new IllegalStateException("Unknown activation state: " + activation);
        }

        return state;
    }
}
