/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Copyright (C) 2008-2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: VMMAction.java 3616 2009-05-05 08:29:09Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.service.action;

import java.lang.reflect.Constructor;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import org.ow2.jasmine.jadort.api.entities.deployment.VMImageBean;
import org.ow2.jasmine.jadort.api.entities.topology.VMBean;
import org.ow2.jasmine.vmm.api.VirtualMachineMXBean.PowerState;

/**
 * Action for a VM.
 * 
 * @author Malek Chahine
 * @author S. Ali Tokmen
 */
public abstract class VMMAction extends AbstractJMXAction {

    /**
     * States of virtual machine hosts.
     */
    public static enum VMState {
        RUNNING, HALTED
    }

    private static Map<String, VMMAction> pool = new Hashtable<String, VMMAction>();

    public static VMMAction getVMMAction(final VMBean vm) {
        synchronized (VMMAction.pool) {
            String key = VMMAction.getKey(vm);
            VMMAction vmmAction = VMMAction.pool.get(key);
            if (vmmAction == null) {
                vmmAction = VMMAction.newInstance(vm);
                VMMAction.pool.put(key, vmmAction);
            }
            return vmmAction;
        }
    }

    private static String getKey(final VMBean vm) {
        StringBuilder sb = new StringBuilder();
        sb.append(vm.getConnector().getConnectorUrl());
        if (vm.getConnector().getUsername() != null && vm.getConnector().getPassword() != null) {
            sb.append(vm.getConnector().getUsername());
            sb.append(vm.getConnector().getPassword());
        }
        return sb.toString();
    }

    protected static VMMAction newInstance(final VMBean vm) {
        try {
            Constructor<? extends VMMAction> constructor;
            constructor = VMMAction.getConstructor(JasmineVMMAction.class);
            return constructor.newInstance(vm);
        } catch (Exception e) {
            throw new IllegalArgumentException("Failed creating the VM action", e);
        }
    }

    @SuppressWarnings("unchecked")
    protected static Constructor<? extends VMMAction> getConstructor(final Class<? extends VMMAction> actionType)
        throws Exception {
        Constructor[] constructors = actionType.getDeclaredConstructors();
        for (Constructor<? extends VMMAction> constructor : constructors) {
            Class<?>[] params = constructor.getParameterTypes();
            if (params.length == 1 && params[0] == VMBean.class) {
                return constructor;
            }
        }
        return null;
    }

    /**
     * Gets the full VM name of a <b>RUNNING</b> VM.
     * 
     * @param vmName Name of the VM host to get the full VM name for.
     * @return Full VM name. That might be equal to the vmName passed as
     *         argument.
     * @throws Exception If any exception occurs.
     */
    public abstract String getFullVMName(final String vmName) throws Exception;

    /**
     * Sends the "start" signal to a VM host.
     * 
     * @param vmName Name of the VM host to start.
     * @throws Exception If any exception occurs.
     */
    public abstract void startVM(final String vmName) throws Exception;

    /**
     * Creates the VM with the given name and deploys the given image on it.
     * 
     * @param copyVM Name of the VM host to copy properties (disk size, memory,
     *        etc.) from.
     * @param newVMName Name of the new VM host to create.
     * @param vmImageUUID UUID of the VM image to deploy.
     * @throws Exception If any exception occurs.
     */
    public abstract void deployImageOnVM(final String copyVM, final String newVMName, final String vmImageUUID)
        throws Exception;

    /**
     * Destroys a VM host.
     * 
     * @param vmName Name of the VM host to destroy.
     * @throws Exception If any exception occurs.
     */
    public abstract void destroyVM(final String vmName) throws Exception;

    /**
     * Sends the "stop" signal to a VM host.
     * 
     * @param vmName Name of the VM host to stop.
     * @throws Exception If any exception occurs.
     */
    public abstract void stopVM(final String vmName) throws Exception;

    /**
     * Gets the state of a VM host.
     * 
     * @param vmName Name of the VM host to get the state of.
     * @throws Exception If any exception occurs.
     */
    public abstract PowerState getVMState(final String vmName) throws Exception;

    /**
     * Wait for the given VM host to reach a given state.
     * 
     * @param vmName Name of the VM host to get the state of.
     * @param state State to wait for.
     * @param timeout Number of seconds after which to throw.
     * @throws Exception If any exception occurs.
     */
    public abstract void waitForVMState(final String vmName, final PowerState state, final long timeout) throws Exception;

    /**
     * Gets the list of VM images on the current host's image repository.
     * 
     * @return List of VM images on the current host's image repository.
     * @throws Exception If any exception occurs.
     */
    public abstract List<VMImageBean> getVMImages() throws Exception;

}
