/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Copyright (C) 2008-2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: WorkerAction.java 3676 2009-05-11 17:41:17Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.service.action;

import java.lang.reflect.Constructor;
import java.util.Hashtable;
import java.util.Map;

import org.ow2.jasmine.jadort.api.entities.topology.WorkerBean;
import org.ow2.jasmine.jadort.api.entities.topology.WorkerBean.Type;

/**
 * Action for a worker
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
public abstract class WorkerAction extends AbstractAction {

    public enum State {
        ACTIVE, DISABLED, STOPPED
    };

    private static Map<String, WorkerAction> pool = new Hashtable<String, WorkerAction>();

    public static WorkerAction getWorkerAction(final WorkerBean worker) {
        synchronized (WorkerAction.pool) {
            String key = WorkerAction.getKey(worker);
            WorkerAction workerAction = WorkerAction.pool.get(key);
            if (workerAction == null) {
                workerAction = WorkerAction.newInstance(worker);
                WorkerAction.pool.put(key, workerAction);
            }
            return workerAction;
        }
    }

    private static String getKey(final WorkerBean worker) {
        StringBuilder sb = new StringBuilder();
        sb.append(worker.getName());
        sb.append(worker.getType());
        sb.append(worker.getConnector().getConnectorUrl());
        if (worker.getConnector().getUsername() != null && worker.getConnector().getPassword() != null) {
            sb.append(worker.getConnector().getUsername());
            sb.append(worker.getConnector().getPassword());
        }
        return sb.toString();
    }

    protected static WorkerAction newInstance(final WorkerBean worker) {
        try {
            Constructor<? extends WorkerAction> constructor;

            if (worker.getType().equals(Type.MOD_JK)) {
                constructor = WorkerAction.getConstructor(ModJKWorkerAction.class);
            } else if (worker.getType().equals(Type.MOD_PROXY_BALANCER)) {
                constructor = WorkerAction.getConstructor(ModProxyBalancerWorkerAction.class);
            } else if (worker.getType().equals(Type.DUMMY)) {
                constructor = WorkerAction.getConstructor(DummyWorkerAction.class);
            } else {
                throw new IllegalArgumentException("Unknown worker type :" + worker.getType());
            }

            return constructor.newInstance(worker);
        } catch (Exception e) {
            throw new IllegalArgumentException("Failed creating the worker action", e);
        }
    }

    @SuppressWarnings("unchecked")
    protected static Constructor<? extends WorkerAction> getConstructor(final Class<? extends WorkerAction> actionType)
        throws Exception {
        Constructor[] constructors = actionType.getDeclaredConstructors();
        for (Constructor<? extends WorkerAction> constructor : constructors) {
            Class<?>[] params = constructor.getParameterTypes();
            if (params.length == 1 && params[0] == WorkerBean.class) {
                return constructor;
            }
        }
        return null;
    }

    /**
     * Enable worker. When a worker is enabled, it redirects requests to the
     * target server.<br>
     * <br>
     * To disable a worker, call {@link WorkerAction#disable()} .
     */
    public abstract void enable() throws Exception;

    /**
     * Disable worker. When a worker is disabled, it redirects requests to the
     * server that it links to if the client has a valid session on the target
     * server.<br>
     * <br>
     * To enable a disabled worker, call {@link WorkerAction#enable()} .
     */
    public abstract void disable() throws Exception;

    /**
     * @return The current worker state.
     */
    public abstract State getState() throws Exception;
}
