/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Copyright (C) 2008-2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JkConnection.java 2965 2009-01-12 12:14:51Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.service.action.modJK;

import org.apache.jk.status.JkBalancer;
import org.apache.jk.status.JkBalancerMember;
import org.apache.jk.status.JkStatus;

/**
 * Connection to the mod_jk manager.
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
public final class JkConnection {

    /**
     * Update accessor.
     */
    private JkUpdateAccessor accessor = new JkUpdateAccessor();

    /**
     * mod_jk status cache.
     */
    private JkStatus status = null;

    /**
     * mod_jk manager URL.
     */
    private String urlJkStatus = null;

    /**
     * User name for the mod_jk manager URL.
     */
    private String username = null;

    /**
     * Password for the mod_jk manager URL.
     */
    private String password = null;

    /**
     * Create a mod_jk connection without any password protection.
     * 
     * @param urlJkStatus The mod_jk manager's URL.
     */
    public JkConnection(final String urlJkStatus) {
        this(urlJkStatus, null, null);
    }

    /**
     * Create a mod_jk connection.
     * 
     * @param urlJkStatus The mod_jk manager's URL.
     * @param username User name to use, null not to use any.
     * @param password Password to use, null not to use any.
     */
    public JkConnection(final String urlJkStatus, final String username, final String password) {
        if (!urlJkStatus.endsWith("/")) {
            this.urlJkStatus = urlJkStatus + '/';
        } else {
            this.urlJkStatus = urlJkStatus;
        }
        this.username = username;
        this.password = password;
    }

    /**
     * Gets the latest mod_jk status.
     * 
     * @throws JkConnectionException If exception querying the accessor.
     */
    private void getJkStatus() throws JkConnectionException {
        try {
            this.status = this.accessor.status(this.urlJkStatus, this.username, this.password);
            if (this.status.getResult().getType().equals("OK")
                && this.status.getResult().getMessage().equals("Action finished")) {
                // Everything is OK
            } else {
                throw new JkConnectionException("Invalid status: codeType=" + this.status.getResult().getType() + ", message="
                    + this.status.getResult().getMessage());
            }
        } catch (Exception e) {
            throw new JkConnectionException("Exception querying the accessor", e);
        }
    }

    /**
     * @return The last obtained mod_jk status object (cached).
     */
    public JkStatus readStatus_Offline() {
        return this.status;
    }

    /**
     * @return The current mod_jk status object (read from server).
     */
    public JkStatus readStatus_Online() throws JkConnectionException {
        this.getJkStatus();
        return this.readStatus_Offline();
    }

    /**
     * @return The last obtained web server software (cached).
     */
    public String readWebServerSoftware_Offline() {
        if (null != this.status) {
            return this.status.getSoftware().getWeb_server();
        } else {
            return null;
        }
    }

    /**
     * @return The current web server software (read from server).
     */
    public String readWebServerSoftware_Online() throws JkConnectionException {
        this.getJkStatus();
        return this.readWebServerSoftware_Offline();
    }

    /**
     * @return The last obtained mod_jk version (cached).
     */
    public String readJkVersion_Offline() {
        if (null != this.status) {
            return this.status.getSoftware().getJk_version();
        } else {
            return null;
        }
    }

    /**
     * @return The current mod_jk version (read from server).
     */
    public String readJkVersion_Online() throws JkConnectionException {
        this.getJkStatus();
        return this.readJkVersion_Offline();
    }

    /**
     * Updates the balancer.
     * 
     * @param pJkBalancer Object describing new mod_jk configuration.
     * @throws JkConnectionException If exception querying the balancer.
     */
    public void updateBalancer(final JkBalancer pJkBalancer) throws JkConnectionException {
        try {
            this.status = this.accessor.update(this.urlJkStatus, this.username, this.password, pJkBalancer);
            if (this.status.getResult().getType().equals("OK")
                && this.status.getResult().getMessage().equals("Action finished")) {
                // Everything is OK
            } else {
                throw new JkConnectionException("Invalid status: codeType=" + this.status.getResult().getType() + ", message="
                    + this.status.getResult().getMessage());
            }
        } catch (Exception e) {
            throw new JkConnectionException("Exception querying the balancer", e);
        }
    }

    /**
     * Updates the balancer member.
     * 
     * @param pJkBalancer Object describing new mod_jk configuration.
     * @param pJkBalancerMember Object describing new mod_jk configuration.
     * @throws JkConnectionException If exception querying the balancer.
     */
    public void updateBalancerMember(final JkBalancer pJkBalancer, final JkBalancerMember pJkBalancerMember)
        throws JkConnectionException {
        try {
            this.status = this.accessor.update(this.urlJkStatus, this.username, this.password, pJkBalancer, pJkBalancerMember);
            if (this.status.getResult().getType().equals("OK")
                && this.status.getResult().getMessage().equals("Action finished")) {
                // Everything is OK
            } else {
                throw new JkConnectionException("Invalid status: codeType=" + this.status.getResult().getType() + ", message="
                    + this.status.getResult().getMessage());
            }
        } catch (Exception e) {
            throw new JkConnectionException("Exception querying the balancer", e);
        }
    }
}
