/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Copyright (C) 2008-2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: IJadortService.java 3500 2009-04-22 13:57:09Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.api;

import java.io.File;
import java.io.Serializable;
import java.net.URL;
import java.util.List;

import org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean;
import org.ow2.jasmine.jadort.api.entities.deployment.VMImageBean;
import org.ow2.jasmine.jadort.api.entities.topology.GroupBean;
import org.ow2.jasmine.jadort.api.entities.topology.ServerBean;
import org.ow2.jasmine.jadort.api.entities.topology.WorkerBean;

/**
 * JaDOrT service's EJB3 interface. Gives direct access to the JaDOrT engine.
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
public interface IJadortService {

    /**
     * The JNDI name of the JaDOrT Stateful EJB3 bean.
     */
    static final String EJB_JNDI_NAME = "ow2.jasmine.jadort.ServiceSFB";

    /**
     * Operation types.<br>
     * <br>
     * We don't use an enum here to preserve compatibility with IIOP runtimes on
     * Java EE 5. You can check out the bug report and comments on
     * http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=6277781 for details.
     */
    public static final class OperationType implements Serializable {

        private static final long serialVersionUID = -7508039622857282263L;

        public static final OperationType MIGRATE = new OperationType("MIGRATE");

        public static final OperationType MAINTAIN = new OperationType("MAINTAIN");

        private static final OperationType[] values = new OperationType[] {OperationType.MIGRATE, OperationType.MAINTAIN};

        private String operationType;

        /**
         * @param operationType OperationType.
         */
        private OperationType(final String operationType) {
            this.operationType = operationType;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean equals(final Object object) {
            if ((object != null) && (object instanceof OperationType)) {
                OperationType operationType = (OperationType) object;
                if (operationType.operationType.equals(this.operationType)) {
                    return true;
                }
            }

            return false;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public int hashCode() {
            return this.operationType.hashCode();
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String toString() {
            return this.operationType;
        }

        /**
         * @return All OperationType values.
         */
        public static OperationType[] values() {
            return OperationType.values;
        }

        /**
         * Get the OperationType for a string.
         * 
         * @param string OperationType string.
         * @return OperationType for that string.
         */
        public static OperationType valueOf(final String string) {
            for (OperationType operationType : OperationType.values) {
                if (operationType.operationType.equals(string)) {
                    return operationType;
                }
            }

            throw new IllegalArgumentException("Unknown value: " + string);
        }
    }

    /**
     * Returns the current Step of the current operation
     * 
     * @return The current step.
     */
    OperationStateBean.Step getCurrentStep() throws JadortServiceException;

    /**
     * Returns the list of operations
     * 
     * @return List of Operations.
     */
    List<OperationStateBean> getOperationsList() throws JadortServiceException;

    /**
     * Delete an Operation from the operation list
     * 
     * @param operation the operation to be deleted
     */
    void deleteOperation(final OperationStateBean operation) throws JadortServiceException;

    /**
     * Select an Operation from the operation list
     * 
     * @param operation the operation to be selected
     */
    void selectOperation(final OperationStateBean operation) throws JadortServiceException;

    /**
     * Create a new operation
     * 
     * @param newOperationName the name of the new operation that will be
     *        created
     */
    void createNewOperation(final String newOperationName) throws JadortServiceException;

    /**
     * Returns the current operation. You need to use this method in order to
     * get all details related to the progress of the current operation, the
     * list of groups, the list of applications, etc.
     * 
     * @return The current operation.
     */
    OperationStateBean getCurrentOperation() throws JadortServiceException;

    /**
     * goes to next step
     * 
     * @see IJadortService#canGoToNextStep()
     */
    void next() throws JadortServiceException;

    /**
     * Goes back to the previous step.
     * 
     * @see IJadortService#canGoToPreviousStep()
     */
    void previous() throws JadortServiceException;

    /**
     * Loads the topology from a .xml file.
     * 
     * @param xmlTopoFile the file path of the .xml file uploaded by the user,
     *        this file contains the topology description
     */
    void loadTopology(final File xmlTopoFile) throws JadortServiceException;

    /**
     * Select the group that JaDOrT will be maintaining or migrating
     * 
     * @param selectedGroup the group that JaDOrT will be maintaining or
     *        migrating
     */
    void selectGroup(final GroupBean selectedGroup) throws JadortServiceException;

    /**
     * Select the operation type
     * 
     * @param operationType the operation type
     */
    void selectOperationType(final OperationType operationType) throws JadortServiceException;

    /**
     * Select a VM image from the VM images list. If this method is called, the
     * selected VM image will be deployed on all maintained VMs. If you want to
     * deploy a different VM image on each server, don't call this method but go
     * to the next step directly.
     * 
     * @param selectedVMImage the VM image to be selected
     */
    void selectVMImage(final VMImageBean selectedVMImage) throws JadortServiceException;

    /**
     * Select a VM image from the VM images list for a given server.
     * 
     * @param selectedVMImage the VM image to be selected
     * @param server the server on which this image shall be deployed
     */
    void selectVMImage(final VMImageBean selectedVMImage, final ServerBean server) throws JadortServiceException;

    /**
     * Create a new application
     * 
     * @param url URL where to fetch the application. Can be local, remote
     *        (HTTP, FTP, ...)
     */
    void selectApplication(final URL url) throws JadortServiceException;

    /**
     * Select the servers to maintain
     * 
     * @param selectedServers the servers to maintain
     */
    void selectServers(List<ServerBean> selectedServers) throws JadortServiceException;

    /**
     * Restarts a ServerBean that's in an error.
     * 
     * @param server the server on which the task will be restarted
     */
    void restartServer(final ServerBean server) throws JadortServiceException;

    /**
     * Checks if the problem on a given ServerBean has been resolved.
     * 
     * @return true if the problem is resolved and it can be ignored safely,
     *         false if the problem still persists
     * @param server the server that will be checked
     */
    boolean checkServer(final ServerBean server) throws JadortServiceException;

    /**
     * Ignores the error on a given ServerBean.
     * 
     * @param server the server that will be ignored
     */
    void ignoreServer(final ServerBean server) throws JadortServiceException;

    /**
     * Aborts the task a ServerBean is currently doing.
     * 
     * @param server the server that will be aborted
     */
    void abortServer(final ServerBean server) throws JadortServiceException;

    /**
     * Restarts a WorkerBean that's in an error.
     * 
     * @param worker the worker on which the task will be restarted
     */
    void restartWorker(final WorkerBean worker) throws JadortServiceException;

    /**
     * Checks if the problem on a given WorkerBean has been resolved.
     * 
     * @return true if the problem is resolved and it can be ignored safely,
     *         false if the problem still persists
     * @param worker the worker that will be checked
     */
    boolean checkWorker(final WorkerBean worker) throws JadortServiceException;

    /**
     * Ignores the error on a given WorkerBean.
     * 
     * @param worker the worker that will be ignored
     */
    void ignoreWorker(final WorkerBean worker) throws JadortServiceException;

    /**
     * Aborts the task a WorkerBean is currently doing.
     * 
     * @param worker the worker that will be aborted
     */
    void abortWorker(final WorkerBean worker) throws JadortServiceException;

}
