/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Copyright (C) 2008-2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ServerProgressBean.java 3904 2009-06-09 13:08:00Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.api.entities.deployment;

import javax.persistence.Basic;
import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.Lob;
import javax.persistence.OneToOne;
import javax.persistence.Version;

import org.ow2.jasmine.jadort.api.IJadortService.OperationType;
import org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean.ActionState;
import org.ow2.jasmine.jadort.api.entities.topology.ServerBean;

/**
 * Bean that represents the progress of the operations a server.
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
@Entity(name = "JaDOrT_ServerProgressBean")
public class ServerProgressBean extends ProgressBean<ServerProgressState> {

    /**
     * Auto-generated serial version ID.
     */
    private static final long serialVersionUID = -2851071984012556925L;

    /**
     * Auto-generated identifier of the ServerProgressBean in the database.
     */
    @Id
    @GeneratedValue
    private Integer id;

    /**
     * Auto-generated version of the ServerProgressBean. Used internally for
     * checking integrity.
     */
    @Version
    @SuppressWarnings("unused")
    private Integer version;

    /**
     * The server associated with this ServerProgressBean.
     */
    // Cascade for all except CascadeType.REMOVE, since removing a
    // ServerProgressBean doesn't imply that the associated ServerBean
    // (created as part of a GroupBean) should be deleted!
    @OneToOne(cascade = {CascadeType.MERGE, CascadeType.PERSIST, CascadeType.REFRESH})
    private ServerBean server;

    /**
     * Deployment log
     */
    @Lob
    @Basic(fetch = FetchType.EAGER)
    private String log;

    /**
     * The old deployment item. Currently, it represents either:
     * <ul>
     * <li>If the operation type is {@link OperationType#MIGRATE}, it is the
     * name of the old application's path</li>
     * <li>If the operation type is {@link OperationType#MAINTAIN} with a VM
     * maintenance, it is the name of the old VM host's name</li>
     * </ul>
     */
    private String oldDeploymentItem;

    /**
     * The new deployment item. Currently, it represents either:
     * <ul>
     * <li>If the operation type is {@link OperationType#MIGRATE}, it is the
     * name of the new application's path</li>
     * <li>If the operation type is {@link OperationType#MAINTAIN} with a VM
     * maintenance, it is the name of the new VM host's name</li>
     * </ul>
     */
    private String newDeploymentItem;

    /**
     * The VM image that's currently being deployed.
     */
    @OneToOne(cascade = CascadeType.ALL)
    private VMImageBean vmImage;

    /**
     * Deployment progress state
     */
    private ServerProgressState serverProgressState;

    /**
     * Deployment state
     */
    private ActionState actionState;

    /**
     * Progress percent of the ServerProgressBean.
     */
    private int progressPercent;

    /**
     * Default ServerProgressBean constructor
     */
    public ServerProgressBean() {
        this.progressPercent = 0;
        this.actionState = ActionState.WAITING;
        this.serverProgressState = ServerProgressState.INITIAL;
    }

    /**
     * Construct a ServerProgressBean with its associated server
     * 
     * @param server The server associated the deployment
     */
    public ServerProgressBean(final ServerBean server) {
        this();
        this.server = server;
    }

    /**
     * Get the ServerProgressBean id
     * 
     * @return the ServerProgressBean id
     */
    public Integer getId() {
        return this.id;
    }

    /**
     * Set the ServerProgressBean id
     * 
     * @param id the ServerProgressBean id
     */
    public void setId(final Integer id) {
        this.id = id;
    }

    /**
     * Server getter
     * 
     * @return The associated server
     */
    public ServerBean getServer() {
        return this.server;
    }

    /**
     * Change the associated server
     * 
     * @param server New associated server
     */
    public void setServer(final ServerBean server) {
        this.server = server;
    }

    /**
     * Returns the name of the server associated with this ServerProgressBean
     * 
     * @return the name of the server associated with this ServerProgressBean
     */
    public String getServerName() {
        return this.server.getName();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getLog() {
        return this.log;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setLog(final String log) {
        this.log = log;
    }

    /**
     * @return The old deployment item. Currently, it represents either:
     *         <ul>
     *         <li>If the operation type is {@link OperationType#MIGRATE}, it is
     *         the name of the old application's path</li>
     *         <li>If the operation type is {@link OperationType#MAINTAIN} with
     *         a VM maintenance, it is the name of the old VM host's name</li>
     *         </ul>
     */
    public String getOldDeploymentItem() {
        return this.oldDeploymentItem;
    }

    /**
     * @param oldDeploymentItem The old deployment item. Currently, it
     *        represents either:
     *        <ul>
     *        <li>If the operation type is {@link OperationType#MIGRATE}, it is
     *        the name of the old application's path</li>
     *        <li>If the operation type is {@link OperationType#MAINTAIN} with a
     *        VM maintenance, it is the name of the old VM host's name</li>
     *        </ul>
     */
    public void setOldDeploymentItem(final String oldDeploymentItem) {
        this.oldDeploymentItem = oldDeploymentItem;
    }

    /**
     * @return The new deployment item. Currently, it represents either:
     *         <ul>
     *         <li>If the operation type is {@link OperationType#MIGRATE}, it is
     *         the name of the new application's path</li>
     *         <li>If the operation type is {@link OperationType#MAINTAIN} with
     *         a VM maintenance, it is the name of the new VM host's name</li>
     *         </ul>
     */
    public String getNewDeploymentItem() {
        return this.newDeploymentItem;
    }

    /**
     * @param newDeploymentItem The new deployment item. Currently, it
     *        represents either:
     *        <ul>
     *        <li>If the operation type is {@link OperationType#MIGRATE}, it is
     *        the name of the new application's path</li>
     *        <li>If the operation type is {@link OperationType#MAINTAIN} with a
     *        VM maintenance, it is the name of the new VM host's name</li>
     *        </ul>
     */
    public void setNewDeploymentItem(final String newDeploymentItem) {
        this.newDeploymentItem = newDeploymentItem;
    }

    /**
     * @return true if oldDeploymentItem is not null and different of
     *         newDeploymentItem, false otherwise.
     */
    public boolean hasOldDeploymentItem() {
        return this.oldDeploymentItem != null && !this.oldDeploymentItem.equals(this.newDeploymentItem);
    }

    /**
     * Get the VM Image to deploy
     * 
     * @return the VM Image to deploy
     */
    public VMImageBean getVmImage() {
        return this.vmImage;
    }

    /**
     * Set the VM Image to deploy
     * 
     * @param VMImage the VM Image to deploy
     */
    public void setVmImage(final VMImageBean vmImage) {
        this.vmImage = vmImage;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ServerProgressState getProgressState() {
        return this.serverProgressState;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setProgressState(final ServerProgressState serverProgressState) {
        this.serverProgressState = serverProgressState;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ActionState getActionState() {
        return this.actionState;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setActionState(final ActionState actionState) {
        this.actionState = actionState;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int getProgress() {
        return this.progressPercent;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setProgress(final int progressPercent) {
        if (progressPercent < 0 || progressPercent > 100) {
            throw new IllegalArgumentException("ProgressPercent is a percentage (between 0 and 100)");
        }

        this.progressPercent = progressPercent;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(this.getClass().getSimpleName());
        sb.append("[server='");
        sb.append(this.server);
        sb.append("']");
        return sb.toString();
    }

    /**
     * {@inheritDoc}
     * 
     * @return true if <code>obj</code> is a {@link ServerProgressBean} and the
     *         {@link ServerProgressBean#getId()} is the same for both objects,
     *         false otherwise.
     */
    @Override
    public boolean equals(final Object obj) {
        if (obj != null && obj instanceof ServerProgressBean) {
            ServerProgressBean other = (ServerProgressBean) obj;
            if (this.id != null && other.id != null) {
                return this.id.equals(other.id);
            }
        }

        // Else not same type or some parts are null
        return false;
    }

}
