/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Copyright (C) 2008-2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: WorkerBean.java 3485 2009-04-20 15:46:33Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.api.entities.topology;

import java.io.Serializable;

import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.OneToOne;

/**
 * Bean represents a worker. In the XSD, each instance of this object is a
 * Balancer.
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
@Entity(name = "JaDOrT_WorkerBean")
public class WorkerBean implements Serializable {

    /**
     * Worker type.<br>
     * <br>
     * We don't use an enum here to preserve compatibility with IIOP runtimes on
     * Java EE 5. You can check out the bug report and comments on
     * http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=6277781 for details.
     */
    public static final class Type implements Serializable {

        private static final long serialVersionUID = 8006861800369211346L;

        public static final Type MOD_JK = new Type("MOD_JK");

        public static final Type MOD_PROXY_BALANCER = new Type("MOD_PROXY_BALANCER");

        public static final Type DUMMY = new Type("DUMMY");

        private static final Type[] values = new Type[] {Type.MOD_JK, Type.MOD_PROXY_BALANCER, Type.DUMMY};

        private String type;

        /**
         * @param type Type.
         */
        private Type(final String type) {
            this.type = type;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public boolean equals(final Object object) {
            if ((object != null) && (object instanceof Type)) {
                Type type = (Type) object;
                if (type.type.equals(this.type)) {
                    return true;
                }
            }

            return false;
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public int hashCode() {
            return this.type.hashCode();
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public String toString() {
            return this.type;
        }

        /**
         * @return All Type values.
         */
        public static Type[] values() {
            return Type.values;
        }

        /**
         * Get the Type for a string.
         * 
         * @param string Type string.
         * @return Type for that string.
         */
        public static Type valueOf(final String string) {
            for (Type type : Type.values) {
                if (type.type.equals(string)) {
                    return type;
                }
            }

            throw new IllegalArgumentException("Unknown value: " + string);
        }
    }

    /**
     * Auto-generated serial version ID.
     */
    private static final long serialVersionUID = -8338670679543686581L;

    /**
     * Auto-generated identifier of the WorkerBean in the database.
     */
    @Id
    @GeneratedValue
    private Integer id;

    /**
     * Worker's name
     */
    private String name;

    /**
     * Server name associated with this worker
     */
    private String serverName;

    /**
     * Worker name associated with this worker
     */
    private String workerName;

    /**
     * Worker type
     */
    private Type type;

    /**
     * Worker connector
     */
    @OneToOne(cascade = CascadeType.ALL)
    private ConnectorBean connector;

    /**
     * Worker server.
     */
    @OneToOne(cascade = CascadeType.ALL)
    private ServerBean server;

    /**
     * Get the worker id
     * 
     * @return the worker id
     */
    public Integer getId() {
        return this.id;
    }

    /**
     * Set the worker id
     * 
     * @param id the worker id
     */
    public void setId(final Integer id) {
        this.id = id;
    }

    /**
     * Get the worker's name
     * 
     * @return the worker's name
     */
    public String getName() {
        return this.name;
    }

    /**
     * Set the worker's name
     * 
     * @param name the worker's name
     */
    public void setName(final String name) {
        this.name = name;
    }

    /**
     * Get the server name associated with this worker
     * 
     * @return the server name associated with this worker
     */
    public String getServerName() {
        return this.serverName;
    }

    /**
     * Set the server name associated with this worker
     * 
     * @param serverName the server name associated with this worker
     */
    public void setServerName(final String serverName) {
        this.serverName = serverName;
    }

    /**
     * Get the worker name associated with this worker
     * 
     * @return the worker name associated with this worker
     */
    public String getWorkerName() {
        return this.workerName;
    }

    /**
     * Set the worker name associated with this worker
     * 
     * @param workerName the worker name associated with this worker
     */
    public void setWorkerName(final String workerName) {
        this.workerName = workerName;
    }

    /**
     * Get the worker connector
     * 
     * @return the worker connector
     */
    public ConnectorBean getConnector() {
        return this.connector;
    }

    /**
     * Set the worker connector
     * 
     * @param connector the worker connector
     */
    public void setConnector(final ConnectorBean connector) {
        this.connector = connector;
    }

    /**
     * Get the worker server
     * 
     * @return the worker server
     */
    public ServerBean getServer() {
        return this.server;
    }

    /**
     * Set the worker server
     * 
     * @param server the worker server
     */
    public void setServer(final ServerBean server) {
        this.server = server;
    }

    /**
     * @return worker type
     */
    public Type getType() {
        return this.type;
    }

    /**
     * @param type worker type
     */
    public void setType(final Type type) {
        this.type = type;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(this.getClass().getSimpleName());
        sb.append("[name='");
        sb.append(this.name);
        sb.append("', type='");
        sb.append(this.type);
        sb.append("']");
        return sb.toString();
    }

    /**
     * {@inheritDoc}
     * 
     * @return true if <code>obj</code> is a {@link WorkerBean} and the
     *         {@link WorkerBean#getId()} is the same for both objects, false
     *         otherwise.
     */
    @Override
    public boolean equals(final Object obj) {
        if (obj != null && obj instanceof WorkerBean) {
            WorkerBean other = (WorkerBean) obj;
            if (this.id != null && other.id != null) {
                return this.id.equals(other.id);
            }
        }

        // Else not same type or some parts are null
        return false;
    }

}
