/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Copyright (C) 2008-2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ApplicationBeanTest.java 5645 2009-11-27 17:12:10Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.api.entities.deployment;

import java.io.File;
import java.util.UUID;

import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean.ApplicationType;
import org.ow2.util.archive.api.ArchiveException;
import org.ow2.util.ee.deploy.impl.helper.DeployableHelperException;

/**
 * Tests the ApplicationBean class.
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
public class ApplicationBeanTest {

    ApplicationBean application;

    /**
     * @throws java.lang.Exception
     */
    @Before
    public void setUp() throws Exception {
        this.application = new ApplicationBean();
    }

    /**
     * @throws java.lang.Exception
     */
    @After
    public void tearDown() throws Exception {
        this.application = null;
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean#ApplicationBean()}
     * .
     */
    @Test
    public void testApplicationBean() {
        Assert.assertNotNull(this.application);
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean#ApplicationBean(java.io.File)}
     * .
     */
    @Test
    public void testApplicationBeanWithValidFile() throws Exception {
        File file = new File("src/test/resources/application1.ear");

        ApplicationBean application = new ApplicationBean(file);
        Assert.assertNotNull(application);
        Assert.assertEquals(file, application.getFile());
        Assert.assertEquals(ApplicationType.JAVA_EE, application.getApplicationType());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean#ApplicationBean(java.io.File)}
     * .
     */
    @Test
    public void testApplicationBeanWithInvalidFile() throws Exception {
        File file = new File("src/test/resources/noApplication.war");

        try {
            new ApplicationBean(file);
            Assert.fail();
        } catch (DeployableHelperException e) {
            // That failure is expected
        } catch (ArchiveException e) {
            // That failure is expected
        }
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean#ApplicationBean(java.io.File)}
     * with a bundle.
     */
    @Test
    public void testApplicationBeanWithBundle() throws Exception {
        File file = new File("src/test/resources/ow2-util-jmx-api-1.0.18.jar");

        ApplicationBean application = new ApplicationBean(file);
        Assert.assertNotNull(application);
        Assert.assertEquals(file, application.getFile());
        Assert.assertEquals(ApplicationType.OSGI, application.getApplicationType());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean#ApplicationBean(java.lang.String)}
     * .
     */
    @Test
    public void testApplicationBeanName() {
        String name = this.getClass().getSimpleName() + '#' + UUID.randomUUID();

        ApplicationBean application = new ApplicationBean(name);
        Assert.assertNotNull(application);
        Assert.assertEquals(application.getName(), name);
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean#ApplicationBean(java.lang.String, java.lang.String)}
     * .
     */
    @Test
    public void testApplicationBeanVersion() {
        String name = this.getClass().getSimpleName() + '#' + UUID.randomUUID();
        String version = UUID.randomUUID().toString();

        ApplicationBean application = new ApplicationBean(name, version);
        Assert.assertNotNull(application);
        Assert.assertEquals(application.getName(), name);
        Assert.assertEquals(application.getVersion(), version);
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean#getId()}
     * .
     */
    @Test
    public void testGetSetId() {
        Integer id = (int) (Math.random() * 100);

        Assert.assertNull(this.application.getId());
        this.application.setId(id);
        Assert.assertEquals(this.application.getId(), id);
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean#getName()}
     * .
     */
    @Test
    public void testGetSetName() {
        String name = this.getClass().getSimpleName() + '#' + UUID.randomUUID();

        Assert.assertNull(this.application.getName());
        this.application.setName(name);
        Assert.assertNotNull(this.application.getName());
        Assert.assertEquals(name, this.application.getName());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean#getVersion()}
     * .
     */
    @Test
    public void testGetSetVersion() {
        String version = UUID.randomUUID().toString();

        Assert.assertNull(this.application.getVersion());
        this.application.setVersion(version);
        Assert.assertNotNull(this.application.getVersion());
        Assert.assertEquals(version, this.application.getVersion());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean#getApplicationType()}
     * .
     */
    @Test
    public void testGetSetApplicationType() {
        Assert.assertEquals(ApplicationType.JAVA_EE, this.application.getApplicationType());

        for (ApplicationType applicationType : ApplicationType.values()) {
            this.application.setApplicationType(applicationType);
            Assert.assertEquals(applicationType, this.application.getApplicationType());
        }
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean#getFile()}
     * .
     */
    @Test
    public void testGetSetFile() {
        File file = new File("File#" + UUID.randomUUID());

        Assert.assertNull(this.application.getFile());
        this.application.setFile(file);
        Assert.assertNotNull(this.application.getFile());
        Assert.assertEquals(file, this.application.getFile());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean#getPolicy()}
     * .
     */
    @Test
    public void testGetSetPolicy() {
        String policy = "Policy#" + UUID.randomUUID();

        Assert.assertNull(this.application.getPolicy());
        this.application.setPolicy(policy);
        Assert.assertNotNull(this.application.getPolicy());
        Assert.assertEquals(policy, this.application.getPolicy());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean#getState()}
     * .
     */
    @Test
    public void testGetSetState() {
        String state = "State#" + UUID.randomUUID();

        Assert.assertNull(this.application.getState());
        this.application.setState(state);
        Assert.assertNotNull(this.application.getState());
        Assert.assertEquals(state, this.application.getState());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean#equals(Object)}
     * .
     */
    @Test
    public void testEquals() {
        Integer id1 = (int) (Math.random() * 100);
        Integer id2 = (int) (Math.random() * 100);
        while (id1.equals(id2)) {
            id2 = (int) (Math.random() * 100);
        }

        ApplicationBean bean1 = new ApplicationBean();
        ApplicationBean bean2 = new ApplicationBean();

        // Both identifiers are null
        Assert.assertFalse(bean1.equals(bean2));

        // bean2.getId() is null
        bean1.setId(id1);
        Assert.assertFalse(bean1.equals(bean2));

        // bean1.getId() is null
        bean1.setId(null);
        bean2.setId(id1);
        Assert.assertFalse(bean1.equals(bean2));

        bean1.setId(id1);
        Assert.assertEquals(bean1, bean2);

        bean1.setId(id1);
        bean2.setId(id2);
        Assert.assertFalse(bean1.equals(bean2));

        bean1.setId(id2);
        bean2.setId(id2);
        Assert.assertEquals(bean1, bean2);
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean#toString()}
     * .
     */
    @Test
    public void testToString() {
        Assert.assertNotNull(this.application.toString());
        Assert.assertFalse(this.application.toString().contains(this.application.getClass().getName()));
    }
}
