/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Copyright (C) 2008-2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: OperationStateBeanTest.java 5667 2009-12-02 10:10:51Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.api.entities.deployment;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.ow2.jasmine.jadort.api.IJadortService.OperationType;
import org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean.Step;
import org.ow2.jasmine.jadort.api.entities.topology.GroupBean;
import org.ow2.jasmine.jadort.api.entities.topology.TopologyBean;

/**
 * Tests the OperationStateBean class.
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
public class OperationStateBeanTest {

    private OperationStateBean operation;

    /**
     * @throws java.lang.Exception
     */
    @Before
    public void setUp() throws Exception {
        this.operation = new OperationStateBean();
    }

    /**
     * @throws java.lang.Exception
     */
    @After
    public void tearDown() throws Exception {
        this.operation = null;
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#getId()}
     * .
     */
    @Test
    public void testGetSetId() {
        Integer id = (int) (Math.random() * 100);

        Assert.assertNull(this.operation.getId());
        this.operation.setId(id);
        Assert.assertEquals(this.operation.getId(), id);
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#getAimedServerProgressState()}
     * .
     */
    @Test
    public void testGetSetAimedServerProgressState() {
        Assert.assertEquals(ServerProgressState.INITIAL, this.operation.getAimedServerProgressState());

        for (ServerProgressState state : ServerProgressState.values()) {
            this.operation.setAimedServerProgressState(state);
            Assert.assertEquals(state, this.operation.getAimedServerProgressState());
        }
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#getAimedProgressPercent()}
     * .
     */
    @Test
    public void testGetSetAimedProgressPercent() {
        int initialProgress = this.operation.getAimedProgressPercent();
        Assert.assertEquals(0, initialProgress);

        int progress = 33;
        this.operation.setAimedProgressPercent(progress);
        Assert.assertEquals(progress, this.operation.getAimedProgressPercent());

        try {
            this.operation.setAimedProgressPercent(-1);
        } catch (IllegalArgumentException ignored) {
            // Ignored
        }
        Assert.assertEquals(progress, this.operation.getAimedProgressPercent());

        try {
            this.operation.setAimedProgressPercent(101);
        } catch (IllegalArgumentException ignored) {
            // Ignored
        }
        Assert.assertEquals(progress, this.operation.getAimedProgressPercent());

        int newProgress = 66;
        this.operation.setAimedProgressPercent(newProgress);
        Assert.assertEquals(newProgress, this.operation.getAimedProgressPercent());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#getAimedWorkerProgressState()}
     * .
     */
    @Test
    public void testGetSetAimedWorkerProgressState() {
        Assert.assertEquals(WorkerProgressState.INITIAL, this.operation.getAimedWorkerProgressState());

        for (WorkerProgressState state : WorkerProgressState.values()) {
            this.operation.setAimedWorkerProgressState(state);
            Assert.assertEquals(state, this.operation.getAimedWorkerProgressState());
        }
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#getCurrentStep()}
     * .
     */
    @Test
    public void testGetSetCurrentStep() {
        Assert.assertEquals(Step.INITIALIZE_TOPOLOGY, this.operation.getCurrentStep());

        for (Step step : Step.values()) {
            if (step.equals(Step.SELECT_OPERATION)) {
                try {
                    this.operation.setCurrentStep(step);
                    Assert.fail();
                } catch (IllegalArgumentException expected) {
                    // The invalid argument has been rejected as expected
                }
            } else {
                this.operation.setCurrentStep(step);
                Assert.assertEquals(step, this.operation.getCurrentStep());
            }
        }

        this.operation.setCurrentStep(null);
        Assert.assertNull(this.operation.getCurrentStep());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#getType()}
     * .
     */
    @Test
    public void testGetSetType() {
        Assert.assertNull(this.operation.getType());

        for (OperationType operationType : OperationType.values()) {
            this.operation.setType(operationType);
            Assert.assertEquals(operationType, this.operation.getType());
        }
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#getCanGoToNextStep()}
     * .
     */
    @Test
    public void testGetSetCanGoToNextStep() {
        Assert.assertFalse(this.operation.getCanGoToNextStep());

        this.operation.setCanGoToNextStep(true);
        Assert.assertTrue(this.operation.getCanGoToNextStep());

        this.operation.setCanGoToNextStep(false);
        Assert.assertFalse(this.operation.getCanGoToNextStep());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#getCanGoToPreviousStep()}
     * .
     */
    @Test
    public void testGetSetCanGoToPreviousStep() {
        Assert.assertFalse(this.operation.getCanGoToPreviousStep());

        this.operation.setCanGoToPreviousStep(true);
        Assert.assertTrue(this.operation.getCanGoToPreviousStep());

        this.operation.setCanGoToPreviousStep(false);
        Assert.assertFalse(this.operation.getCanGoToPreviousStep());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#getSelectedGroup()}
     * .
     */
    @Test
    public void testGetSetSelectedGroup() {
        Assert.assertNull(this.operation.getSelectedGroup());

        GroupBean selectedGroup = new GroupBean();
        selectedGroup.setId(new Integer((int) (Math.random() * 100)));
        this.operation.setSelectedGroup(selectedGroup);
        Assert.assertEquals(selectedGroup, this.operation.getSelectedGroup());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#getTopology()}
     * .
     */
    @Test
    public void testGetSetTopology() {
        Assert.assertNull(this.operation.getTopology());

        TopologyBean topology = new TopologyBean();
        topology.setId(new Integer((int) (Math.random() * 100)));
        this.operation.setTopology(topology);
        Assert.assertEquals(topology, this.operation.getTopology());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#getServerProgressList()}
     * .
     */
    @Test
    public void testGetSetServerProgressList() {
        Assert.assertNull(this.operation.getServerProgressList());

        List<ServerProgressBean> serverProgressList = new ArrayList<ServerProgressBean>();
        this.operation.setServerProgressList(serverProgressList);
        Assert.assertEquals(serverProgressList, this.operation.getServerProgressList());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#getWorkerProgressList()}
     * .
     */
    @Test
    public void testGetSetWorkerProgressList() {
        Assert.assertNull(this.operation.getWorkerProgressList());

        List<WorkerProgressBean> workerProgressList = new ArrayList<WorkerProgressBean>();
        this.operation.setWorkerProgressList(workerProgressList);
        Assert.assertEquals(workerProgressList, this.operation.getWorkerProgressList());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#getAllServerProgressList()}
     * .
     */
    @Test
    public void testGetSetAllServerProgressList() {
        Assert.assertNull(this.operation.getAllServerProgressList());

        Set<ServerProgressBean> serverProgressList = new HashSet<ServerProgressBean>();
        this.operation.setAllServerProgressList(serverProgressList);
        Assert.assertEquals(serverProgressList, this.operation.getAllServerProgressList());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#getAllWorkerProgressList()}
     * .
     */
    @Test
    public void testGetSetAllWorkerProgressList() {
        Assert.assertNull(this.operation.getAllWorkerProgressList());

        Set<WorkerProgressBean> workerProgressList = new HashSet<WorkerProgressBean>();
        this.operation.setAllWorkerProgressList(workerProgressList);
        Assert.assertEquals(workerProgressList, this.operation.getAllWorkerProgressList());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#getApplication()}
     * .
     */
    @Test
    public void testGetSetApplication() {
        Assert.assertNull(this.operation.getApplication());

        ApplicationBean application = new ApplicationBean();
        application.setId(new Integer((int) (Math.random() * 100)));
        this.operation.setApplication(application);
        Assert.assertEquals(application, this.operation.getApplication());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#getVmImage()}
     * .
     */
    @Test
    public void testGetSetVmImage() {
        Assert.assertNull(this.operation.getVmImage());

        VMImageBean vmImage = new VMImageBean();
        vmImage.setUuid(UUID.randomUUID().toString());
        this.operation.setVmImage(vmImage);
        Assert.assertEquals(vmImage, this.operation.getVmImage());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#getDate()}
     * .
     */
    @Test
    public void testGetSetDate() {
        Assert.assertNull(this.operation.getDate());

        Date date = new Date();
        this.operation.setDate(date);
        Assert.assertEquals(date, this.operation.getDate());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#getName()}
     * .
     */
    @Test
    public void testGetSetName() {
        Assert.assertNull(this.operation.getName());

        String name = this.getClass().getSimpleName() + '#' + UUID.randomUUID();
        this.operation.setName(name);
        Assert.assertEquals(name, this.operation.getName());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#getIsDeletable()}
     * .
     */
    @Test
    public void testGetSetIsDeletable() {
        for (Step step : Step.values()) {
            // SELECT_OPERATION cannot be set for a real operation
            if (step != Step.SELECT_OPERATION) {
                this.operation.setCurrentStep(step);
                if (step.equals(Step.EXECUTING_MIGRATION) || step.equals(Step.UNDEPLOY_ERASE_OLD_VERSION)
                    || step.equals(Step.EXECUTING_MIGRATION_OSGI) || step.equals(Step.ERASE_OLD_VERSIONS)
                    || step.equals(Step.SELECT_SERVERS) || step.equals(Step.SELECT_VM_IMAGE_FOR_SERVER)
                    || step.equals(Step.EXECUTING_MAINTENANCE_CLUSTER) || step.equals(Step.EXECUTING_MAINTENANCE_NO_CLUSTER)) {
                    Assert.assertFalse(this.operation.getIsDeletable());
                } else {
                    Assert.assertTrue(this.operation.getIsDeletable());
                }
            }
        }
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#equals(Object)}
     * .
     */
    @Test
    public void testEquals() {
        Integer id1 = (int) (Math.random() * 100);
        Integer id2 = (int) (Math.random() * 100);
        while (id1.equals(id2)) {
            id2 = (int) (Math.random() * 100);
        }

        OperationStateBean bean1 = new OperationStateBean();
        OperationStateBean bean2 = new OperationStateBean();

        // Both identifiers are null
        Assert.assertFalse(bean1.equals(bean2));

        // bean2.getId() is null
        bean1.setId(id1);
        Assert.assertFalse(bean1.equals(bean2));

        // bean1.getId() is null
        bean1.setId(null);
        bean2.setId(id1);
        Assert.assertFalse(bean1.equals(bean2));

        bean1.setId(id1);
        Assert.assertEquals(bean1, bean2);

        bean1.setId(id1);
        bean2.setId(id2);
        Assert.assertFalse(bean1.equals(bean2));

        bean1.setId(id2);
        bean2.setId(id2);
        Assert.assertEquals(bean1, bean2);
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean#toString()}
     * .
     */
    @Test
    public void testToString() {
        Assert.assertNotNull(this.operation.toString());
        Assert.assertFalse(this.operation.toString().contains(this.operation.getClass().getName()));
    }
}
