/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Copyright (C) 2008-2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ServerProgressBeanTest.java 4988 2009-08-20 11:23:41Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.api.entities.deployment;

import java.util.UUID;

import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean.ActionState;
import org.ow2.jasmine.jadort.api.entities.topology.ServerBean;

/**
 * Tests the ServerProgressListBean class.
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
public class ServerProgressBeanTest {

    ServerProgressBean serverProgress;

    /**
     * @throws java.lang.Exception
     */
    @Before
    public void setUp() throws Exception {
        this.serverProgress = new ServerProgressBean();
    }

    /**
     * @throws java.lang.Exception
     */
    @After
    public void tearDown() throws Exception {
        this.serverProgress = null;
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean#ServerProgressListBean()}
     * .
     */
    @Test
    public void testServerProgressListBean() {
        Assert.assertNotNull(this.serverProgress);
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean#ServerProgressListBean(org.ow2.jasmine.jadort.api.entities.topology.ServerBean)}
     * .
     */
    @Test
    public void testServerProgressListBeanServerBean() {
        ServerBean s = new ServerBean();
        s.setId(new Integer((int) (Math.random() * 100)));
        ServerProgressBean serverProgress = new ServerProgressBean(s);

        Assert.assertNotNull(serverProgress);
        Assert.assertEquals(s, serverProgress.getServer());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean#getId()}
     * .
     */
    @Test
    public void testGetSetId() {
        Integer id = (int) (Math.random() * 100);

        Assert.assertNull(this.serverProgress.getId());
        this.serverProgress.setId(id);
        Assert.assertEquals(this.serverProgress.getId(), id);
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean#getServer()}
     * .
     */
    @Test
    public void testGetSetServer() {
        ServerBean s = new ServerBean();
        s.setId(new Integer((int) (Math.random() * 100)));

        Assert.assertNull(this.serverProgress.getServer());
        this.serverProgress.setServer(s);
        Assert.assertEquals(s, this.serverProgress.getServer());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean#getServerName()}
     * .
     */
    @Test
    public void testGetServerName() {
        ServerBean s = new ServerBean();
        String name = this.getClass().getSimpleName() + '#' + UUID.randomUUID();

        s.setName(name);
        this.serverProgress.setServer(s);

        Assert.assertEquals(name, this.serverProgress.getServerName());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean#getState()}
     * .
     */
    @Test
    public void testGetSetState() {
        // Verify that the setState method really is a dummy one
        String state = "State#" + UUID.randomUUID();
        this.serverProgress.setState(state);
        Assert.assertFalse(state.equals(this.serverProgress.getState()));

        Assert.assertEquals("Waiting", this.serverProgress.getState());

        this.serverProgress.setActionState(ActionState.RUNNING);
        Assert.assertEquals("Running", this.serverProgress.getState());

        this.serverProgress.setActionState(ActionState.FINISHED_OK);
        Assert.assertEquals("Done OK", this.serverProgress.getState());

        this.serverProgress.setActionState(ActionState.FINISHED_ERROR);
        Assert.assertEquals("Error !", this.serverProgress.getState());

        this.serverProgress.setActionState(ActionState.WAITING);
        Assert.assertEquals("Waiting", this.serverProgress.getState());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean#getOldDeploymentItem()}
     * .
     */
    @Test
    public void testGetSetOldApplication() {
        String oldApplication = "Application#" + UUID.randomUUID();

        Assert.assertNull(this.serverProgress.getOldDeploymentItem());
        this.serverProgress.setOldDeploymentItem(oldApplication);
        Assert.assertEquals(oldApplication, this.serverProgress.getOldDeploymentItem());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean#getNewDeploymentItem()}
     * .
     */
    @Test
    public void testGetSetNewApplication() {
        String newApplication = "Application#" + UUID.randomUUID();

        Assert.assertNull(this.serverProgress.getNewDeploymentItem());
        this.serverProgress.setNewDeploymentItem(newApplication);
        Assert.assertEquals(newApplication, this.serverProgress.getNewDeploymentItem());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean#getVmImage()}
     * .
     */
    @Test
    public void testGetSetVmImage() {
        Assert.assertNull(this.serverProgress.getVmImage());

        VMImageBean vmImage = new VMImageBean();
        vmImage.setUuid(UUID.randomUUID().toString());
        this.serverProgress.setVmImage(vmImage);
        Assert.assertEquals(vmImage, this.serverProgress.getVmImage());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean#getLog()}
     * .
     */
    @Test
    public void testGetSetAppentLog() {
        StringBuilder log = new StringBuilder("Log#" + UUID.randomUUID());

        this.serverProgress.setLog(log.toString());
        Assert.assertEquals(log.toString(), this.serverProgress.getLog());

        String log2 = "Log#" + UUID.randomUUID();
        this.serverProgress.appendToLog(log2);

        Assert.assertEquals(log.append('\n' + log2).toString(), this.serverProgress.getLog());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean#getProgressState()}
     * .
     */
    @Test
    public void testGetsetProgressState() {
        Assert.assertEquals(ServerProgressState.INITIAL, this.serverProgress.getProgressState());

        for (ServerProgressState state : ServerProgressState.values()) {
            this.serverProgress.setProgressState(state);
            Assert.assertEquals(state, this.serverProgress.getProgressState());
        }

        this.serverProgress.setProgressState(null);
        Assert.assertNull(this.serverProgress.getProgressState());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean#getActionState()}
     * .
     */
    @Test
    public void testGetSetActionState() {
        Assert.assertEquals(ActionState.WAITING, this.serverProgress.getActionState());

        for (ActionState state : ActionState.values()) {
            this.serverProgress.setActionState(state);
            Assert.assertEquals(state, this.serverProgress.getActionState());
        }

        this.serverProgress.setActionState(null);
        Assert.assertNull(this.serverProgress.getActionState());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean#getProgress()}
     * .
     */
    @Test
    public void testGetSetProgress() {
        int initialProgress = this.serverProgress.getProgress();
        Assert.assertEquals(0, initialProgress);

        int progress = 33;
        this.serverProgress.setProgress(progress);
        Assert.assertEquals(progress, this.serverProgress.getProgress());

        try {
            this.serverProgress.setProgress(-1);
        } catch (IllegalArgumentException ignored) {
            // Ignored
        }
        Assert.assertEquals(progress, this.serverProgress.getProgress());

        try {
            this.serverProgress.setProgress(101);
        } catch (IllegalArgumentException ignored) {
            // Ignored
        }
        Assert.assertEquals(progress, this.serverProgress.getProgress());

        int newProgress = 66;
        this.serverProgress.setProgress(newProgress);
        Assert.assertEquals(newProgress, this.serverProgress.getProgress());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean#getIsOnTarget()}
     * .
     */
    @Test
    public void testGetSetIsOnTarget() {
        Assert.assertEquals(false, this.serverProgress.getIsOnTarget());

        this.serverProgress.setIsOnTarget(true);
        Assert.assertEquals(true, this.serverProgress.getIsOnTarget());

        this.serverProgress.setIsOnTarget(false);
        Assert.assertEquals(false, this.serverProgress.getIsOnTarget());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean#equals(Object)}
     * .
     */
    @Test
    public void testEquals() {
        Integer id1 = (int) (Math.random() * 100);
        Integer id2 = (int) (Math.random() * 100);
        while (id1.equals(id2)) {
            id2 = (int) (Math.random() * 100);
        }

        ServerProgressBean bean1 = new ServerProgressBean();
        ServerProgressBean bean2 = new ServerProgressBean();

        // Both identifiers are null
        Assert.assertFalse(bean1.equals(bean2));

        // bean2.getId() is null
        bean1.setId(id1);
        Assert.assertFalse(bean1.equals(bean2));

        // bean1.getId() is null
        bean1.setId(null);
        bean2.setId(id1);
        Assert.assertFalse(bean1.equals(bean2));

        bean1.setId(id1);
        Assert.assertEquals(bean1, bean2);

        bean1.setId(id1);
        bean2.setId(id2);
        Assert.assertFalse(bean1.equals(bean2));

        bean1.setId(id2);
        bean2.setId(id2);
        Assert.assertEquals(bean1, bean2);
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean#toString()}
     * .
     */
    @Test
    public void testToString() {
        Assert.assertNotNull(this.serverProgress.toString());
        Assert.assertFalse(this.serverProgress.toString().contains(this.serverProgress.getClass().getName()));
    }
}
