/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Copyright (C) 2008-2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: WorkerProgressBeanTest.java 3492 2009-04-21 10:16:12Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.api.entities.deployment;

import java.util.UUID;

import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean.ActionState;
import org.ow2.jasmine.jadort.api.entities.topology.WorkerBean;

/**
 * Tests the WorkerProgressListBean class.
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
public class WorkerProgressBeanTest {

    WorkerProgressBean workerProgress;

    /**
     * @throws java.lang.Exception
     */
    @Before
    public void setUp() throws Exception {
        this.workerProgress = new WorkerProgressBean();
    }

    /**
     * @throws java.lang.Exception
     */
    @After
    public void tearDown() throws Exception {
        this.workerProgress = null;
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.WorkerProgressBean#WorkerProgressListBean()}
     * .
     */
    @Test
    public void testWorkerProgressListBean() {
        Assert.assertNotNull(this.workerProgress);
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.WorkerProgressBean#WorkerProgressListBean(org.ow2.jasmine.jadort.api.entities.topology.WorkerBean)}
     * .
     */
    @Test
    public void testWorkerProgressListBeanWorkerBean() {
        WorkerBean w = new WorkerBean();
        w.setId(new Integer((int) (Math.random() * 100)));
        WorkerProgressBean workerProgress = new WorkerProgressBean(w);

        Assert.assertNotNull(workerProgress);
        Assert.assertEquals(w, workerProgress.getWorker());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.WorkerProgressBean#getId()}
     * .
     */
    @Test
    public void testGetSetId() {
        Integer id = (int) (Math.random() * 100);

        Assert.assertNull(this.workerProgress.getId());
        this.workerProgress.setId(id);
        Assert.assertEquals(this.workerProgress.getId(), id);
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.WorkerProgressBean#getWorker()}
     * .
     */
    @Test
    public void testGetSetWorker() {
        WorkerBean w = new WorkerBean();
        w.setId(new Integer((int) (Math.random() * 100)));

        Assert.assertNull(this.workerProgress.getWorker());
        this.workerProgress.setWorker(w);
        Assert.assertEquals(w, this.workerProgress.getWorker());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.WorkerProgressBean#getWorkerName()}
     * .
     */
    @Test
    public void testGetWorkerName() {
        WorkerBean w = new WorkerBean();
        String name = this.getClass().getSimpleName() + '#' + UUID.randomUUID();

        w.setName(name);
        this.workerProgress.setWorker(w);

        Assert.assertEquals(name, this.workerProgress.getWorkerName());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.WorkerProgressBean#getState()}
     * .
     */
    @Test
    public void testGetSetState() {
        // Verify that the setState method really is a dummy one
        String state = "State#" + UUID.randomUUID();
        this.workerProgress.setState(state);
        Assert.assertFalse(state.equals(this.workerProgress.getState()));

        Assert.assertEquals("Waiting", this.workerProgress.getState());

        this.workerProgress.setActionState(ActionState.RUNNING);
        Assert.assertEquals("Running", this.workerProgress.getState());

        this.workerProgress.setActionState(ActionState.FINISHED_OK);
        Assert.assertEquals("Done OK", this.workerProgress.getState());

        this.workerProgress.setActionState(ActionState.FINISHED_ERROR);
        Assert.assertEquals("Error !", this.workerProgress.getState());

        this.workerProgress.setActionState(ActionState.WAITING);
        Assert.assertEquals("Waiting", this.workerProgress.getState());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.WorkerProgressBean#getLog()}
     * .
     */
    @Test
    public void testGetSetAppentLog() {
        StringBuilder log = new StringBuilder("Log#" + UUID.randomUUID());

        this.workerProgress.setLog(log.toString());
        Assert.assertEquals(log.toString(), this.workerProgress.getLog());

        String log2 = "Log#" + UUID.randomUUID();
        this.workerProgress.appendToLog(log2);

        Assert.assertEquals(log.append('\n' + log2).toString(), this.workerProgress.getLog());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.WorkerProgressBean#testGetsetProgressState()}
     * .
     */
    @Test
    public void testGetsetProgressState() {
        Assert.assertEquals(WorkerProgressState.INITIAL, this.workerProgress.getProgressState());

        for (WorkerProgressState state : WorkerProgressState.values()) {
            this.workerProgress.setProgressState(state);
            Assert.assertEquals(state, this.workerProgress.getProgressState());
        }

        this.workerProgress.setProgressState(null);
        Assert.assertNull(this.workerProgress.getProgressState());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.WorkerProgressBean#getActionState()}
     * .
     */
    @Test
    public void testGetSetActionState() {
        Assert.assertEquals(ActionState.WAITING, this.workerProgress.getActionState());

        for (ActionState state : ActionState.values()) {
            this.workerProgress.setActionState(state);
            Assert.assertEquals(state, this.workerProgress.getActionState());
        }

        this.workerProgress.setActionState(null);
        Assert.assertNull(this.workerProgress.getActionState());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.WorkerProgressBean#getProgress()}
     * .
     */
    @Test
    public void testGetSetProgress() {
        int initialProgress = this.workerProgress.getProgress();
        Assert.assertEquals(0, initialProgress);

        int progress = 33;
        this.workerProgress.setProgress(progress);
        Assert.assertEquals(progress, this.workerProgress.getProgress());

        try {
            this.workerProgress.setProgress(-1);
        } catch (IllegalArgumentException ignored) {
            // Ignored
        }
        Assert.assertEquals(progress, this.workerProgress.getProgress());

        try {
            this.workerProgress.setProgress(101);
        } catch (IllegalArgumentException ignored) {
            // Ignored
        }
        Assert.assertEquals(progress, this.workerProgress.getProgress());

        int newProgress = 66;
        this.workerProgress.setProgress(newProgress);
        Assert.assertEquals(newProgress, this.workerProgress.getProgress());
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.WorkerProgressBean#equals(Object)}
     * .
     */
    @Test
    public void testEquals() {
        Integer id1 = (int) (Math.random() * 100);
        Integer id2 = (int) (Math.random() * 100);
        while (id1.equals(id2)) {
            id2 = (int) (Math.random() * 100);
        }

        WorkerProgressBean bean1 = new WorkerProgressBean();
        WorkerProgressBean bean2 = new WorkerProgressBean();

        // Both identifiers are null
        Assert.assertFalse(bean1.equals(bean2));

        // bean2.getId() is null
        bean1.setId(id1);
        Assert.assertFalse(bean1.equals(bean2));

        // bean1.getId() is null
        bean1.setId(null);
        bean2.setId(id1);
        Assert.assertFalse(bean1.equals(bean2));

        bean1.setId(id1);
        Assert.assertEquals(bean1, bean2);

        bean1.setId(id1);
        bean2.setId(id2);
        Assert.assertFalse(bean1.equals(bean2));

        bean1.setId(id2);
        bean2.setId(id2);
        Assert.assertEquals(bean1, bean2);
    }

    /**
     * Test method for
     * {@link org.ow2.jasmine.jadort.api.entities.deployment.WorkerProgressBean#toString()}
     * .
     */
    @Test
    public void testToString() {
        Assert.assertNotNull(this.workerProgress.toString());
        Assert.assertFalse(this.workerProgress.toString().contains(this.workerProgress.getClass().getName()));
    }
}
