/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Copyright (C) 2008-2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: WorkerProgressBean.java 3904 2009-06-09 13:08:00Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.api.entities.deployment;

import javax.persistence.Basic;
import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.Lob;
import javax.persistence.OneToOne;
import javax.persistence.Version;

import org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean.ActionState;
import org.ow2.jasmine.jadort.api.entities.topology.WorkerBean;

/**
 * Bean that represents the progress of the operations a worker.
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
@Entity(name = "JaDOrT_WorkerProgressBean")
public class WorkerProgressBean extends ProgressBean<WorkerProgressState> {

    /**
     * Auto-generated serial version ID.
     */
    private static final long serialVersionUID = -2851071984012556925L;

    /**
     * Auto-generated identifier of the WorkerProgressBean in the database.
     */
    @Id
    @GeneratedValue
    private Integer id;

    /**
     * Auto-generated version of the WorkerProgressBean. Used internally for
     * checking integrity.
     */
    @Version
    @SuppressWarnings("unused")
    private Integer version;

    /**
     * The worker associated with this WorkerProgressBean.
     */
    // Cascade for all except CascadeType.REMOVE, since removing a
    // WorkerProgressBean doesn't imply that the associated WorkerBean
    // (created as part of a migration operation) should be deleted!
    @OneToOne(cascade = {CascadeType.MERGE, CascadeType.PERSIST, CascadeType.REFRESH})
    private WorkerBean worker;

    /**
     * Deployment log
     */
    @Lob
    @Basic(fetch = FetchType.EAGER)
    private String log;

    /**
     * Deployment progress state
     */
    private WorkerProgressState workerProgressState;

    /**
     * Deployment state
     */
    private ActionState actionState;

    /**
     * Progress percent of the workerProgress.
     */
    private int progressPercent;

    /**
     * Default WorkerProgressBean constructor
     */
    public WorkerProgressBean() {
        this.progressPercent = 0;
        this.actionState = ActionState.WAITING;
        this.workerProgressState = WorkerProgressState.INITIAL;
    }

    /**
     * Construct a WorkerProgressBean with its associated worker
     * 
     * @param worker The worker associated the deployment
     */
    public WorkerProgressBean(final WorkerBean worker) {
        this();
        this.worker = worker;
    }

    /**
     * Get the WorkerProgressBean id
     * 
     * @return the WorkerProgressBean id
     */
    public Integer getId() {
        return this.id;
    }

    /**
     * Set the WorkerProgressBean id
     * 
     * @param id the WorkerProgressBean id
     */
    public void setId(final Integer id) {
        this.id = id;
    }

    /**
     * Associated worker getter
     * 
     * @return The associated worker
     */
    public WorkerBean getWorker() {
        return this.worker;
    }

    /**
     * Change the associated worker
     * 
     * @param worker New associated worker
     */
    public void setWorker(final WorkerBean worker) {
        this.worker = worker;
    }

    /**
     * Returns the name of the worker associated with this WorkerProgressBean
     * 
     * @return the name of the worker associated with this WorkerProgressBean
     */
    public String getWorkerName() {
        return this.worker.getName();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getLog() {
        return this.log;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setLog(final String log) {
        this.log = log;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public WorkerProgressState getProgressState() {
        return this.workerProgressState;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setProgressState(final WorkerProgressState workerProgressState) {
        this.workerProgressState = workerProgressState;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ActionState getActionState() {
        return this.actionState;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setActionState(final ActionState actionState) {
        this.actionState = actionState;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int getProgress() {
        return this.progressPercent;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setProgress(final int progressPercent) {
        if (progressPercent < 0 || progressPercent > 100) {
            throw new IllegalArgumentException("ProgressPercent is a percentage (between 0 and 100)");
        }

        this.progressPercent = progressPercent;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(this.getClass().getSimpleName());
        sb.append("[worker='");
        sb.append(this.worker);
        sb.append("']");
        return sb.toString();
    }

    /**
     * {@inheritDoc}
     * 
     * @return true if <code>obj</code> is a {@link WorkerProgressBean} and the
     *         {@link WorkerProgressBean#getId()} is the same for both objects,
     *         false otherwise.
     */
    @Override
    public boolean equals(final Object obj) {
        if (obj != null && obj instanceof WorkerProgressBean) {
            WorkerProgressBean other = (WorkerProgressBean) obj;
            if (this.id != null && other.id != null) {
                return this.id.equals(other.id);
            }
        }

        // Else not same type or some parts are null
        return false;
    }

}
