/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008-2009 Bull S.A.S.
 * Copyright (C) 2008-2009 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JkUpdateAccessor.java 10052 2012-05-29 13:33:07Z pelletib $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.service.action.modJK;

import java.net.HttpURLConnection;
import org.apache.jk.status.JkBalancer;
import org.apache.jk.status.JkBalancerMember;
import org.apache.jk.status.JkStatus;
import org.apache.jk.status.JkStatusAccessor;
import org.apache.tomcat.util.digester.Digester;


/**
 * Creates the HTTP GET requests for mod_jk updates.
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
public class JkUpdateAccessor extends JkStatusAccessor {

    /**
     * Parses Apache mod_jk Status from base url http://host:port/jkstatus
     * 
     * @param url mod_jk manager URL.
     * @param username User name to use when connecting, null not to use any.
     * @param password Password to use when connecting, null not to use any.
     * @param balancer Balancer to update.
     * @return Current status.
     * @throws JkConnectionException If response parsing fails.
     */
    public JkStatus update(final String url, final String username, final String password, final JkBalancer balancer)
        throws JkConnectionException {
        if (url == null || "".equals(url)) {
            return null;
        }
        HttpURLConnection hconn = null;
        JkStatus status = null;

        try {
            hconn = this.openConnection(url + "?mime=xml&cmd=update" + this.createUpdateParameterLink(balancer), username,
                password);
            Digester digester = JkActionParser.getActionDigester();
            synchronized (digester) {
                status = (JkStatus) digester.parse(hconn.getInputStream());
            }
        } catch (Exception e) {
            throw new JkConnectionException("Failed parsing response", e);
        } finally {
            if (hconn != null) {
                try {
                    hconn.disconnect();
                } catch (Throwable u) {
                    ;
                }
                hconn = null;
            }
        }
        return status;
    }

    /**
     * Create JkStatus worker update link
     * <ul>
     * </b>http://host:port/jkstatus?cmd=update&mime=txt&w=loadbalancer&lm=1&ll=
     * 1&lr=2&lt=60&ls=true&lf=false <br/>
     * <br/>
     * Tcp worker parameter: <br/>
     * <ul>
     * <li><b>w:<b/> name loadbalancer</li>
     * <li><b>vlm:<b/> method (lb strategy)</li>
     * <li><b>vll:<b/> lock</li>
     * <li><b>vlr:<b/> retries</li>
     * <li><b>vlt:<b/> recover wait timeout</li>
     * <li><b>vls:<b/> sticky session</li>
     * <li><b>vlf:<b/> force sticky session</li>
     * </ul>
     * <ul>
     * <li>vlm=1 or Requests</li>
     * <li>vlm=2 or Traffic</li>
     * <li>vlm=3 or Busyness</li>
     * <li>vlm=4 or Sessions</li>
     * </ul>
     * <ul>
     * <li>vll=1 or Optimistic</li>
     * <li>vll=2 or Pessimistic</li>
     * </ul>
     * 
     * @return create jkstatus update worker link
     */
    protected String createUpdateParameterLink(final JkBalancer balancer) {
        StringBuffer sb = new StringBuffer();
        String lbMethod = balancer.getMethod();
        String locking = balancer.getLock();

        sb.append("&w=");
        sb.append(balancer.getName());
        sb.append("&vls=");
        sb.append(balancer.isSticky_session());
        sb.append("&vlf=");
        sb.append(balancer.isSticky_session_force());
        sb.append("&vlr=");
        sb.append(balancer.getRetries());
        sb.append("&vlt=");
        sb.append(balancer.getRecover_time());

        if ("Request".equals(lbMethod)) {
            sb.append("&vlm=0");
        } else if ("Traffic".equals(lbMethod)) {
            sb.append("&vlm=1");
        } else if ("Busyness".equals(lbMethod)) {
            sb.append("&vlm=2");
        } else if ("Sessions".equals(lbMethod)) {
            sb.append("&vlm=3");
        }
        if ("Optimistic".equals(locking)) {
            sb.append("&vll=0");
        } else if ("Pessimistic".equals(locking)) {
            sb.append("&vll=1");
        }
        return sb.toString();
    }

    /**
     * Parses Apache mod_jk Status from base url http://host:port/jkstatus
     * 
     * @param url mod_jk manager URL.
     * @param username User name to use when connecting, null not to use any.
     * @param password Password to use when connecting, null not to use any.
     * @param balancer Balancer to update.
     * @param balancerMember Balancer member to update.
     * @return Current status.
     * @throws JkConnectionException If response parsing fails.
     */
    public JkStatus update(final String url, final String username, final String password, final JkBalancer balancer,
        final JkBalancerMember balancerMember) throws JkConnectionException {
        if (url == null || "".equals(url)) {
            return null;
        }
        HttpURLConnection hconn = null;
        JkStatus status = null;

        try {
            hconn = this.openConnection(
                url + "?mime=xml&cmd=update" + this.createUpdateParameterLink(balancer, balancerMember), username, password);
            Digester digester = JkActionParser.getActionDigester();
            synchronized (digester) {
                status = (JkStatus) digester.parse(hconn.getInputStream());
            }
        } catch (Exception e) {
            throw new JkConnectionException("Failed parsing response", e);
        } finally {
            if (hconn != null) {
                try {
                    hconn.disconnect();
                } catch (Throwable u) {
                    ;
                }
                hconn = null;
            }
        }
        return status;
    }

    /**
     * Create JkStatus worker update link
     * <ul>
     * </b>http://host:port/jkstatus?cmd=update&mime=txt&w=loadbalancer&sw=
     * member&vwa= 0&vwf=1&vwn=memberName&vwr=&vwc=&vwd=0 <br/>
     * <br/>
     * Tcp worker parameter: <br/>
     * <ul>
     * <li><b>w:<b/> name loadbalancer</li>
     * <li><b>sw:<b/> name member</li>
     * <li><b>vwa:<b/> activation</li>
     * <li><b>vwf:<b/> LB factor</li>
     * <li><b>vwn:<b/> route</li>
     * <li><b>vwr:<b/> redirect route</li>
     * <li><b>vwc:<b/> cluster domain</li>
     * <li><b>vwd:<b/> distance</li>
     * </ul>
     * <ul>
     * <li>vwa=1 or Active or ACT</li>
     * <li>vwa=2 or Disabled or DIS</li>
     * <li>vwa=3 or Stopped or STP</li>
     * </ul>
     * 
     * @return create jkstatus update worker link
     */
    protected String createUpdateParameterLink(final JkBalancer balancer, final JkBalancerMember balancerMember) {
        StringBuffer sb = new StringBuffer();
        String activation = balancerMember.getActivation();

        sb.append("&w=");
        sb.append(balancer.getName());
        sb.append("&sw=");
        sb.append(balancerMember.getName());
        if ("ACT".equals(activation)) {
            sb.append("&vwa=0");
        } else if ("DIS".equals(activation)) {
            sb.append("&vwa=1");
        } else if ("STP".equals(activation)) {
            sb.append("&vwa=2");
        }
        sb.append("&vwf=");
        sb.append(balancerMember.getLbfactor());
        sb.append("&vwn=");
        sb.append(balancerMember.getRoute());
        sb.append("&vwr=");
        sb.append(balancerMember.getRedirect());
        sb.append("&vwc=");
        sb.append(balancerMember.getDomain());
        sb.append("&vwd=");
        sb.append(balancerMember.getDistance());
        return sb.toString();
    }
}
