/**
 * JaDOrT: JASMINe Deployment Orchestration Tool
 * Copyright (C) 2008 Bull S.A.S.
 * Copyright (C) 2008 France Telecom R&D
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JadortServiceClient.java 2913 2008-12-18 12:08:01Z alitokmen $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.jadort.client;

import java.io.File;
import java.io.Serializable;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import javax.naming.Context;
import javax.naming.InitialContext;

import org.ow2.jasmine.jadort.api.IJadortService;
import org.ow2.jasmine.jadort.api.JadortServiceException;
import org.ow2.jasmine.jadort.api.entities.deployment.ApplicationBean;
import org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean;
import org.ow2.jasmine.jadort.api.entities.deployment.ServerProgressBean;
import org.ow2.jasmine.jadort.api.entities.deployment.WorkerProgressBean;
import org.ow2.jasmine.jadort.api.entities.deployment.OperationStateBean.Step;
import org.ow2.jasmine.jadort.api.entities.topology.GroupBean;
import org.ow2.jasmine.jadort.api.entities.topology.ServerBean;
import org.ow2.jasmine.jadort.api.entities.topology.WorkerBean;

/**
 * This class handles the communication between the Web client and the JaDOrT
 * Service stateful bean.
 * 
 * @author Malek Chahine
 * @author Remy Bresson
 * @author S. Ali Tokmen
 */
public class JadortServiceClient implements Serializable {

    private static final long serialVersionUID = 3148455079928943579L;

    private IJadortService jadortService;

    private static final Map<OperationStateBean.Step, Integer> Step_PageIndex = new Hashtable<Step, Integer>();
    static {
        JadortServiceClient.Step_PageIndex.put(Step.SELECT_OPERATION, 0);
        JadortServiceClient.Step_PageIndex.put(Step.INITIALIZE_TOPOLOGY, 1);
        JadortServiceClient.Step_PageIndex.put(Step.SELECT_GROUP, 2);
        JadortServiceClient.Step_PageIndex.put(Step.SELECT_APPLICATION, 3);
        JadortServiceClient.Step_PageIndex.put(Step.EXECUTING_MIGRATION_PART1, 4);
        JadortServiceClient.Step_PageIndex.put(Step.EXECUTING_MIGRATION_PART2, 5);
        JadortServiceClient.Step_PageIndex.put(Step.FINISHED, 6);
        JadortServiceClient.Step_PageIndex.put(Step.SELECT_SERVERS, 7);
        JadortServiceClient.Step_PageIndex.put(Step.EXECUTING_MAINTENANCE_CLUSTER, 8);
        JadortServiceClient.Step_PageIndex.put(Step.EXECUTING_MAINTENANCE_NO_CLUSTER, 9);
    }

    private JadortServiceExceptionWithInfo analyzeException(final Exception e) {
        // Check for JadortServiceException
        for (Throwable cause = e; cause != null; cause = cause.getCause()) {
            if (cause instanceof JadortServiceException && cause.getMessage() != null) {
                return new JadortServiceExceptionWithInfo(cause.getMessage());
            }
        }

        // If not JadortServiceException, the exception has been thrown by
        // the container. Output the stack trace, since this probably is a bug.
        Throwable cause = e.getCause();
        if (cause == null) {
            cause = e;
        }
        String message = "Container exception " + cause.getClass().getName() + " in the JadortService client: "
            + cause.getMessage();
        synchronized (this) {
            System.out.println(message);
            e.printStackTrace();
        }
        return new JadortServiceExceptionWithInfo(message);
    }

    public void initialize() throws JadortServiceExceptionWithInfo {
        try {
            Context context = new InitialContext();
            this.jadortService = (IJadortService) context.lookup("ow2.jasmine.jadort.ServiceSFB");
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public boolean canGoToNextStep() throws JadortServiceExceptionWithInfo {
        try {
            return this.jadortService.canGoToNextStep();
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public boolean canGoToPreviousStep() throws JadortServiceExceptionWithInfo {
        try {
            return this.jadortService.canGoToPreviousStep();
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public boolean checkServer(final ServerBean server) throws JadortServiceExceptionWithInfo {
        try {
            return this.jadortService.checkServer(server);
        } catch (Exception ignored) {
            // Ignore exception and return false, ActionScript will warn anyway
            return false;
        }
    }

    public void createApplication(final String urlOrFile) throws JadortServiceExceptionWithInfo {
        try {
            URL url;
            try {
                url = new URL(urlOrFile);
            } catch (MalformedURLException e) {
                // urlOrFile is not a URL, it is a file
                //
                // Use File.oURI().toURL() and not File.toURL() since
                // File.toURL() doesn't properly escape the URL.
                url = new File(urlOrFile).toURI().toURL();
            }
            this.jadortService.createApplication(url);
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public ApplicationBean getApplication() throws JadortServiceExceptionWithInfo {
        try {
            OperationStateBean operation = this.jadortService.getCurrentOperation();
            return operation.getApplication();
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public void ignoreServer(final ServerBean server) throws JadortServiceExceptionWithInfo {
        try {
            this.jadortService.ignoreServer(server);
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public List<ServerProgressBean> getServerProgressList() throws JadortServiceExceptionWithInfo {
        try {
            return this.jadortService.getServerProgressList();
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public List<WorkerProgressBean> getWorkerProgressList() throws JadortServiceExceptionWithInfo {
        try {
            return this.jadortService.getWorkerProgressList();
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public List<GroupBean> getGroups() throws JadortServiceExceptionWithInfo {
        try {
            return this.jadortService.getGroups();
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public int getPageIndex() throws JadortServiceExceptionWithInfo {
        try {
            OperationStateBean operation = this.jadortService.getCurrentOperation();
            return JadortServiceClient.Step_PageIndex.get(operation.getCurrentStep());
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public GroupBean getSelectedGroup() throws JadortServiceExceptionWithInfo {
        try {
            return this.jadortService.getSelectedGroup();
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public int getActiveSessions() throws JadortServiceExceptionWithInfo {
        try {
            return this.jadortService.getActiveSessions();
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public void loadTopology(final String xmlTopoFile) throws JadortServiceExceptionWithInfo {
        try {
            this.jadortService.loadTopology(new File(xmlTopoFile));
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public int next() throws JadortServiceExceptionWithInfo {
        try {
            this.jadortService.next();
            return this.getPageIndex();
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public int previous() throws JadortServiceExceptionWithInfo {
        try {
            this.jadortService.previous();
            return this.getPageIndex();
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public void restartServer(final ServerBean server) throws JadortServiceExceptionWithInfo {
        try {
            this.jadortService.restartServer(server);
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public void abortServer(final ServerBean server) throws JadortServiceExceptionWithInfo {
        try {
            this.jadortService.abortServer(server);
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public void selectGroup(final GroupBean selectedGroup, final String operationType) throws JadortServiceExceptionWithInfo {
        try {
            // Only send out a subset to avoid ClassCastException
            GroupBean selectedGroupCopy = new GroupBean();
            selectedGroupCopy.setId(selectedGroup.getId());
            selectedGroupCopy.setName(selectedGroup.getName());
            selectedGroupCopy.setClustered(selectedGroup.getClustered());
            if (operationType.equalsIgnoreCase(IJadortService.OperationType.MIGRATE.toString())) {
                this.jadortService.selectGroup(selectedGroupCopy, IJadortService.OperationType.MIGRATE);
            } else if (operationType.equalsIgnoreCase(IJadortService.OperationType.MAINTAIN.toString())) {
                this.jadortService.selectGroup(selectedGroupCopy, IJadortService.OperationType.MAINTAIN);
            }
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public OperationStateBean createNewOperation(final String newOperationName) throws JadortServiceExceptionWithInfo {
        try {
            this.jadortService.createNewOperation(newOperationName);
            return this.jadortService.getCurrentOperation();
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public List<OperationStateBean> getOperationsList() throws JadortServiceExceptionWithInfo {
        try {
            return this.jadortService.getOperationsList();
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public OperationStateBean selectOperation(final OperationStateBean operation) throws JadortServiceExceptionWithInfo {
        try {
            this.jadortService.selectOperation(operation);
            return this.jadortService.getCurrentOperation();
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public void deleteOperation(final OperationStateBean operation) throws JadortServiceExceptionWithInfo {
        try {
            this.jadortService.deleteOperation(operation);
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public void selectServers(final List<ServerBean> selectedServers) throws JadortServiceExceptionWithInfo {
        try {
            // Create a (serializable) ArrayList
            List<ServerBean> selectedServersCopy = new ArrayList<ServerBean>(selectedServers);
            this.jadortService.selectServers(selectedServersCopy);
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public List<ServerBean> getServers() throws JadortServiceExceptionWithInfo {
        try {
            return this.jadortService.getServers();
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public void restartWorker(final WorkerBean worker) throws JadortServiceExceptionWithInfo {
        try {
            this.jadortService.restartWorker(worker);
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public boolean checkWorker(final WorkerBean worker) throws JadortServiceExceptionWithInfo {
        try {
            return this.jadortService.checkWorker(worker);
        } catch (Exception ignored) {
            // Ignore exception and return false, ActionScript will warn anyway
            return false;
        }
    }

    public void abortWorker(final WorkerBean worker) throws JadortServiceExceptionWithInfo {
        try {
            this.jadortService.abortWorker(worker);
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }

    public void ignoreWorker(final WorkerBean worker) throws JadortServiceExceptionWithInfo {
        try {
            this.jadortService.ignoreWorker(worker);
        } catch (Exception e) {
            throw this.analyzeException(e);
        }
    }
}
