/**
 * JASMINe
 * Copyright (C) 2010-2011 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JasmineEventNotificationUpdate.java 8278 2011-06-10 12:12:00Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.event.beans;

import org.ow2.jasmine.event.api.IJasmineEventNotification;
import org.ow2.jasmine.event.component.SearchFilter;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * Contains informations to update a stored {@link JasmineEventNotification}.
 * The update will add a {@link JasmineEventNotificationDetail}, or a
 * {@link JasmineEventNotificationExecutedAction}. Filters have to be set to get
 * a unique notification to update.
 *
 * @author Guillaume Renault
 */
public class JasmineEventNotificationUpdate implements IJasmineEventNotification,Serializable {

    /**
     * serialVersionUID.
     */
    private static final long serialVersionUID = -7505249031966343802L;

    /**
     * ID of the notification to update
     */
    private String id;

    /**
     * the list of filters to apply to find the notification to update.
     */
    private List<SearchFilter> searchFilters;

    /**
     * The detail to add.
     */
    private List<JasmineEventNotificationDetail> details;

    /**
     * The action to add.
     */
    private JasmineEventNotificationExecutedAction action;

    /**
     * Date to update
     */
    private Date date;

    /**
     * Message to update.
     */
    private String message;

    /**
     * Comment about the update.
     */
    private String comment;

    /**
     * The user who created the update Object. It may be the System itself.
     */
    private String user;

    /**
     * Contains the new rule name for the notification to update.
     */
    private String rule;

    /**
     * Maximum number a notificationUpdate can be re-injected in eventswitch
     */
    private int maxRetry;

    /**
     * Initialize the {@link SearchFilter} list and the {@link JasmineEventNotificationDetail} list.
     *
     */
    public JasmineEventNotificationUpdate() {
        this.searchFilters = new ArrayList<SearchFilter>();
        this.details = new ArrayList<JasmineEventNotificationDetail>();
    }

    /**
     * Initialize the {@link SearchFilter} list and set the detail to add.
     *
     * @param p_details the {@link JasmineEventNotificationDetail} to add.
     */
    public JasmineEventNotificationUpdate(final List<JasmineEventNotificationDetail> p_details) {
        this.searchFilters = new ArrayList<SearchFilter>();
        this.details = p_details;
    }

    /**
     * Initialize the {@link SearchFilter} list and set the action to add.
     *
     * @param p_action the {@link JasmineEventNotificationExecutedAction} to
     *        add.
     */
    public JasmineEventNotificationUpdate(final JasmineEventNotificationExecutedAction p_action) {
        this.searchFilters = new ArrayList<SearchFilter>();
        this.action = p_action;
    }

    /**
     * @return the ID of the notification to update or updated
     */
    public String getId() {
        return id;
    }

    /**
     * @param id the ID of the notification to update or updated
     */
    public void setId(String id) {
        this.id = id;
    }

    /**
     * @return the detail
     */
    public List<JasmineEventNotificationDetail> getDetails() {
        return details;
    }

    /**
     * @param detail the detail to set
     */
    public final void addDetail(final JasmineEventNotificationDetail detail) {
        this.details.add(detail);
    }

    /**
     * @return the action
     */
    public final JasmineEventNotificationExecutedAction getAction() {
        return action;
    }

    /**
     * @param action the action to set
     */
    public final void setAction(final JasmineEventNotificationExecutedAction action) {
        this.action = action;
    }

    /**
     * @return the filters
     */
    public List<SearchFilter> getSearchFilters() {
        return searchFilters;
    }

    /**
     * Add a {@link SearchFilter} to the list.
     *
     * @param searchFilter the {@link SearchFilter} to add.
     */
    public void addSearchFilter(final SearchFilter searchFilter) {
        this.searchFilters.add(searchFilter);
    }

    /**
     * @return the date
     */
    public Date getDate() {
        return date;
    }

    /**
     *
     * @param date the date of the notification update
     */
    public void setDate(Date date) {
        this.date = date;
    }

    /**
     * @return the message
     */
    public final String getMessage() {
        return message;
    }

    /**
     * @param message the message to set
     */
    public final void setMessage(final String message) {
        this.message = message;
    }

    /**
     * @return the comment
     */
    public final String getComment() {
        return comment;
    }

    /**
     * @param comment the comment to set
     */
    public final void setComment(final String comment) {
        this.comment = comment;
    }

    /**
     * @return the user
     */
    public final String getUser() {
        return user;
    }

    /**
     * @param user the user to set
     */
    public final void setUser(final String user) {
        this.user = user;
    }


    public String getRule()
    {
        return rule;
    }

    public void setRule(String rule)
    {
        this.rule=rule;
    }

    /**
     * @return the max retry
     */
    public final int getMaxRetry() {
        return maxRetry;
    }

    /**
     * @param maxRetry 
     */
    public final void setMaxRetry(final int maxRetry) {
        this.maxRetry = maxRetry;
    }

    @Override
    public String toString() {
        StringBuffer buffer = new StringBuffer();
        buffer.append("JasmineEventNotificationUpdate [ ");
        buffer.append("rule = ");
        buffer.append(this.rule);
        buffer.append(" ; user = ");
        buffer.append(this.user);
        buffer.append(" ; detail to update/add = [");
        for (JasmineEventNotificationDetail jend : details) {
            buffer.append("[");
            buffer.append(jend.getLabel());
            buffer.append(" = ");
            buffer.append(jend.getMessage());
            buffer.append("],");
        }
        buffer.append("] ; filters = [");
        for(SearchFilter sf : this.searchFilters) {
            Map<String, String> filters = sf.getFilters();
            for(String key : filters.keySet()) {
                buffer.append(key);
                buffer.append("=");
                buffer.append(filters.get(key));
                buffer.append(";");
            }
        }

        buffer.append("] => ID = ");
        buffer.append(this.id);
        buffer.append(" ]");
        return buffer.toString();
    }
}
