/**
 * JASMINe
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ManagedConnectionFactoryImpl.java 7238 2010-12-01 15:57:47Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.adapter.jmx.pool.outbound;

import java.io.PrintWriter;
import java.io.Serializable;
import java.util.Iterator;
import java.util.Set;
import java.util.logging.Logger;

import javax.resource.ResourceException;
import javax.resource.spi.ConnectionManager;
import javax.resource.spi.ConnectionRequestInfo;
import javax.resource.spi.ManagedConnection;
import javax.resource.spi.ManagedConnectionFactory;
import javax.security.auth.Subject;

/**
 * Implementation of the ManagedConnectionFactory interface. This can provide a
 * new JmxCnxConnectionFactory.
 *
 * @author Guillaume Renault
 *
 */
public class ManagedConnectionFactoryImpl implements ManagedConnectionFactory, Serializable {

    /**
     * Unique generated Serial Version ID.
     */
    private static final long serialVersionUID = -7939114269986635829L;

    /**
     * Writer for the log.
     */
    private PrintWriter printWriter = null;

    /**
     * Logger static.
     */
    private static Logger logger = Logger.getLogger(ManagedConnectionFactoryImpl.class.getName());

    /**
     * Creates a JmxCnxConnectionFactory instance. The Connection Factory
     * instance gets initialized with a default ConnectionManager provided by
     * the resource adapter.
     *
     * @see javax.resource.spi.
     *      ManagedConnectionFactory#createConnectionFactory()
     * @return EIS-specific Connection Factory instance or
     *         javax.resource.cci.ConnectionFactory instance
     * @throws ResourceException Generic exception
     *             ResourceAdapterInternalException Resource adapter related
     *             error condition
     */
    public final Object createConnectionFactory() throws ResourceException {
        return new JMXConnectionFactoryImpl(this, null);
    }

    /**
     * Creates a JmxCnxConnectionFactory instance. The Connection Factory
     * instance gets initialized with the passed ConnectionManager. In the
     * managed scenario, ConnectionManager is provided by the application
     * server.
     *
     * @see javax.resource.spi.ManagedConnectionFactory#
     *      createConnectionFactory(ConnectionManager)
     * @param cxManager ConnectionManager to be associated with created EIS
     *            connection factory instance.
     * @return EIS-specific Connection Factory instance or
     *         javax.resource.cci.ConnectionFactory instance.
     * @throws ResourceException Generic exception
     *             ResourceAdapterInternalException Resource adapter related
     *             error condition
     */
    public final Object createConnectionFactory(final ConnectionManager cxManager) throws ResourceException {
        return new JMXConnectionFactoryImpl(this, cxManager);
    }

    /**
     * Creates a new physical connection to the underlying EIS resource manager.
     *
     * ManagedConnectionFactory uses the security information (passed as
     * Subject) and additional ConnectionRequestInfo (which is specific to
     * ResourceAdapter and opaque to application server) to create this new
     * connection.
     *
     * @see javax.resource.spi.ManagedConnectionFactory#
     *      createManagedConnection(Subject, ConnectionRequestInfo)
     * @param subject Caller's security information. arg1 Additional resource
     * @param cxRequestInfo Additional resource adapter specific connection
     *            request information
     * @return null.
     * @throws ResourceException generic exception
     */
    public final ManagedConnection createManagedConnection(final Subject subject, final ConnectionRequestInfo cxRequestInfo)
            throws ResourceException {

        ManagedConnectionImpl mci = new ManagedConnectionImpl(this);

        return mci;
    }

    /**
     * Get the log writer for this ManagedConnectionFactory instance.
     *
     * The log writer is a character output stream to which all logging and
     * tracing messages for this ManagedConnectionFactory instance will be
     * printed
     *
     * ApplicationServer manages the association of output stream with the
     * ManagedConnectionFactory. When a ManagedConnectionFactory object is
     * created the log writer is initially null, in other words, logging is
     * disabled.
     *
     * @see javax.resource.spi.ManagedConnectionFactory#getLogWriter()
     * @return PrintWriter
     * @throws ResourceException generic exception.
     */
    public final PrintWriter getLogWriter() throws ResourceException {
        return this.printWriter;
    }

    /**
     * Returns a matched connection from the candidate set of connections.
     *
     * ManagedConnectionFactory uses the security info (as in Subject) and
     * information provided through ConnectionRequestInfo and additional
     * Resource Adapter specific criteria to do matching. Note that criteria
     * used for matching is specific to a resource adapter and is not prescribed
     * by the Connector specification.
     *
     * This method returns a ManagedConnection instance that is the best match
     * for handling the connection allocation request.
     *
     * @see javax.resource.spi.ManagedConnectionFactory#
     *      matchManagedConnections(Set, Subject, ConnectionRequestInfo)
     * @param connectionSet Candidate connection set.
     * @param subject Caller's security information.
     * @param cxRequestInfo Additional resource adapter specific connection
     *            request information.
     * @return null
     * @throws ResourceException Generic exception
     */
    @SuppressWarnings("unchecked")
    public final ManagedConnection matchManagedConnections(final Set connectionSet, final Subject subject,
            final ConnectionRequestInfo cxRequestInfo) throws ResourceException {

        if (connectionSet == null) {
            return null;
        }

        Iterator<Object> it = connectionSet.iterator();
        Object obj = null;
        ManagedConnectionImpl mc = null;
        while (it.hasNext()) {
            try {
                obj = it.next();
                if (obj instanceof ManagedConnectionImpl) {
                    ManagedConnectionImpl mc1 = (ManagedConnectionImpl) obj;
                    if (this.equals(mc1.getLocalMCFI())) {
                        mc = mc1;
                        break;
                    }
                }
            } catch (Exception ex) {
                throw new ResourceException(ex.getMessage());
            }
        }

        return mc;
    }

    /**
     * Set the log writer for this ManagedConnectionFactory instance.
     *
     * The log writer is a character output stream to which all logging and
     * tracing messages for this ManagedConnectionfactory instance will be
     * printed.
     *
     * ApplicationServer manages the association of output stream with the
     * ManagedConnectionFactory. When a ManagedConnectionFactory object is
     * created the log writer is initially null, in other words, logging is
     * disabled. Once a log writer is associated with a
     * ManagedConnectionFactory, logging and tracing for
     * ManagedConnectionFactory instance is enabled.
     *
     * The ManagedConnection instances created by ManagedConnectionFactory
     * "inherits" the log writer, which can be overridden by ApplicationServer
     * using ManagedConnection.setLogWriter to set ManagedConnection specific
     * logging and tracing.
     *
     * @see javax.resource.spi.ManagedConnectionFactory#
     *      setLogWriter(PrintWriter)
     * @param out PrintWriter - an out stream for error logging and tracing.
     * @throws ResourceException Generic exception.
     */
    public final void setLogWriter(final PrintWriter out) throws ResourceException {
        this.printWriter = out;
    }

    /**
     * Get the printWriter.
     *
     * @return the printWriter
     */
    public PrintWriter getPrintWriter() {
        return this.printWriter;
    }

    /**
     * Get the logger.
     *
     * @return the logger
     */
    public static Logger getLogger() {
        return logger;
    }

}
