/**
 * JASMINe
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ManagedConnectionImpl.java 7238 2010-12-01 15:57:47Z jlegrand $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.adapter.jmx.pool.outbound;

import java.io.PrintWriter;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.resource.ResourceException;
import javax.resource.spi.ConnectionEvent;
import javax.resource.spi.ConnectionEventListener;
import javax.resource.spi.ConnectionRequestInfo;
import javax.resource.spi.LocalTransaction;
import javax.resource.spi.ManagedConnection;
import javax.resource.spi.ManagedConnectionMetaData;
import javax.security.auth.Subject;
import javax.transaction.xa.XAResource;

import org.ow2.jasmine.adapter.jmx.pool.api.JMXConnectionParam;

/**
 * Manager of a connection.
 *
 * @author Guillaume Renault
 *
 */
public class ManagedConnectionImpl implements ManagedConnection {

    /**
     * Set of listeners.
     */
    private Set<ConnectionEventListener> cels = null;

    /**
     * The connection managed by the manager.
     */
    private JMXConnectionImpl connection = null;

    /**
     * The manager of the connection factory.
     */
    private ManagedConnectionFactoryImpl localMCFI = null;

    /**
     * Logger.
     */
    private Logger logger = Logger.getLogger(ManagedConnectionImpl.class.getName());

    /**
     * create a new connection manager.
     *
     * @param mcfi the managed factory on which this manager will be associated
     */
    public ManagedConnectionImpl(final ManagedConnectionFactoryImpl mcfi) {
        this.localMCFI = mcfi;
        this.logger.log(Level.FINE, "New managed connection");
        this.cels = new HashSet<ConnectionEventListener>();
    }

    /**
     * Get the managed factory on which this manager is associated.
     *
     * @return the managed factory on which this manager is associated
     */
    public ManagedConnectionFactoryImpl getLocalMCFI() {
        return this.localMCFI;
    }

    /**
     * Called on event.
     *
     * @param code the code of the event.
     * @param ch the object to handle
     */
    public void signalEvent(final int code, final Object ch) {
        ConnectionEvent ce = null;
        ce = new ConnectionEvent(this, code);

        if (ch != null) {
            ce.setConnectionHandle(ch);
        }
        for (Iterator<ConnectionEventListener> it = this.cels.iterator(); it.hasNext();) {
            switch (code) {
            case ConnectionEvent.CONNECTION_CLOSED:
                (it.next()).connectionClosed(ce);
                break;
            case ConnectionEvent.LOCAL_TRANSACTION_STARTED:
                (it.next()).localTransactionStarted(ce);
                break;
            case ConnectionEvent.LOCAL_TRANSACTION_COMMITTED:
                (it.next()).localTransactionCommitted(ce);
                break;
            case ConnectionEvent.LOCAL_TRANSACTION_ROLLEDBACK:
                (it.next()).localTransactionRolledback(ce);
                break;
            case ConnectionEvent.CONNECTION_ERROR_OCCURRED:
                (it.next()).connectionErrorOccurred(ce);
                break;
            default:
                throw new IllegalArgumentException("Illegal eventType: " + code);
            }
        }
    }

    /**
     * Release the JMX connection.
     *
     * @param connectionJMX JMX connection to release
     */
    public void release(final JMXConnectionImpl connectionJMX) {
        if (connectionJMX == this.connection) {
            this.signalEvent(ConnectionEvent.CONNECTION_CLOSED, connectionJMX);
        }
    }

    /**
     * Close the JMX Connection.
     *
     * @param connectionJMX the JMX Connection to close
     */
    public void close(final JMXConnectionImpl connectionJMX) {
        try {
            connectionJMX.close();
        } catch (ResourceException e) {
            this.logger.log(Level.SEVERE, "Error occured while closing the JMX connection");
        }
        this.signalEvent(ConnectionEvent.CONNECTION_CLOSED, connectionJMX);
    }

    /**
     * Adds a connection event listener to the ManagedConnection instance. The
     * registered ConnectionEventListener instances are notified of connection
     * close and error events, also of local transaction related events on the
     * Managed Connection.
     *
     * @see javax.resource.spi.ManagedConnection#addConnectionEventListener(ConnectionEventListener)
     * @param listener a new ConnectionEventListener to be registered
     */
    public void addConnectionEventListener(final ConnectionEventListener listener) {
        this.logger.log(Level.FINE, "Add connection listener");
        this.cels.add(listener);
    }

    /**
     * Used by the container to change the association of an application-level
     * connection handle with a ManagedConneciton instance. The container should
     * find the right ManagedConnection instance and call the
     * associateConnection method.
     *
     * The resource adapter is required to implement the associateConnection
     * method. The method implementation for a ManagedConnection should
     * dissociate the connection handle (passed as a parameter) from its
     * currently associated ManagedConnection and associate the new connection
     * handle with itself.
     *
     * Not implemented.
     *
     * @see javax.resource.spi.ManagedConnection#associateConnection(Object)
     * @param connection Application-level connection handle
     *
     * @throws ResourceException generic exception if operation fails
     *
     */
    public void associateConnection(final Object connection) throws ResourceException {
        this.logger.log(Level.WARNING, "Not implemented.");
    }

    /**
     * Application server calls this method to force any cleanup on the
     * ManagedConnection instance.
     *
     * The method ManagedConnection.cleanup initiates a cleanup of the any
     * client-specific state as maintained by a ManagedConnection instance. The
     * cleanup should invalidate all connection handles that had been created
     * using this ManagedConnection instance. Any attempt by an application
     * component to use the connection handle after cleanup of the underlying
     * ManagedConnection should result in an exception.
     *
     * The cleanup of ManagedConnection is always driven by an application
     * server. An application server should not invoke ManagedConnection.cleanup
     * when there is an uncompleted transaction (associated with a
     * ManagedConnection instance) in progress.
     *
     * The invocation of ManagedConnection.cleanup method on an already
     * cleaned-up connection should not throw an exception.
     *
     * The cleanup of ManagedConnection instance resets its client specific
     * state and prepares the connection to be put back in to a connection pool.
     * The cleanup method should not cause resource adapter to close the
     * physical pipe and reclaim system resources associated with the physical
     * connection.
     *
     * @see javax.resource.spi.ManagedConnection#cleanup()
     *
     * @throws ResourceException generic exception if operation fails
     */
    public void cleanup() throws ResourceException {

        this.logger.log(Level.FINE, "cleanup called ...");
        if (this.connection.getState() == JMXConnectionImpl.States.CLOSED) {
            this.connection = null;
        }
    }

    /**
     * Destroys the physical connection to the underlying resource manager.
     *
     * To manage the size of the connection pool, an application server can
     * explictly call ManagedConnection.destroy to destroy a physical
     * connection. A resource adapter should destroy all allocated system
     * resources for this ManagedConnection instance when the method destroy is
     * called.
     *
     * @see javax.resource.spi.ManagedConnection#destroy()
     *
     * @throws ResourceException generic exception if operation failed
     */
    public void destroy() throws ResourceException {

        this.logger.log(Level.FINE, "destroy called ...");
        if (this.connection != null) {
            this.connection.close();
        }
    }

    /**
     * Creates a new connection handle for the underlying physical connection
     * represented by the ManagedConnection instance. This connection handle is
     * used by the application code to refer to the underlying physical
     * connection. This connection handle is associated with its
     * ManagedConnection instance in a resource adapter implementation specific
     * way.
     *
     * The ManagedConnection uses the Subject and additional ConnectionRequest
     * Info (which is specific to resource adapter and opaque to application
     * server) to set the state of the physical connection.
     *
     * @see javax.resource.spi.ManagedConnection#getConnection(Subject,
     *      ConnectionRequestInfo)
     *
     * @param subject security context as JAAS subject
     * @param cxRequestInfo ConnectionRequestInfo instance
     *
     * @return generic Object instance representing the connection handle. For
     *         CCI, the connection handle created by a ManagedConnection
     *         instance is of the type javax.resource.cci.Connection.
     *
     * @throws ResourceException - generic exception if operation fails
     *
     */
    public Object getConnection(final Subject subject, final ConnectionRequestInfo cxRequestInfo) throws ResourceException {

        JMXConnectionParam j = (JMXConnectionParam) cxRequestInfo;
        Logger.getLogger(ManagedConnectionImpl.class.getName()).log(
            Level.FINE,
            "\n" + "get connection ..." + "params : " + "\turl " + j.getUrlJMX() + "\tuser " + j.getUserJMX() + "\tpassword "
                    + j.getPasswordJMX());
        if (this.connection == null) {
            this.connection = new JMXConnectionImpl(this, j);
        }
        this.connection.setState(JMXConnectionImpl.States.GET);
        return this.connection;
    }

    /**
     * Not implemented.
     *
     * @see javax.resource.spi.ManagedConnection#getLocalTransaction()
     *
     * @return null
     *
     * @throws ResourceException generic exception if operation fails
     */
    public LocalTransaction getLocalTransaction() throws ResourceException {
        this.logger.log(Level.WARNING, "Not implemented.");
        return null;
    }

    /**
     * Not implemented.
     *
     * @see javax.resource.spi.ManagedConnection#getLogWriter()
     *
     * @return null
     *
     * @throws ResourceException generic exception if operation fails
     */
    public PrintWriter getLogWriter() throws ResourceException {
        this.logger.log(Level.WARNING, "Not implemented.");
        return null;
    }

    /**
     * Not implemented.
     *
     * @see javax.resource.spi.ManagedConnection#getMetaData()
     *
     * @return null
     *
     * @throws ResourceException generic exception if operation fails
     */
    public ManagedConnectionMetaData getMetaData() throws ResourceException {
        this.logger.log(Level.WARNING, "Not implemented.");
        return null;
    }

    /**
     * Not implemented.
     *
     * @see javax.resource.spi.ManagedConnection#getXAResource()
     *
     * @return null
     *
     * @throws ResourceException generic exception if operation fails
     */
    public XAResource getXAResource() throws ResourceException {
        this.logger.log(Level.WARNING, "Not implemented.");
        return null;
    }

    /**
     * Removes an already registered connection event listener from the
     * ManagedConnection instance.
     *
     * @see javax.resource.spi.ManagedConnection#removeConnectionEventListener(ConnectionEventListener)
     *
     * @param listener already registered connection event listener to be
     *            removed
     */
    public void removeConnectionEventListener(final ConnectionEventListener listener) {
        this.logger.log(Level.FINE, "Remove connection listener");
        this.cels.remove(listener);

    }

    /**
     * Not implemented.
     *
     * @see javax.resource.spi.ManagedConnection#setLogWriter(PrintWriter)
     *
     * @param out Character Output stream to be associated
     *
     * @throws ResourceException generic exception if operation fails
     *
     */
    public void setLogWriter(final PrintWriter out) throws ResourceException {
        this.logger.log(Level.WARNING, "Not implemented.");

    }

}
