/**
 * JASMINe
 * Copyright (C) 2005-2007 Bull S.A.S.
 * Contact: jasmine@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JmxCnxWrapper.java 38 2007-12-12 13:03:04Z tokmensa $
 * --------------------------------------------------------------------------
 */
package org.ow2.jasmine.monitoring.mbeancmd;

import java.io.IOException;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceAlreadyExistsException;
import javax.management.InstanceNotFoundException;
import javax.management.IntrospectionException;
import javax.management.InvalidAttributeValueException;
import javax.management.ListenerNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanInfo;
import javax.management.MBeanRegistrationException;
import javax.management.MBeanServerConnection;
import javax.management.NotCompliantMBeanException;
import javax.management.NotificationFilter;
import javax.management.NotificationListener;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.management.QueryExp;
import javax.management.ReflectionException;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXServiceURL;

/**
 * JMX connection wrapper.
 */
public class JmxCnxWrapper implements MBeanServerConnection {
    /**
     * Constructor: saves the JMX connection information.
     *
     * @param url JMX URL to use in the wrapper.
     * @param user User name.
     * @param password Password.
     */
    public JmxCnxWrapper(final String url, final String user, final String password) {
        this.url = url;
        this.user = user;
        this.password = password;
    }

    /**
     * Adds a listener to a registered MBean.
     *
     * A notification emitted by an MBean will be forwarded by the MBeanServer
     * to the listener. If the source of the notification is a reference to an
     * MBean object, the MBean server will replace it by that MBean's
     * ObjectName. Otherwise the source is unchanged.
     *
     * @see javax.management.MBeanServerConnection#addNotificationListener(javax.management.ObjectName,
     *      javax.management.NotificationListener,
     *      javax.management.NotificationFilter, java.lang.Object)
     * @param name The name of the MBean on which the listener should be added.
     * @param listener The listener object which will handle the notifications
     *            emitted by the registered MBean.
     * @param filter The filter object. If filter is null, no filtering will be
     *            performed before handling notifications.
     * @param handback The context to be sent to the listener when a
     *            notification is emitted.
     * @throws InstanceNotFoundException The MBean name provided does not match
     *             any of the registered MBeans.
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     */
    public void addNotificationListener(final ObjectName name, final NotificationListener listener,
            final NotificationFilter filter, final Object handback) throws InstanceNotFoundException, IOException {
        try {
            this.connect();
            this.mbscnx.addNotificationListener(name, listener, filter, handback);
        } catch (IOException e) {
            this.analyseException(e);
        }
    }

    /**
     * Adds a listener to a registered MBean.
     *
     * A notification emitted by an MBean will be forwarded by the MBeanServer
     * to the listener. If the source of the notification is a reference to an
     * MBean object, the MBean server will replace it by that MBean's
     * ObjectName. Otherwise the source is unchanged.
     *
     * The listener object that receives notifications is the one that is
     * registered with the given name at the time this method is called. Even if
     * it is subsequently unregistered, it will continue to receive
     * notifications.
     *
     * @see javax.management.MBeanServerConnection#addNotificationListener(javax.management.ObjectName,
     *      javax.management.ObjectName, javax.management.NotificationFilter,
     *      java.lang.Object)
     *
     * @param name The name of the MBean on which the listener should be added.
     * @param listener The object name of the listener which will handle the
     *            notifications emitted by the registered MBean.
     * @param filter The filter object. If filter is null, no filtering will be
     *            performed before handling notifications.
     * @param handback The context to be sent to the listener when a
     *            notification is emitted.
     * @throws InstanceNotFoundException The MBean name of the notification
     *             listener or of the notification broadcaster does not match
     *             any of the registered MBeans.
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     */
    public void addNotificationListener(final ObjectName name, final ObjectName listener, final NotificationFilter filter,
            final Object handback) throws InstanceNotFoundException, IOException {
        try {
            this.connect();
            this.mbscnx.addNotificationListener(name, listener, filter, handback);
        } catch (IOException e) {
            this.analyseException(e);
        }
    }

    /**
     * Instantiates and registers an MBean in the MBean server. The MBean server
     * will use its Default Loader Repository to load the class of the MBean. An
     * object name is associated to the MBean. If the object name given is null,
     * the MBean must provide its own name by implementing the MBeanRegistration
     * interface and returning the name from the preRegister method.
     *
     * This method is equivalent to createMBean(className, name, (Object[])
     * null, (String[]) null)
     *
     * @see javax.management.MBeanServerConnection#createMBean(java.lang.String,
     *      javax.management.ObjectName)
     * @param className The class name of the MBean to be instantiated.
     * @param name The object name of the MBean. May be null.
     * @throws ReflectionException Wraps a java.lang.ClassNotFoundException or a
     *             java.lang.Exception that occurred when trying to invoke the
     *             MBean's constructor.
     * @throws InstanceAlreadyExistsException The MBean is already under the
     *             control of the MBean server.
     * @throws MBeanRegistrationException The preRegister (MBeanRegistration
     *             interface) method of the MBean has thrown an exception. The
     *             MBean will not be registered.
     * @throws MBeanException The constructor of the MBean has thrown an
     *             exception
     * @throws NotCompliantMBeanException This class is not a JMX compliant
     *             MBean
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     * @return An ObjectInstance, containing the ObjectName and the Java class
     *         name of the newly instantiated MBean. If the contained ObjectName
     *         is n, the contained Java class name is
     *         getMBeanInfo(n).getClassName().
     */
    public ObjectInstance createMBean(final String className, final ObjectName name) throws ReflectionException,
            InstanceAlreadyExistsException, MBeanRegistrationException, MBeanException, NotCompliantMBeanException, IOException {
        ObjectInstance oi = null;
        try {
            this.connect();
            oi = this.mbscnx.createMBean(className, name);
        } catch (IOException e) {
            this.analyseException(e);
        }
        return oi;
    }

    /**
     * Instantiates and registers an MBean in the MBean server. The class loader
     * to be used is identified by its object name. An object name is associated
     * to the MBean. If the object name of the loader is null, the ClassLoader
     * that loaded the MBean server will be used. If the MBean's object name
     * given is null, the MBean must provide its own name by implementing the
     * MBeanRegistration interface and returning the name from the preRegister
     * method.
     *
     * This method is equivalent to createMBean(className, name, loaderName,
     * (Object[]) null, (String[]) null).
     *
     * @see javax.management.MBeanServerConnection#createMBean(java.lang.String,
     *      javax.management.ObjectName, javax.management.ObjectName)
     *
     * @param className The class name of the MBean to be instantiated.
     * @param name The object name of the MBean. May be null.
     * @param loaderName The object name of the class loader to be used.
     * @return An ObjectInstance, containing the ObjectName and the Java class
     *         name of the newly instantiated MBean. If the contained ObjectName
     *         is n, the contained Java class name is
     *         getMBeanInfo(n).getClassName().
     * @throws ReflectionException Wraps a java.lang.ClassNotFoundException or a
     *             java.lang.Exception that occurred when trying to invoke the
     *             MBean's constructor.
     * @throws InstanceAlreadyExistsException The MBean is already under the
     *             control of the MBean server.
     * @throws MBeanRegistrationException The preRegister (MBeanRegistration
     *             interface) method of the MBean has thrown an exception. The
     *             MBean will not be registered.
     * @throws MBeanException The constructor of the MBean has thrown an
     *             exception
     * @throws NotCompliantMBeanException This class is not a JMX compliant
     *             MBean
     * @throws InstanceNotFoundException The specified class loader is not
     *             registered in the MBean server.
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     */
    public ObjectInstance createMBean(final String className, final ObjectName name, final ObjectName loaderName)
            throws ReflectionException, InstanceAlreadyExistsException, MBeanRegistrationException, MBeanException,
            NotCompliantMBeanException, InstanceNotFoundException, IOException {
        ObjectInstance oi = null;
        try {
            this.connect();
            oi = this.mbscnx.createMBean(className, name, loaderName);
        } catch (IOException e) {
            this.analyseException(e);
        }
        return oi;
    }

    /**
     * Instantiates and registers an MBean in the MBean server. The MBean server
     * will use its Default Loader Repository to load the class of the MBean. An
     * object name is associated to the MBean. If the object name given is null,
     * the MBean must provide its own name by implementing the MBeanRegistration
     * interface and returning the name from the preRegister method.
     *
     * @see javax.management.MBeanServerConnection#createMBean(java.lang.String,
     *      javax.management.ObjectName, java.lang.Object[], java.lang.String[])
     *
     * @param className The class name of the MBean to be instantiated.
     * @param name The object name of the MBean. May be null.
     * @param params An array containing the parameters of the constructor to be
     *            invoked.
     * @param signature An array containing the signature of the constructor to
     *            be invoked.
     *
     * @return An ObjectInstance, containing the ObjectName and the Java class
     *         name of the newly instantiated MBean. If the contained ObjectName
     *         is n, the contained Java class name is
     *         getMBeanInfo(n).getClassName().
     *
     * @throws ReflectionException Wraps a java.lang.ClassNotFoundException or a
     *             java.lang.Exception that occurred when trying to invoke the
     *             MBean's constructor.
     * @throws InstanceAlreadyExistsException The MBean is already under the
     *             control of the MBean server.
     * @throws MBeanRegistrationException The preRegister (MBeanRegistration
     *             interface) method of the MBean has thrown an exception. The
     *             MBean will not be registered.
     * @throws MBeanException The constructor of the MBean has thrown an
     *             exception
     * @throws NotCompliantMBeanException This class is not a JMX compliant
     *             MBean
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     *
     */
    public ObjectInstance createMBean(final String className, final ObjectName name, final Object[] params,
            final String[] signature) throws ReflectionException, InstanceAlreadyExistsException, MBeanRegistrationException,
            MBeanException, NotCompliantMBeanException, IOException {
        ObjectInstance oi = null;
        try {
            this.connect();
            oi = this.mbscnx.createMBean(className, name, params, signature);
        } catch (IOException e) {
            this.analyseException(e);
        }
        return oi;
    }

    /**
     * Instantiates and registers an MBean in the MBean server. The class loader
     * to be used is identified by its object name. An object name is associated
     * to the MBean. If the object name of the loader is not specified, the
     * ClassLoader that loaded the MBean server will be used. If the MBean
     * object name given is null, the MBean must provide its own name by
     * implementing the MBeanRegistration interface and returning the name from
     * the preRegister method.
     *
     * @see javax.management.MBeanServerConnection#createMBean(java.lang.String,
     *      javax.management.ObjectName, javax.management.ObjectName,
     *      java.lang.Object[], java.lang.String[])
     *
     * @param className The class name of the MBean to be instantiated.
     * @param name The object name of the MBean. May be null.
     * @param params An array containing the parameters of the constructor to be
     *            invoked.
     * @param signature An array containing the signature of the constructor to
     *            be invoked.
     * @param loaderName The object name of the class loader to be used.
     *
     * @return An ObjectInstance, containing the ObjectName and the Java class
     *         name of the newly instantiated MBean. If the contained ObjectName
     *         is n, the contained Java class name is
     *         getMBeanInfo(n).getClassName().
     *
     * @throws ReflectionException Wraps a java.lang.ClassNotFoundException or a
     *             java.lang.Exception that occurred when trying to invoke the
     *             MBean's constructor.
     * @throws InstanceAlreadyExistsException The MBean is already under the
     *             control of the MBean server.
     * @throws MBeanRegistrationException The preRegister (MBeanRegistration
     *             interface) method of the MBean has thrown an exception. The
     *             MBean will not be registered.
     * @throws MBeanException The constructor of the MBean has thrown an
     *             exception
     * @throws NotCompliantMBeanException This class is not a JMX compliant
     *             MBean
     * @throws InstanceNotFoundException The specified class loader is not
     *             registered in the MBean server.
     *
     * @throws IOException - A communication problem occurred when talking to
     *             the MBean server.
     *
     */
    public ObjectInstance createMBean(final String className, final ObjectName name, final ObjectName loaderName,
            final Object[] params, final String[] signature) throws ReflectionException, InstanceAlreadyExistsException,
            MBeanRegistrationException, MBeanException, NotCompliantMBeanException, InstanceNotFoundException, IOException {
        ObjectInstance oi = null;
        try {
            this.connect();
            oi = this.mbscnx.createMBean(className, name, loaderName, params, signature);
        } catch (IOException e) {
            this.analyseException(e);
        }
        return oi;
    }

    /**
     * Gets the value of a specific attribute of a named MBean. The MBean is
     * identified by its object name.
     *
     * @see javax.management.MBeanServerConnection#getAttribute(javax.management.ObjectName,
     *      java.lang.String)
     *
     * @param name The object name of the MBean from which the attribute is to
     *            be retrieved.
     * @param attribute A String specifying the name of the attribute to be
     *            retrieved.
     * @return The value of the retrieved attribute.
     * @throws AttributeNotFoundException The attribute specified is not
     *             accessible in the MBean.
     * @throws MBeanException Wraps an exception thrown by the MBean's getter.
     * @throws InstanceNotFoundException The MBean specified is not registered
     *             in the MBean server.
     * @throws ReflectionException Wraps a java.lang.Exception thrown when
     *             trying to invoke the setter.
     *
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     */
    public Object getAttribute(final ObjectName name, final String attribute) throws MBeanException, AttributeNotFoundException,
            InstanceNotFoundException, ReflectionException, IOException {
        Object att = null;
        try {
            this.connect();
            att = this.mbscnx.getAttribute(name, attribute);
        } catch (IOException e) {
            this.analyseException(e);
        }
        return att;
    }

    /**
     * Enables the values of several attributes of a named MBean. The MBean is
     * identified by its object name.
     *
     * @see javax.management.MBeanServerConnection#getAttributes(javax.management.ObjectName,
     *      java.lang.String[])
     *
     * @param name The object name of the MBean from which the attributes are
     *            retrieved.
     * @param attributes A list of the attributes to be retrieved.
     * @return The list of the retrieved attributes.
     * @throws InstanceNotFoundException The MBean specified is not registered
     *             in the MBean server.
     * @throws ReflectionException An exception occurred when trying to invoke
     *             the getAttributes method of a Dynamic MBean.
     *
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     */
    public AttributeList getAttributes(final ObjectName name, final String[] attributes) throws InstanceNotFoundException,
            ReflectionException, IOException {
        AttributeList attl = new AttributeList();
        try {
            this.connect();
            attl = this.mbscnx.getAttributes(name, attributes);
        } catch (IOException e) {
            this.analyseException(e);
        }
        return attl;
    }

    /**
     * Returns the default domain used for naming the MBean. The default domain
     * name is used as the domain part in the ObjectName of MBeans if no domain
     * is specified by the user.
     *
     * @see javax.management.MBeanServerConnection#getDefaultDomain()
     *
     * @return The default domain.
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     */
    public String getDefaultDomain() throws IOException {
        String domain = null;
        try {
            this.connect();
            domain = this.mbscnx.getDefaultDomain();
        } catch (IOException e) {
            this.analyseException(e);
        }
        return domain;
    }

    /**
     * Returns the list of domains in which any MBean is currently registered. A
     * string is in the returned array if and only if there is at least one
     * MBean registered with an ObjectName whose getDomain() is equal to that
     * string. The order of strings within the returned array is not defined.
     *
     * @see javax.management.MBeanServerConnection#getDomains()
     *
     * @return the list of domains.
     * @throws IOException - A communication problem occurred when talking to
     *             the MBean server.
     */
    public String[] getDomains() throws IOException {
        String[] domains = null;
        try {
            this.connect();
            domains = this.mbscnx.getDomains();
        } catch (IOException e) {
            this.analyseException(e);
        }
        return domains;
    }

    /**
     * Returns the number of MBeans registered in the MBean server.
     *
     * @see javax.management.MBeanServerConnection#getMBeanCount()
     *
     * @return the number of MBeans registered.
     *
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     */
    public Integer getMBeanCount() throws IOException {
        Integer count = null;
        try {
            this.connect();
            count = this.mbscnx.getMBeanCount();
        } catch (IOException e) {
            this.analyseException(e);
        }
        return count;
    }

    /**
     * This method discovers the attributes and operations that an MBean exposes
     * for management.
     *
     * @see javax.management.MBeanServerConnection#getMBeanInfo(javax.management.ObjectName)
     *
     * @param name The name of the MBean to analyze
     * @return An instance of MBeanInfo allowing the retrieval of all attributes
     *         and operations of this MBean.
     * @throws IntrospectionException An exception occurred during
     *             introspection.
     * @throws InstanceNotFoundException The MBean specified was not found.
     * @throws ReflectionException An exception occurred when trying to invoke
     *             the getMBeanInfo of a Dynamic MBean.
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     */
    public MBeanInfo getMBeanInfo(final ObjectName name) throws InstanceNotFoundException, IntrospectionException,
            ReflectionException, IOException {
        MBeanInfo mbi = null;
        try {
            this.connect();
            mbi = this.mbscnx.getMBeanInfo(name);
        } catch (IOException e) {
            this.analyseException(e);
        }
        return mbi;
    }

    /**
     * Gets the ObjectInstance for a given MBean registered with the MBean
     * server.
     *
     * @see javax.management.MBeanServerConnection#getObjectInstance(javax.management.ObjectName)
     *
     * @param name - The object name of the MBean.
     * @return The ObjectInstance associated with the MBean specified by name.
     *         The contained ObjectName is name and the contained class name is
     *         getMBeanInfo(name).getClassName().
     * @throws InstanceNotFoundException - The MBean specified is not registered
     *             in the MBean server.
     * @throws IOException - A communication problem occurred when talking to
     *             the MBean server.
     */
    public ObjectInstance getObjectInstance(final ObjectName name) throws InstanceNotFoundException, IOException {
        ObjectInstance oi = null;
        try {
            this.connect();
            oi = this.mbscnx.getObjectInstance(name);
        } catch (IOException e) {
            this.analyseException(e);
        }
        return oi;
    }

    /**
     * Invokes an operation on an MBean.
     *
     * @see javax.management.MBeanServerConnection#invoke(javax.management.ObjectName,
     *      java.lang.String, java.lang.Object[], java.lang.String[])
     *
     * @param name The object name of the MBean on which the method is to be
     *            invoked.
     * @param operationName The name of the operation to be invoked.
     * @param params An array containing the parameters to be set when the
     *            operation is invoked
     * @param signature An array containing the signature of the operation. The
     *            class objects will be loaded using the same class loader as
     *            the one used for loading the MBean on which the operation was
     *            invoked.
     * @return The object returned by the operation, which represents the result
     *         of invoking the operation on the MBean specified.
     * @throws InstanceNotFoundException The MBean specified is not registered
     *             in the MBean server.
     * @throws MBeanException Wraps an exception thrown by the MBean's invoked
     *             method.
     * @throws ReflectionException Wraps a java.lang.Exception thrown while
     *             trying to invoke the method.
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     */
    public Object invoke(final ObjectName name, final String operationName, final Object[] params, final String[] signature)
            throws InstanceNotFoundException, MBeanException, ReflectionException, IOException {
        Object o = null;
        try {
            this.connect();
            o = this.mbscnx.invoke(name, operationName, params, signature);
        } catch (IOException e) {
            this.analyseException(e);
        }
        return o;
    }

    /**
     * Returns true if the MBean specified is an instance of the specified
     * class, false otherwise.
     *
     * If name does not name an MBean, this method throws
     * InstanceNotFoundException.
     *
     * Otherwise, let X be the MBean named by name, L be the ClassLoader of X, N
     * be the class name in X's MBeanInfo.
     *
     * If N equals className, the result is true.
     *
     * Otherwise, if L successfully loads both N and className, and the second
     * class is assignable from the first, the result is true.
     *
     * Otherwise, the result is false.
     *
     * @see javax.management.MBeanServerConnection#isInstanceOf(javax.management.ObjectName,
     *      java.lang.String)
     *
     * @param name The ObjectName of the MBean.
     * @param className The name of the class.
     *
     * @return true if the MBean specified is an instance of the specified class
     *         according to the rules above, false otherwise.
     * @throws InstanceNotFoundException The MBean specified is not registered
     *             in the MBean server.
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     */
    public boolean isInstanceOf(final ObjectName name, final String className) throws InstanceNotFoundException, IOException {
        boolean b = false;
        try {
            this.connect();
            b = this.mbscnx.isInstanceOf(name, className);
        } catch (IOException e) {
            this.analyseException(e);
        }
        return b;
    }

    /**
     * Checks whether an MBean, identified by its object name, is already
     * registered with the MBean server.
     *
     * @see javax.management.MBeanServerConnection#isRegistered(javax.management.ObjectName)
     *
     * @param name The object name of the MBean to be checked.
     * @return True if the MBean is already registered in the MBean server,
     *         false otherwise.
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     */
    public boolean isRegistered(final ObjectName name) throws IOException {
        boolean b = false;
        try {
            this.connect();
            b = this.mbscnx.isRegistered(name);
        } catch (IOException e) {
            this.analyseException(e);
        }
        return b;
    }

    /**
     * Gets MBeans controlled by the MBean server. This method allows any of the
     * following to be obtained: All MBeans, a set of MBeans specified by
     * pattern matching on the ObjectName and/or a Query expression, a specific
     * MBean. When the object name is null or no domain and key properties are
     * specified, all objects are to be selected (and filtered if a query is
     * specified). It returns the set of ObjectInstance objects (containing the
     * ObjectName and the Java Class name) for the selected MBeans.
     *
     * @see javax.management.MBeanServerConnection#queryMBeans(javax.management.ObjectName,
     *      javax.management.QueryExp)
     *
     * @param name The object name pattern identifying the MBeans to be
     *            retrieved. If null or no domain and key properties are
     *            specified, all the MBeans registered will be retrieved.
     * @param query The query expression to be applied for selecting MBeans. If
     *            null no query expression will be applied for selecting MBeans.
     * @return A set containing the ObjectInstance objects for the selected
     *         MBeans. If no MBean satisfies the query an empty list is
     *         returned.
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     */
    @SuppressWarnings("unchecked")
    public Set<ObjectInstance> queryMBeans(final ObjectName name, final QueryExp query) throws IOException {
        Set<ObjectInstance> ons = null;
        try {
            this.connect();
            ons = this.mbscnx.queryMBeans(name, query);
        } catch (IOException e) {
            this.analyseException(e);
        }
        if (ons == null) {
            ons = new LinkedHashSet<ObjectInstance>();
        }
        return ons;
    }

    /**
     * Gets the names of MBeans controlled by the MBean server. This method
     * enables any of the following to be obtained: The names of all MBeans, the
     * names of a set of MBeans specified by pattern matching on the ObjectName
     * and/or a Query expression, a specific MBean name (equivalent to testing
     * whether an MBean is registered). When the object name is null or no
     * domain and key properties are specified, all objects are selected (and
     * filtered if a query is specified). It returns the set of ObjectNames for
     * the MBeans selected.
     *
     * @see javax.management.MBeanServerConnection#queryNames(javax.management.ObjectName,
     *      javax.management.QueryExp)
     *
     * @param name The object name pattern identifying the MBean names to be
     *            retrieved. If null or no domain and key properties are
     *            specified, the name of all registered MBeans will be
     *            retrieved.
     * @param query The query expression to be applied for selecting MBeans. If
     *            null no query expression will be applied for selecting MBeans.
     * @return A set containing the ObjectNames for the MBeans selected. If no
     *         MBean satisfies the query, an empty list is returned.
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     */
    @SuppressWarnings("unchecked")
    public Set<ObjectName> queryNames(final ObjectName name, final QueryExp query) throws IOException {
        Set<ObjectName> ons = null;
        try {
            this.connect();
            ons = this.mbscnx.queryNames(name, query);
        } catch (IOException e) {
            this.analyseException(e);
        }
        if (ons == null) {
            ons = new LinkedHashSet<ObjectName>();
        }
        return ons;
    }

    /**
     * Removes a listener from a registered MBean.
     *
     * If the listener is registered more than once, perhaps with different
     * filters or callbacks, this method will remove all those registrations.
     *
     * @see javax.management.MBeanServerConnection#removeNotificationListener(javax.management.ObjectName,
     *      javax.management.NotificationListener)
     *
     * @param name The name of the MBean on which the listener should be
     *            removed.
     * @param listener The object name of the listener to be removed.
     *
     * @throws InstanceNotFoundException The MBean name provided does not match
     *             any of the registered MBeans.
     * @throws ListenerNotFoundException The listener is not registered in the
     *             MBean.
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     */
    public void removeNotificationListener(final ObjectName name, final NotificationListener listener)
            throws InstanceNotFoundException, ListenerNotFoundException, IOException {
        try {
            this.connect();
            this.mbscnx.removeNotificationListener(name, listener);
        } catch (IOException e) {
            this.analyseException(e);
        }
    }

    /**
     * Removes a listener from a registered MBean.
     *
     * If the listener is registered more than once, perhaps with different
     * filters or callbacks, this method will remove all those registrations.
     *
     * @see javax.management.MBeanServerConnection#removeNotificationListener(javax.management.ObjectName,
     *      javax.management.ObjectName)
     *
     * @param name The name of the MBean on which the listener should be
     *            removed.
     * @param listener The object name of the listener to be removed.
     *
     * @throws InstanceNotFoundException The MBean name provided does not match
     *             any of the registered MBeans.
     * @throws ListenerNotFoundException The listener is not registered in the
     *             MBean.
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     */
    public void removeNotificationListener(final ObjectName name, final ObjectName listener) throws InstanceNotFoundException,
            ListenerNotFoundException, IOException {
        try {
            this.connect();
            this.mbscnx.removeNotificationListener(name, listener);
        } catch (IOException e) {
            this.analyseException(e);
        }
    }

    /**
     * Removes a listener from a registered MBean.
     *
     * The MBean must have a listener that exactly matches the given listener,
     * filter, and handback parameters. If there is more than one such listener,
     * only one is removed.
     *
     * The filter and handback parameters may be null if and only if they are
     * null in a listener to be removed.
     *
     * @see javax.management.MBeanServerConnection#removeNotificationListener(javax.management.ObjectName,
     *      javax.management.NotificationListener,
     *      javax.management.NotificationFilter, java.lang.Object)
     *
     * @param name The name of the MBean on which the listener should be
     *            removed.
     * @param listener A listener that was previously added to this MBean.
     * @param filter The filter that was specified when the listener was added.
     * @param handback The handback that was specified when the listener was
     *            added.
     *
     * @throws InstanceNotFoundException The MBean name provided does not match
     *             any of the registered MBeans.
     * @throws ListenerNotFoundException The listener is not registered in the
     *             MBean, or it is not registered with the given filter and
     *             handback.
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     */
    public void removeNotificationListener(final ObjectName name, final NotificationListener listener,
            final NotificationFilter filter, final Object handback) throws InstanceNotFoundException, ListenerNotFoundException,
            IOException {
        try {
            this.connect();
            this.mbscnx.removeNotificationListener(name, listener, filter, handback);
        } catch (IOException e) {
            this.analyseException(e);
        }
    }

    /**
     * Removes a listener from a registered MBean.
     *
     * The MBean must have a listener that exactly matches the given listener,
     * filter, and handback parameters. If there is more than one such listener,
     * only one is removed.
     *
     * The filter and handback parameters may be null if and only if they are
     * null in a listener to be removed.
     *
     * @see javax.management.MBeanServerConnection#removeNotificationListener(javax.management.ObjectName,
     *      javax.management.ObjectName, javax.management.NotificationFilter,
     *      java.lang.Object)
     *
     * @param name The name of the MBean on which the listener should be
     *            removed.
     * @param listener A listener that was previously added to this MBean.
     * @param filter The filter that was specified when the listener was added.
     * @param handback The handback that was specified when the listener was
     *            added.
     * @throws InstanceNotFoundException The MBean name provided does not match
     *             any of the registered MBeans.
     * @throws ListenerNotFoundException The listener is not registered in the
     *             MBean, or it is not registered with the given filter and
     *             handback.
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     *
     */
    public void removeNotificationListener(final ObjectName name, final ObjectName listener, final NotificationFilter filter,
            final Object handback) throws InstanceNotFoundException, ListenerNotFoundException, IOException {
        try {
            this.connect();
            this.mbscnx.removeNotificationListener(name, listener, filter, handback);
        } catch (IOException e) {
            this.analyseException(e);
        }
    }

    /**
     * Sets the value of a specific attribute of a named MBean. The MBean is
     * identified by its object name.
     *
     * @see javax.management.MBeanServerConnection#setAttribute(javax.management.ObjectName,
     *      javax.management.Attribute)
     * @param name The name of the MBean within which the attribute is to be
     *            set.
     * @param attribute The identification of the attribute to be set and the
     *            value it is to be set to.
     * @throws InstanceNotFoundException The MBean specified is not registered
     *             in the MBean server.
     * @throws AttributeNotFoundException The attribute specified is not
     *             accessible in the MBean.
     * @throws InvalidAttributeValueException The value specified for the
     *             attribute is not valid.
     * @throws MBeanException Wraps an exception thrown by the MBean's setter.
     * @throws ReflectionException Wraps a java.lang.Exception thrown when
     *             trying to invoke the setter.
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     */
    public void setAttribute(final ObjectName name, final Attribute attribute) throws InstanceNotFoundException,
            AttributeNotFoundException, InvalidAttributeValueException, MBeanException, ReflectionException, IOException {
        try {
            this.connect();
            this.mbscnx.setAttribute(name, attribute);
        } catch (IOException e) {
            this.analyseException(e);
        }
    }

    /**
     * Sets the values of several attributes of a named MBean. The MBean is
     * identified by its object name.
     *
     * @see javax.management.MBeanServerConnection#setAttributes(javax.management.ObjectName,
     *      javax.management.AttributeList)
     *
     * @param name The object name of the MBean within which the attributes are
     *            to be set.
     * @param attributes A list of attributes: The identification of the
     *            attributes to be set and the values they are to be set to.
     *
     * @return The list of attributes that were set, with their new values.
     *
     * @throws InstanceNotFoundException The MBean specified is not registered
     *             in the MBean server.
     * @throws ReflectionException An exception occurred when trying to invoke
     *             the getAttributes method of a Dynamic MBean.
     *
     * @throws IOException A communication problem occurred when talking to the
     *             MBean server.
     */
    public AttributeList setAttributes(final ObjectName name, final AttributeList attributes) throws InstanceNotFoundException,
            ReflectionException, IOException {

        AttributeList attl = new AttributeList();
        try {
            this.connect();
            attl = this.mbscnx.setAttributes(name, attributes);
        } catch (IOException e) {
            this.analyseException(e);
        }
        return attl;
    }

    /**
     * Unregisters an MBean from the MBean server. The MBean is identified by
     * its object name. Once the method has been invoked, the MBean may no
     * longer be accessed by its object name.
     *
     * @see javax.management.MBeanServerConnection#unregisterMBean(javax.management.ObjectName)
     *
     * @param name The object name of the MBean to be unregistered.
     *
     * @throws InstanceNotFoundException - The MBean specified is not registered
     *             in the MBean server.
     * @throws MBeanRegistrationException - The preDeregister
     *             ((MBeanRegistration interface) method of the MBean has thrown
     *             an exception.
     *
     * @throws IOException - A communication problem occurred when talking to
     *             the MBean server.
     */
    public void unregisterMBean(final ObjectName name) throws InstanceNotFoundException, MBeanRegistrationException, IOException {
        try {
            this.connect();
            this.mbscnx.unregisterMBean(name);
        } catch (IOException e) {
            this.analyseException(e);
        }
    }

    /**
     * @param cnx setMBeanServerConnection to use in the JMX connector.
     */
    public void setMBeanServerConnection(final MBeanServerConnection cnx) {
        this.mbscnx = cnx;
    }

    /**
     * Connects to the JMX URL.
     *
     * @throws IOException If connection failed.
     */
    public void connect() throws IOException {
        if (this.mbscnx == null) {
            Map<String, String[]> env = null;
            if (this.user != null && this.password != null) {
                env = new HashMap<String, String[]>(1);
                String[] creds = {this.user, this.password};
                env.put(JMXConnector.CREDENTIALS, creds);
            }
            System.out.println(this.url);
            JMXConnector c = JMXConnectorFactory.connect(new JMXServiceURL(this.url), env);
            this.mbscnx = c.getMBeanServerConnection();
            this.jmxConnector = c;
        }
    }

    /**
     * Tests the JmxCnxWrapper implementation.
     *
     * @param args URL to use as first argument, user name to use as second and
     *            password to use as third. All arguments are optional.
     */
    public static void main(final String[] args) {

        // bad name
        // String url = "service:jmx:rmi:///jndi/rmi:///jrmpconnector_jonas";

        // wrong port or host
        // String url =
        // "service:jmx:rmi:///jndi/rmi://localhost:2345/jrmpconnector_jonas";

        // unresoved host
        // String url =
        // "service:jmx:rmi:///jndi/rmi://total:2345/jrmpconnector_jonas";

        // right addres
        String url = "service:jmx:rmi:///jndi/rmi://localhost:1099/jrmpconnector_jonas";
        String user = null;
        String password = null;

        if (args.length > 0) {
            url = args[0];
        }
        if (args.length > 1) {
            user = args[1];
        }
        if (args.length > 2) {
            password = args[2];
        }
        JmxCnxWrapper cnx = new JmxCnxWrapper(url, user, password);

        for (int i = 1; i <= 100; i++) {
            try {
                Integer cnt = cnx.getMBeanCount();
                System.out.println("COUNT= " + cnt);
            } catch (IOException e) {
                System.err.println("FWA : " + e.getMessage());
            }
            try {
                Thread.sleep(5000);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }

    }

    /**
     * Analyses e, closes JMX if necessary and and throws e back.
     *
     * @param e Exception to analyse.
     *
     * @throws IOException e.
     */
    private void analyseException(final IOException e) throws IOException {
        boolean toClose = false;
        if (e instanceof java.rmi.ConnectException) {
            toClose = true;
        } else if (e instanceof java.rmi.ConnectIOException) {
            toClose = true;
        } else if (e instanceof java.rmi.UnknownHostException) {
            toClose = true;

            /*
             * else if (e instanceof java.rmi.RemoteException) { // So what ? }
             */

        } else {
            String s = e.getMessage();
            if (s.startsWith("javax.naming.NameNotFoundException")) {
                System.err.println("Persistent error : " + e.getMessage());
            } else if (s.startsWith("javax.naming.ServiceUnavailableException")) {
                // ex: java.rmi.ConnectException: Connection refused to host
                toClose = true;
            } else if (s.startsWith("javax.naming.ConfigurationException")) {
                // ex: java.rmi.UnknownHostException
                toClose = true;
            }
        }
        if (toClose) {
            this.closeJMX();
        }
        throw e;
    }

    /**
     * Closes the JMX connection.
     */
    private void closeJMX() {
        if (this.jmxConnector != null) {
            try {
                this.jmxConnector.close();
            } catch (IOException e) {
                System.err.println("FWA closing connector");
                // e.printStackTrace(System.err);
            }
        }
        this.mbscnx = null;
    }

    /**
     * Connection to the MBean server.
     */
    private MBeanServerConnection mbscnx = null;

    /**
     * JMX connector.
     */
    private JMXConnector jmxConnector = null;

    /**
     * JMX URL to connect to.
     */
    private String url = null;

    /**
     * User name to use when connecting to JMX.
     */
    private String user = null;

    /**
     * Password to use when connecting to JMX.
     */
    private String password = null;
}
